<?php

namespace App\Http\Livewire\Development\Samples;

use App\Models\Styles;
use App\Models\Samples;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Title;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

#[Title('Samples')]
class SamplesList extends FilterableComponent
{
	use WithPagination;
	use Filterable;

	
	#[Validate([
		'season' => ['nullable', 'exists:seasons,id'], // Assuming you have a 'seasons' table
		'customer' => ['nullable', 'exists:customers,id'],
		'factory' => ['nullable', 'exists:suppliers,id'],
		'category' => ['nullable', 'in:ladies,mens,accessories,childrens'], // Assuming you have a 'categories' table
		'department' => ['nullable', 'exists:departments,id'], // Assuming you have a 'departments' table
		'search' => ['nullable', 'string', 'max:30'],
		'status' => ['nullable', 'in:pending,approved,rejected'],  // Adjust validation rules for 'status' as needed
		'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
	])]
	public $paginate = 50, $season, $customer, $factory, $category, $department, $search, $status, $edit = [], $hideFilters;

	public $columns = [
		'Style' => TRUE,
		'Version' => TRUE,
		'Colourway' => TRUE,
		'Type' => TRUE,
		'Expected' => TRUE,
		'Sent' => TRUE,
		'Tracking' => TRUE,
		'Weight' => TRUE,
		'Size' => TRUE,
		'Qty' => TRUE,
		'Status' => TRUE,
		'Comments' => TRUE,
		'Dont Charge' => TRUE,
		'Dont Charge Dev' => TRUE,
		'Notes for Finance' => TRUE,
		'PO' => TRUE,
	];

	protected function filterKey(): string
    {
        return 'samplelistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['season', 'customer', 'factory', 'category', 'department', 'search', 'status', 'columns', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }
	
    #[On('refresh-columns')]
    public function refreshDesigns()
    {
        $this->loadFilterSettings();
    }

	public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['season', 'customer', 'factory', 'category', 'department', 'search', 'status', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

	public $styleEditID = 0;

	#[Computed]
	public function seasons(){
		return Seasons::allCached();
	}
	#[Computed]
	public function customers(){
		return Customer::whereHas('styles')->get();
	}
	#[Computed]
	public function factories(){
		return Suppliers::whereHas('style_versions')->get();
	}
	#[Computed]
	public function departments(){
		return Departments::allCached();
	}

	#[On('close-modal')]
	public function closeStyleEdit(){
		$this->styleEditID = 0;
		$this->render();
		$this->paginate = 50;
        $this->reset('edit');
	}

    #[On('refreshData')]
    public function refreshData()
    {
        // Refresh to show updated sample data
        $this->resetPage();
    }

    public function render()
    {
		Gate::authorize('style:read');
		$samples = Samples::select(
			'styles.customer_ref',
			'colourways.name as colourway',
			'colourways.id as colourway_id',
			'colourways.image',
			'samples.id',
			'sample_types.name as sample_type',
			'styles.id as style_id',
			'style_versions.name as style_version',
			'style_versions.id as style_version_id',
			'samples.date_sent',
			'samples.tracking',
			'samples.weight',
			'sizes.name as size',
			'samples.date_expected',
			'samples.status',
			'samples.do_not_charge',
			'samples.comments',
			'samples.qty',
			'samples.do_not_charge_dev',
			'samples.fn_notes',
			'samples.po')
        ->selectRaw('seasons.locked as season_locked')
		->selectRaw('(SELECT COUNT(s.id) FROM samples s JOIN style_versions sv ON sv.id = colourways.style_versions_id WHERE sv.styles_id=styles.id)  AS count_styles')
		->join('sample_types', 'sample_types.id', '=', 'samples.sample_types_id')
		->join('colourways', 'colourways.id', '=', 'samples.colourways_id')
		->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
		->join('styles', 'styles.id', '=', 'style_versions.styles_id')
		->join('seasons', 'seasons.id', '=', 'styles.seasons_id')
		->join('sizes', 'sizes.id', '=', 'samples.sizes_id')
		->search($this->search)
        ->whereIn('styles.departments_id', Auth::user()->user_departments->pluck('departments_id')->toArray())
		->when(!empty($this->season),function($query){
			$query->whereRelation('colourways.style_versions.styles', 'seasons_id', $this->season);
		})
		->when(!empty($this->customer),function($query){
			$query->whereRelation('colourways.style_versions.styles', 'customers_id', $this->customer);
		})
		->when(!empty($this->factory),function($query){
			$query->whereRelation('colourways.style_versions', 'factory_id', $this->factory);
		})
		->when(!empty($this->category),function($query){
			$query->whereRelation('colourways.style_versions.styles', 'category', $this->category);
		})
		->when(!empty($this->department),function($query){
			$query->whereRelation('colourways.style_versions.styles', 'departments_id', $this->department);
		})
		->when(!empty($this->status),function($query){
			$query->where('status', $this->status);
		})
		->orderBy('style_id')->orderBy('style_version_id')->orderBy('colourway_id')->orderBy('sample_type')
		->paginate($this->paginate);

        // dd(array_combine($samples->pluck('id')->toArray(), $samples->pluck('id')->toArray()));

        return view('livewire.development.samples.samples-list', ['samples' => $samples]);
    }

    public function selectAll($ids){
        $this->edit = array_fill_keys(array_keys(array_flip($ids)), TRUE);
    }

	public function editSamples(){
		$this->dispatch('samples-edit', samples: $this->edit);
	}
}
