<?php

namespace App\Http\Livewire\Development\Samples;

use App\Models\Sizes;
use App\Models\Styles;
use App\Models\Samples;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Colourways;
use App\Models\SampleTypes;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;

class BulkAdd extends BaseComponent
{
    public $season, $customer, $style, $styleVersion, $colourway, $samples = [];

    // Validation rules
    #[Validate([
        'season' => ['nullable', 'exists:seasons,id'],
        'customer' => ['nullable', 'exists:customers,id'],
        'style' => ['nullable', 'exists:styles,id'],
        'styleVersion' => ['nullable', 'exists:style_versions,id'],
        'colourway' => ['nullable', 'exists:colourways,id'],
        'samples' => ['array'],
        'samples.*.sample_types.name' => ['nullable', 'string', 'max:255'],
        'samples.*.date_expected' => ['nullable', 'date'],
        'samples.*.date_sent' => ['nullable', 'date', 'before_or_equal:samples.*.date_expected'],
        'samples.*.tracking' => ['nullable', 'string', 'max:255'],
        'samples.*.weight' => ['nullable', 'numeric', 'min:0'],
        'samples.*.size' => ['nullable', 'string', 'max:255'],
        'samples.*.qty' => ['nullable', 'integer', 'min:0'],
        'samples.*.status' => ['nullable', 'string', 'in:Pending,Completed'],
        'samples.*.comments' => ['nullable', 'string', 'max:500'],
        'samples.*.do_not_charge' => ['nullable', 'boolean'],
        'samples.*.do_not_charge_dev' => ['nullable', 'boolean'],
    ])]


    public function updatedCustomer(){
        $this->reset(['style', 'styleVersion', 'colourway', 'samples']);
    }
    public function updatedSeason(){
        $this->reset(['style', 'styleVersion', 'colourway', 'samples']);
    }
    public function updatedStyleVersion(){
        $this->reset(['colourway', 'samples']);
    }
    
    public function updatedStyle($value){
        $this->reset(['styleVersion', 'colourway', 'samples']);
        
        // Auto-select if only one version
        $versions = $this->styleVersions();
        if($versions->count() == 1){
            $this->styleVersion = $versions->first()->id;
        }
    }
    
    public function updatedColourway(){
        if ($this->colourway) {
            $this->samples = $this->getSamples(); // Ensure samples are in array format for binding.
        } else {
            $this->samples = [];
        }
    }

    #[Computed]
    public function seasons(){
        return Seasons::allCached();
    }

    #[Computed]
    public function customers(){
        return Customer::allCached();
    }

    #[Computed]
    public function sampleTypes(){
        return SampleTypes::allCached();
    }

    public function styles(){
        if($this->customer && $this->season){
            return Styles::where('customers_id', $this->customer)->where('seasons_id', $this->season)->get();
        }
        else{
            return [];
        }
    }

    public function styleVersions(){
        if($this->style){
            $versions = StyleVersions::where('styles_id', $this->style)->get();
            return $versions;
        }
        else{
            return [];
        }
    }

    public function colourways(){
        if($this->styleVersion){
            return Colourways::where('style_versions_id', $this->styleVersion)->get();
        }
        else{
            return [];
        }
    }

    public function getSamples(){
        $samples = Samples::where('colourways_id', $this->colourway)->with('sample_types', 'sizes')->get();
        foreach($samples as $s=>$sample){
            $samples[$s]->size = $sample->sizes?->name;
        }
        return $samples->toArray();
    }

    public function updated($propertyName)
    {
        // Validate or handle any updates here, if needed.
        $this->validateOnly($propertyName);
    }

    public function save()
    {
        Gate::authorize('style:update');
        foreach ($this->samples as $sampleData) {
            if($sampleData['size']){
                $size = Sizes::firstOrCreate(
                    [
                        'name' => $sampleData['size']
                    ]
                );
                $sampleData['sizes_id'] = $size->id;
            }

            $data = [
                'colourways_id' => $sampleData['colourways_id'] ?? null,
                'sample_types_id' => $sampleData['sample_types_id'] ?? null,
                'date_expected' => $sampleData['date_expected'] ?? null,
                'date_sent' => $sampleData['date_sent'] ?? null,
                'tracking' => $sampleData['tracking'] ?? null,
                'weight' => isset($sampleData['weight']) && $sampleData['weight'] !== '' ? $sampleData['weight'] : null,
                'sizes_id' => $sampleData['sizes_id'] ?? null,
                'qty' => $sampleData['qty'] ?? null,
                'status' => $sampleData['status'] ?? "pending",
                'comments' => $sampleData['comments'] ?? null,
                'do_not_charge' => $sampleData['do_not_charge'] ?? null,
                'do_not_charge_dev' => $sampleData['do_not_charge_dev'] ?? null,
            ];

            if(isset($sampleData['id']) && $sampleData['id']) {
                // Update existing sample
                $sample = Samples::find($sampleData['id']);
                if($sample) {
                    $sample->update($data);
                }
            } else {
                // Create new sample
                $sample = Samples::create($data);
            }
        }
        
        session()->flash('message', 'Samples updated successfully.');
        session()->flash('alert-class', 'alert-success');
    }

    public function deleteSample($index)
    {
        if($this->samples[$index]['id']){
            $sample = Samples::find($this->samples[$index]['id']);
            $sample->delete();
        }

        // Remove the sample from the samples array
        unset($this->samples[$index]);

        // Reindex the array to maintain consistent keys
        $this->samples = array_values($this->samples);

        session()->flash('message', 'Sample deleted successfully.');
    }

    public function addSample()
    {
        $this->samples[] = [
            'colourways_id' => $this->colourway, // Default to null; user will select
            'sample_types_id' => null, // Default to null; user will select
            'date_expected' => null,
            'date_sent' => null,
            'tracking' => null,
            'weight' => null,
            'size' => null,
            'qty' => null,
            'status' => 'Pending', // Default status
            'comments' => null,
            'do_not_charge' => false,
            'do_not_charge_dev' => false,
        ];
    }

    public function render()
    {
        // if ($this->colourway) {
        //     $this->samples = $this->getSamples()->toArray(); // Ensure samples are in array format for binding.
        // }
        return view('livewire.development.samples.bulk-add');
    }
}
