<?php

namespace App\Http\Livewire\Development\Reports;

use App\Models\Styles;
use App\Models\Samples;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Livewire\Attributes\Title;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use Illuminate\Database\Query\JoinClause;
use App\Http\Livewire\FilterableComponent;
use Illuminate\Support\Str;

class StylesOverviewBeta extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    #[Validate([
        'season' => ['nullable', 'exists:seasons,id'],
        'customer' => ['nullable', 'array'],
        'factory' => ['nullable', 'exists:suppliers,id'],
        'category' => ['nullable', 'in:ladies,mens,accessories,childrens'],
        'department' => ['nullable', 'exists:departments,id'],
        'search' => ['nullable', 'string', 'max:30'],
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
        'orderBy' => ['nullable', 'in:rt,proto'],
        'sortField' => ['nullable', 'string'],
        'sortDirection' => ['nullable', 'in:asc,desc'],
        'colourwayComments' => ['nullable', 'array'],
        'colourwayComments.*' => ['nullable', 'string'],
    ])]
    public $paginate = 10, $season, $customer = [], $factory, $category, $department, $search, $hideFilters;

    public $orderBy;
    public $sortField = null;
    public $sortDirection = 'asc';
    public $showCustomerDropdown = false;
    public $colourwayComments = [];
    public $styleEditID, $sampleEditID;

    // Beta-specific properties
    public $viewMode = 'table'; // 'table' or 'cards'
    public $showFilters = true;
    public $quickFilters = [];
    public $selectedItems = [];
    public $selectAll = false;
    public $showBulkActions = false;
    public $exportFormat = 'csv';
    public $showAdvancedFilters = false;
    public $statusFilter = '';
    public $dateRange = '';
    public $weightRange = '';
    public $carryover = false;

    protected function filterKey(): string
    {
        return 'stylesoverviewbetafilters';
    }

    protected function columnsVersion(): float
    {
        return 1.8;
    }

    protected function filters(): array
    {
        return ['season', 'customer', 'factory', 'category', 'department', 'search', 'columns', 'paginate', 'orderBy', 'sortField', 'sortDirection', 'viewMode', 'quickFilters', 'statusFilter', 'dateRange', 'weightRange'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refreshDesigns()
    {
        $this->loadFilterSettings();
    }

    public $columns = [
        'Customer' => TRUE,
        'Factory' => TRUE,
        'Category' => TRUE,
        'Image' => TRUE,
        'RT Style No' => TRUE,
        'Customer Style No' => TRUE,
        'Description' => TRUE,
        'Version' => TRUE,
        'Colourway' => TRUE,
        'Date of Issue' => TRUE,
        'Carryover' => TRUE,
        'Yarn Ends' => TRUE,
        'Sample Yarn' => TRUE,
        'Sample Accessories' => TRUE,
        'Proto Approved' => TRUE,
        'Fit Approved' => TRUE,
        'Sealer Approved' => TRUE,
        'SMS Approved' => TRUE,
        'Photo Approved' => TRUE,
        'Shipment Approved' => TRUE,
        'Composition' => TRUE,
        'Yarns' => TRUE,
        'Gauge' => TRUE,
        'Weight' => TRUE,
        'Recent Sample Comment' => TRUE,
        'Colourway Comments' => TRUE,
    ];

    public function updated($propertyName)
    {
        if ($propertyName !== 'customer') {
            $this->validateOnly($propertyName);
        }

        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            if (in_array($propertyName, ['season', 'customer', 'factory', 'category', 'department', 'search', 'paginate', 'orderBy', 'viewMode', 'statusFilter', 'dateRange', 'weightRange'])) {
                $this->resetPage();
            }
            $this->saveFilterSettings();
        }
    }

    public function updatedColourwayComments($val, $cw_id)
    {
        $cw = Colourways::find($cw_id);
        if ($cw) {
            $cw->comments = $val;
            $cw->save();
        }
    }

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selectedItems = $this->colourways->pluck('id')->toArray();
        } else {
            $this->selectedItems = [];
        }
        $this->showBulkActions = !empty($this->selectedItems);
    }

    public function toggleItemSelection($itemId)
    {
        if (in_array($itemId, $this->selectedItems)) {
            $this->selectedItems = array_diff($this->selectedItems, [$itemId]);
        } else {
            $this->selectedItems[] = $itemId;
        }
        $this->showBulkActions = !empty($this->selectedItems);
        $this->selectAll = count($this->selectedItems) === $this->colourways->count();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::allCached();
    }

    #[Computed]
    public function customers()
    {
        return Customer::whereHas('styles')->get();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::whereHas('style_versions')->get();
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached();
    }

    #[On('close-modal')]
    public function closeModals()
    {
        $this->reset(['styleEditID', 'sampleEditID']);
    }

    public function mount()
    {
        $this->loadFilterSettings();

        if (!is_array($this->customer)) {
            $this->customer = [];
        }

        if (!empty($this->customer) && array_keys($this->customer) === range(0, count($this->customer) - 1)) {
            $normalized = [];
            foreach ($this->customer as $customerId) {
                $normalized[$customerId] = 1;
            }
            $this->customer = $normalized;
        }
    }

    public function select($filter, $value = 0)
    {
        if (!property_exists($this, $filter)) {
            return;
        }
        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        if (in_array($filter, ['season', 'customer', 'factory', 'category', 'department'])) {
            $this->resetPage();
            $this->saveFilterSettings();
            return;
        }
        $this->updated($filter);
    }

    public function resetFilters(): void
    {
        $this->reset([
            'season',
            'customer',
            'factory',
            'category',
            'department',
            'search',
            'orderBy',
            'sortField',
            'sortDirection',
            'quickFilters',
            'statusFilter',
            'dateRange',
            'weightRange',
            'selectedItems',
            'selectAll',
            'showBulkActions',
        ]);
        $this->paginate = 10;
        $this->viewMode = 'table';
        $this->resetPage();
        $this->saveFilterSettings();
    }

    public function toggleViewMode(): void
    {
        $this->viewMode = $this->viewMode === 'table' ? 'cards' : 'table';
        $this->saveFilterSettings();
    }

    public function applyQuickFilter(string $filter): void
    {
        $this->quickFilters = [$filter];
        
        // Apply more specific filtering logic
        switch ($filter) {
            case 'rejected':
                // Show styles that have ANY rejected samples
                $this->statusFilter = 'rejected';
                break;
            case 'carryover':
                // Show only carryover styles
                $this->carryover = true;
                break;
            case 'completed':
                // Show styles where all required samples are approved or not required
                $this->statusFilter = 'completed';
                break;
            case 'pending':
                // Show styles that have ANY pending samples
                $this->statusFilter = 'pending';
                break;
        }
        
        $this->resetPage();
        $this->saveFilterSettings();
    }

    public function exportData()
    {
        // Get the current filtered data
        $colourways = $this->getExportData();
        
        // Create CSV content
        $csv = $this->generateCsv($colourways);
        
        // Return as download
        return response()->streamDownload(function () use ($csv) {
            echo $csv;
        }, 'styles-overview-' . now()->format('Y-m-d') . '.csv');
    }

    private function getExportData()
    {
        Gate::authorize('style:read');

        $customerFilter = $this->normalizeIdArray($this->customer);

        return Colourways::select([
            'colourways.id',
            'styles.id as styles_id',
            'customers.name as customer',
            'suppliers.name as factory',
            'styles.category',
            'designs.id as rt',
            'styles.customer_ref',
            'designs.description',
            'style_versions.name as version',
            'style_versions_id',
            'colourways.name as colourway',
            'designs.issued as date_of_issue',
            'styles.carryover',
            'style_versions.yarn_ends',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'gauges.gauge',
            'colourways.img_thumb as image',
            'style_versions.sample_yarn',
            'style_versions.sample_yarn_date',
            'style_versions.sample_accessories',
            'style_versions.sample_accessories_date',
            'colourways.comments as colourway_comments',
        ])
        ->selectRaw('seasons.locked season_locked')
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"1\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"1\"') AS UNSIGNED) END as proto_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"2\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"2\"') AS UNSIGNED) END as fit_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"3\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"3\"') AS UNSIGNED) END as sealer_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"5\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"5\"') AS UNSIGNED) END as sms_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"6\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"6\"') AS UNSIGNED) END as photo_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"7\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"7\"') AS UNSIGNED) END as shipment_required")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=1 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as proto")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=2 order by created_at DESC limit 0,1) as fit")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=3 order by created_at DESC limit 0,1) as sealer")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=5 order by created_at DESC limit 0,1) as sms")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=6 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as photo")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=7 order by created_at DESC limit 0,1) as shipment")
        ->selectRaw("(SELECT CONCAT('[', GROUP_CONCAT(JSON_OBJECT('id', yc.id,'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_ref', yc.reference, 'colour_id', yc.id, 'yarn_id', y.id, 'percentage', cy.percentage, 'comp', CONCAT('[', (SELECT group_concat(json_object('material', material, 'percentage', percentage)) from yarn_compositions ycomp JOIN materials m on m.id=ycomp.materials_id where ycomp.yarn_id=y.id), ']') )), ']') FROM colourway_yarns cy JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id JOIN yarns y ON y.id = yc.yarn_id JOIN counts ON counts.id = y.counts_id WHERE cy.colourways_id=colourways.id) AS yarns")
        ->selectRaw("(SELECT weight from samples where samples.colourways_id=colourways.id AND samples.weight IS NOT NULL order by created_at DESC limit 0,1) as weight")
        ->selectRaw("(SELECT comments from samples where samples.colourways_id=colourways.id AND samples.comments IS NOT NULL order by created_at DESC limit 0,1) as comments")
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'styles.seasons_id')
        ->join('customers', 'customers.id', '=', 'styles.customers_id')
        ->join('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->join('gauges', 'gauges.id', '=', 'style_versions.gauge_id')
        ->search($this->search)
        ->whereIn('styles.departments_id', Auth::user()->user_departments->pluck('departments_id')->toArray())
        ->when(!empty($this->season), function ($query) {
            $query->whereRelation('style_versions.styles', 'seasons_id', $this->season);
        })
        ->when(!empty($customerFilter), function ($query) use ($customerFilter) {
            $query->whereRelationIn('style_versions.styles', 'customers_id', $customerFilter);
        })
        ->when(!empty($this->factory), function ($query) {
            $query->whereRelation('style_versions', 'factory_id', $this->factory);
        })
        ->when(!empty($this->category), function ($query) {
            $query->whereRelation('style_versions.styles', 'category', $this->category);
        })
        ->when(!empty($this->department), function ($query) {
            $query->whereRelation('style_versions.styles', 'departments_id', $this->department);
        })
        ->when(!empty($this->statusFilter), function ($query) {
            // More sophisticated status filtering
            if ($this->statusFilter === 'pending') {
                $query->whereHas('samples', function($q) {
                    $q->whereIn('status', ['pending', 'sent']);
                });
            } elseif ($this->statusFilter === 'rejected') {
                $query->whereHas('samples', function($q) {
                    $q->where('status', 'rejected');
                });
            }
            // 'completed' status will be filtered after getting the data
        })
        ->when(!empty($this->dateRange), function ($query) {
            // Add date range filtering logic
        })
        ->when(!empty($this->weightRange), function ($query) {
            // Add weight range filtering logic
        })
        ->when(isset($this->carryover) && $this->carryover, function ($query) {
            $query->where('styles.carryover', true);
        })
        ->when(empty($this->sortField) && !empty($this->orderBy), function ($query) {
            if ($this->orderBy === 'rt') {
                $query->orderBy('designs.id');
            } elseif ($this->orderBy === 'proto') {
                $query->orderBy('colourways.id');
            }
        })
        ->when(!empty($this->sortField), function ($query) {
            $columns = $this->sortableColumns();
            $field = $this->sortField;
            $direction = $this->sortDirection === 'desc' ? 'desc' : 'asc';
            if (!isset($columns[$field])) {
                return;
            }
            $config = $columns[$field];
            if ($config['type'] === 'column') {
                $query->orderBy($config['value'], $direction);
            } elseif ($config['type'] === 'raw') {
                $query->orderByRaw($config['value'] . ' ' . $direction);
            }
        })
        ->orderBy('styles.cancelled')
        ->orderBy('colourways.cancelled')
        ->orderBy('styles.id')
        ->get();
    }

    private function generateCsv($colourways)
    {
        $headers = [
            'Customer', 'Factory', 'Category', 'RT Style No', 'Customer Style No', 
            'Description', 'Version', 'Colourway', 'Date of Issue', 'Carryover',
            'Yarn Ends', 'Sample Yarn', 'Sample Accessories', 'Proto Approved',
            'Fit Approved', 'Sealer Approved', 'SMS Approved', 'Photo Approved',
            'Shipment Approved', 'Composition', 'Yarns', 'Gauge', 'Weight',
            'Recent Sample Comment', 'Colourway Comments'
        ];

        $csv = implode(',', $headers) . "\n";

        foreach ($colourways as $colourway) {
            $composition = [];
            if (!empty($colourway->yarns)) {
                $yarns = json_decode($colourway->yarns);
                foreach ($yarns as $yarn) {
                    if (!empty($yarn->comp)) {
                        foreach (json_decode($yarn->comp) as $material) {
                            if (!empty($composition[$material->material])) {
                                $composition[$material->material] = number_format($composition[$material->material] + (($material->percentage / 100) * $yarn->percentage), 0);
                            } else {
                                $composition[$material->material] = number_format(($material->percentage / 100) * $yarn->percentage, 0);
                            }
                        }
                    }
                }
            }

            $row = [
                $colourway->customer,
                $colourway->factory,
                Str::title($colourway->category),
                'RTD' . $colourway->rt,
                $colourway->customer_ref,
                Str::title($colourway->description),
                Str::title($colourway->version),
                Str::title($colourway->colourway),
                $colourway->date_of_issue ? \Carbon\Carbon::parse($colourway->date_of_issue)->format('d-M-Y') : '',
                $colourway->carryover ? 'Yes' : 'No',
                $colourway->yarn_ends,
                $colourway->sample_yarn,
                $colourway->sample_accessories,
                $this->getSampleStatus($colourway->proto),
                $this->getSampleStatus($colourway->fit),
                $this->getSampleStatus($colourway->sealer),
                $this->getSampleStatus($colourway->sms),
                $this->getSampleStatus($colourway->photo),
                $this->getSampleStatus($colourway->shipment),
                implode(', ', array_map(fn($k, $v) => "$k $v%", array_keys($composition), $composition)),
                $colourway->yarns,
                $colourway->gauge,
                $colourway->weight,
                $colourway->comments,
                $colourway->colourway_comments
            ];

            $csv .= '"' . implode('","', array_map('addslashes', $row)) . '"' . "\n";
        }

        return $csv;
    }

    private function getSampleStatus($sampleJson)
    {
        if (empty($sampleJson)) return '';
        $sample = json_decode($sampleJson);
        return $sample->status ?? '';
    }

    private function getCleanStatusDisplay($sampleJson, $required = false)
    {
        if (empty($sampleJson)) {
            return $required ? 'Not Ordered' : 'Not Required';
        }
        
        $sample = json_decode($sampleJson);
        $status = $sample->status ?? '';
        
        switch ($status) {
            case 'approved':
                return 'Approved';
            case 'rejected':
                return 'Rejected';
            case 'pending':
                return 'Pending';
            case 'sent':
                return 'Sent';
            default:
                return $required ? 'Not Ordered' : 'Not Required';
        }
    }

    private function getStatusBadgeClass($sampleJson, $required = false)
    {
        if (empty($sampleJson)) {
            return $required ? 'bg-warning text-dark' : 'bg-secondary';
        }
        
        $sample = json_decode($sampleJson);
        $status = $sample->status ?? '';
        
        switch ($status) {
            case 'approved':
                return 'bg-success';
            case 'rejected':
                return 'bg-danger';
            case 'pending':
            case 'sent':
                return 'bg-warning text-dark';
            default:
                return $required ? 'bg-warning text-dark' : 'bg-secondary';
        }
    }

    private function getOverallStatus($colourway)
    {
        $statuses = [
            'proto' => $this->getSampleStatus($colourway->proto),
            'fit' => $this->getSampleStatus($colourway->fit),
            'sealer' => $this->getSampleStatus($colourway->sealer),
            'sms' => $this->getSampleStatus($colourway->sms),
            'photo' => $this->getSampleStatus($colourway->photo),
            'shipment' => $this->getSampleStatus($colourway->shipment)
        ];

        $required = [
            'proto' => $colourway->proto_required,
            'fit' => $colourway->fit_required,
            'sealer' => $colourway->sealer_required,
            'sms' => $colourway->sms_required,
            'photo' => $colourway->photo_required,
            'shipment' => $colourway->shipment_required
        ];

        // Check if all required samples are completed (approved or not required)
        $allCompleted = true;
        $hasRejected = false;
        $hasPending = false;

        foreach ($statuses as $type => $status) {
            if ($required[$type]) {
                if ($status === 'rejected') {
                    $hasRejected = true;
                    $allCompleted = false;
                } elseif ($status === 'pending' || $status === 'sent') {
                    $hasPending = true;
                    $allCompleted = false;
                } elseif ($status !== 'approved') {
                    $allCompleted = false;
                }
            }
        }

        if ($hasRejected) return 'rejected';
        if ($hasPending) return 'pending';
        if ($allCompleted) return 'completed';
        return 'in_progress';
    }

    public function toggleAdvancedFilters(): void
    {
        $this->showAdvancedFilters = !$this->showAdvancedFilters;
    }

    public function bulkAction(string $action): void
    {
        if (empty($this->selectedItems)) {
            return;
        }

        switch ($action) {
            case 'export':
                $this->exportData();
                break;
            case 'delete':
                // Implement bulk delete
                break;
            case 'update_status':
                // Implement bulk status update
                break;
        }

        $this->selectedItems = [];
        $this->selectAll = false;
        $this->showBulkActions = false;
    }

    private function getCurrentFilters(): array
    {
        return [
            'season' => $this->season,
            'customer' => $this->customer,
            'factory' => $this->factory,
            'category' => $this->category,
            'department' => $this->department,
            'search' => $this->search,
            'statusFilter' => $this->statusFilter,
            'dateRange' => $this->dateRange,
            'weightRange' => $this->weightRange,
        ];
    }

    public function saveColourwayComment(int $colourwayId): void
    {
        $val = $this->colourwayComments[$colourwayId] ?? null;
        if ($val === null) {
            return;
        }
        $cw = Colourways::find($colourwayId);
        if ($cw) {
            $cw->comments = $val;
            $cw->save();
        }
    }

    #[On('refresh')]
    public function render()
    {
        Gate::authorize('style:read');

        $customerFilter = $this->normalizeIdArray($this->customer);

        $colourways = Colourways::select([
            'colourways.id',
            'styles.id as styles_id',
            'customers.name as customer',
            'suppliers.name as factory',
            'styles.category',
            'designs.id as rt',
            'styles.customer_ref',
            'designs.description',
            'style_versions.name as version',
            'style_versions_id',
            'colourways.name as colourway',
            'designs.issued as date_of_issue',
            'styles.carryover',
            'style_versions.yarn_ends',
            'colourways.cancelled as cw_cancelled',
            'styles.cancelled as s_cancelled',
            'gauges.gauge',
            'colourways.img_thumb as image',
            'style_versions.sample_yarn',
            'style_versions.sample_yarn_date',
            'style_versions.sample_accessories',
            'style_versions.sample_accessories_date',
            'colourways.comments as colourway_comments',
        ])
        ->selectRaw('seasons.locked season_locked')
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"1\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"1\"') AS UNSIGNED) END as proto_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"2\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"2\"') AS UNSIGNED) END as fit_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"3\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"3\"') AS UNSIGNED) END as sealer_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"5\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"5\"') AS UNSIGNED) END as sms_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"6\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"6\"') AS UNSIGNED) END as photo_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"7\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"7\"') AS UNSIGNED) END as shipment_required")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=1 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as proto")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=2 order by created_at DESC limit 0,1) as fit")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=3 order by created_at DESC limit 0,1) as sealer")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=5 order by created_at DESC limit 0,1) as sms")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=6 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as photo")
        ->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=7 order by created_at DESC limit 0,1) as shipment")
        ->selectRaw("(SELECT CONCAT('[', GROUP_CONCAT(JSON_OBJECT('id', yc.id,'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_ref', yc.reference, 'colour_id', yc.id, 'yarn_id', y.id, 'percentage', cy.percentage, 'comp', CONCAT('[', (SELECT group_concat(json_object('material', material, 'percentage', percentage)) from yarn_compositions ycomp JOIN materials m on m.id=ycomp.materials_id where ycomp.yarn_id=y.id), ']') )), ']') FROM colourway_yarns cy JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id JOIN yarns y ON y.id = yc.yarn_id JOIN counts ON counts.id = y.counts_id WHERE cy.colourways_id=colourways.id) AS yarns")
        ->selectRaw("(SELECT weight from samples where samples.colourways_id=colourways.id AND samples.weight IS NOT NULL order by created_at DESC limit 0,1) as weight")
        ->selectRaw("(SELECT comments from samples where samples.colourways_id=colourways.id AND samples.comments IS NOT NULL order by created_at DESC limit 0,1) as comments")
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'styles.seasons_id')
        ->join('customers', 'customers.id', '=', 'styles.customers_id')
        ->join('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->join('gauges', 'gauges.id', '=', 'style_versions.gauge_id')
        ->search($this->search)
        ->whereIn('styles.departments_id', Auth::user()->user_departments->pluck('departments_id')->toArray())
        ->when(!empty($this->season), function ($query) {
            $query->whereRelation('style_versions.styles', 'seasons_id', $this->season);
        })
        ->when(!empty($customerFilter), function ($query) use ($customerFilter) {
            $query->whereRelationIn('style_versions.styles', 'customers_id', $customerFilter);
        })
        ->when(!empty($this->factory), function ($query) {
            $query->whereRelation('style_versions', 'factory_id', $this->factory);
        })
        ->when(!empty($this->category), function ($query) {
            $query->whereRelation('style_versions.styles', 'category', $this->category);
        })
        ->when(!empty($this->department), function ($query) {
            $query->whereRelation('style_versions.styles', 'departments_id', $this->department);
        })
        ->when(!empty($this->statusFilter), function ($query) {
            // More sophisticated status filtering
            if ($this->statusFilter === 'pending') {
                $query->whereHas('samples', function($q) {
                    $q->whereIn('status', ['pending', 'sent']);
                });
            } elseif ($this->statusFilter === 'rejected') {
                $query->whereHas('samples', function($q) {
                    $q->where('status', 'rejected');
                });
            }
            // 'completed' status will be filtered after getting the data
        })
        ->when(!empty($this->dateRange), function ($query) {
            // Add date range filtering logic
        })
        ->when(!empty($this->weightRange), function ($query) {
            // Add weight range filtering logic
        })
        ->when(isset($this->carryover) && $this->carryover, function ($query) {
            $query->where('styles.carryover', true);
        })
        ->when(empty($this->sortField) && !empty($this->orderBy), function ($query) {
            if ($this->orderBy === 'rt') {
                $query->orderBy('designs.id');
            } elseif ($this->orderBy === 'proto') {
                $query->orderBy('colourways.id');
            }
        })
        ->when(!empty($this->sortField), function ($query) {
            $columns = $this->sortableColumns();
            $field = $this->sortField;
            $direction = $this->sortDirection === 'desc' ? 'desc' : 'asc';
            if (!isset($columns[$field])) {
                return;
            }
            $config = $columns[$field];
            if ($config['type'] === 'column') {
                $query->orderBy($config['value'], $direction);
            } elseif ($config['type'] === 'raw') {
                $query->orderByRaw($config['value'] . ' ' . $direction);
            }
        })
        ->orderBy('styles.cancelled')
        ->orderBy('colourways.cancelled')
        ->orderBy('styles.id');

        // Apply 'completed' filter before pagination
        if ($this->statusFilter === 'completed') {
            $colourways = $colourways->get()->filter(function ($colourway) {
                return $this->getOverallStatus($colourway) === 'completed';
            });
            
            // Manually paginate the filtered results
            $perPage = $this->paginate;
            $currentPage = request()->get('page', 1);
            $offset = ($currentPage - 1) * $perPage;
            $items = $colourways->slice($offset, $perPage);
            
            $colourways = new \Illuminate\Pagination\LengthAwarePaginator(
                $items,
                $colourways->count(),
                $perPage,
                $currentPage,
                ['path' => request()->url(), 'pageName' => 'page']
            );
        } else {
            $colourways = $colourways->paginate($this->paginate);
        }

        foreach ($colourways as $colourway) {
            $colourway->yarns = json_decode($colourway->yarns);

            $composition = [];
            if (!empty($colourway->yarns)) {
                foreach ($colourway->yarns as $yarn) {
                    if (!empty($yarn->comp)) {
                        foreach (json_decode($yarn->comp) as $material) {
                            if (!empty($composition[$material->material])) {
                                $composition[$material->material] = number_format($composition[$material->material] + (($material->percentage / 100) * $yarn->percentage), 0);
                            } else {
                                $composition[$material->material] = number_format(($material->percentage / 100) * $yarn->percentage, 0);
                            }
                        }
                    }
                }
            }
            $colourway->composition = collect($composition);
        }

        return view('livewire.development.reports.styles-overview-beta', ['colourways' => $colourways]);
    }

    public function sortBy(string $field): void
    {
        $columns = $this->sortableColumns();
        if (!isset($columns[$field])) {
            return;
        }
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
        $this->saveFilterSettings();
    }

    private function sortableColumns(): array
    {
        $protoApprovedExpr = $this->approvalSortCase([1, 8], 1);
        $fitApprovedExpr = $this->approvalSortCase([2], 2);
        $sealerApprovedExpr = $this->approvalSortCase([3], 3);
        $smsApprovedExpr = $this->approvalSortCase([5], 5);
        $photoApprovedExpr = $this->approvalSortCase([6, 8], 6);
        $shipmentApprovedExpr = $this->approvalSortCase([7], 7);

        $latestCommentsExpr = "(SELECT comments FROM samples WHERE samples.colourways_id = colourways.id AND samples.comments IS NOT NULL ORDER BY created_at DESC LIMIT 1)";
        $latestWeightExpr = "(SELECT weight FROM samples WHERE samples.colourways_id = colourways.id AND samples.weight IS NOT NULL ORDER BY created_at DESC LIMIT 1)";
        $yarnsExpr = "(SELECT CONCAT('[', GROUP_CONCAT(JSON_OBJECT('id', yc.id,'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_ref', yc.reference, 'colour_id', yc.id, 'yarn_id', y.id, 'percentage', cy.percentage, 'comp', CONCAT('[', (SELECT group_concat(json_object('material', material, 'percentage', percentage)) from yarn_compositions ycomp JOIN materials m on m.id=ycomp.materials_id where ycomp.yarn_id=y.id), ']') )), ']') FROM colourway_yarns cy JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id JOIN yarns y ON y.id = yc.yarn_id JOIN counts ON counts.id = y.counts_id WHERE cy.colourways_id=colourways.id)";

        return [
            'customer' => ['type' => 'column', 'value' => 'customers.name'],
            'factory' => ['type' => 'column', 'value' => 'suppliers.name'],
            'category' => ['type' => 'column', 'value' => 'styles.category'],
            'image' => ['type' => 'raw', 'value' => '(colourways.img_thumb IS NOT NULL)'],
            'rt' => ['type' => 'column', 'value' => 'designs.id'],
            'customer_ref' => ['type' => 'column', 'value' => 'styles.customer_ref'],
            'description' => ['type' => 'column', 'value' => 'designs.description'],
            'version' => ['type' => 'column', 'value' => 'style_versions.name'],
            'colourway' => ['type' => 'column', 'value' => 'colourways.name'],
            'date_of_issue' => ['type' => 'column', 'value' => 'designs.issued'],
            'carryover' => ['type' => 'column', 'value' => 'styles.carryover'],
            'yarn_ends' => ['type' => 'column', 'value' => 'style_versions.yarn_ends'],
            'sample_yarn' => ['type' => 'column', 'value' => 'style_versions.sample_yarn'],
            'sample_yarn_date' => ['type' => 'column', 'value' => 'style_versions.sample_yarn_date'],
            'sample_accessories' => ['type' => 'column', 'value' => 'style_versions.sample_accessories'],
            'sample_accessories_date' => ['type' => 'column', 'value' => 'style_versions.sample_accessories_date'],
            'proto_approved' => ['type' => 'raw', 'value' => $protoApprovedExpr],
            'fit_approved' => ['type' => 'raw', 'value' => $fitApprovedExpr],
            'sealer_approved' => ['type' => 'raw', 'value' => $sealerApprovedExpr],
            'sms_approved' => ['type' => 'raw', 'value' => $smsApprovedExpr],
            'photo_approved' => ['type' => 'raw', 'value' => $photoApprovedExpr],
            'shipment_approved' => ['type' => 'raw', 'value' => $shipmentApprovedExpr],
            'composition' => ['type' => 'raw', 'value' => $yarnsExpr],
            'yarns' => ['type' => 'raw', 'value' => $yarnsExpr],
            'gauge' => ['type' => 'column', 'value' => 'gauges.gauge'],
            'weight' => ['type' => 'raw', 'value' => $latestWeightExpr],
            'comments' => ['type' => 'raw', 'value' => $latestCommentsExpr],
            'colourway_comments' => ['type' => 'column', 'value' => 'colourways.comments'],
        ];
    }

    private function approvalSortCase(array $sampleTypeIds, int $requiredJsonKey): string
    {
        $typesList = implode(',', array_map('intval', $sampleTypeIds));
        $latestStatus = "(SELECT status FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $latestSent = "(SELECT date_sent FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $latestId = "(SELECT id FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $requiredExpr = "(CASE WHEN json_extract(samples_required, '$.\"$requiredJsonKey\"') IS NULL THEN 0 ELSE CAST(json_extract(samples_required, '$.\"$requiredJsonKey\"') AS UNSIGNED) END)";

        return "(CASE "
            . "WHEN $latestStatus = 'approved' THEN 0 "
            . "WHEN $latestStatus = 'rejected' THEN 1 "
            . "WHEN $latestSent IS NOT NULL THEN 2 "
            . "WHEN $latestId IS NULL AND $requiredExpr = 1 THEN 3 "
            . "ELSE 4 END)";
    }

    protected function normalizeIdArray($value): array
    {
        if (!is_array($value) || empty($value)) {
            return [];
        }
        $isNumericList = array_keys($value) === range(0, count($value) - 1);
        return $isNumericList ? array_values($value) : array_keys($value);
    }
}



