<?php

namespace App\Http\Livewire\Development\Reports;

use App\Models\Styles;
use App\Models\Samples;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use App\Models\StyleVersions;
use Livewire\Attributes\Title;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use Illuminate\Database\Query\JoinClause;
use App\Http\Livewire\FilterableComponent;

class StylesOverview extends FilterableComponent
{
	use WithPagination;
	use Filterable;

    #[Validate([
        'season' => ['nullable', 'exists:seasons,id'],
        'customer' => ['nullable', 'array'],
        'factory' => ['nullable', 'exists:suppliers,id'],
        'category' => ['nullable', 'in:ladies,mens,accessories,childrens'],
        'department' => ['nullable', 'exists:departments,id'],
        'search' => ['nullable', 'string', 'max:30'],
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
        'orderBy' => ['nullable', 'in:rt,proto'],
        'sortField' => ['nullable', 'string'],
        'sortDirection' => ['nullable', 'in:asc,desc'],
        'colourwayComments' => ['nullable', 'array'],
        'colourwayComments.*' => ['nullable', 'string'], // Validate each comment as a string
    ])]
	public $paginate = 10, $season, $customer = [], $factory, $category, $department, $search, $hideFilters;

    public $orderBy;

    public $sortField = null;
    public $sortDirection = 'asc';

    // Filter dropdown visibility properties
    public $showCustomerDropdown = false;

    public $colourwayComments = [];

	public $styleEditID, $sampleEditID;

	protected function filterKey(): string
    {
        return 'stylesoverviewfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.6;
    }

    protected function filters(): array
    {
        return ['season', 'customer', 'factory', 'category', 'department', 'search', 'columns', 'paginate', 'orderBy', 'sortField', 'sortDirection'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refreshDesigns()
    {
        $this->loadFilterSettings();
    }

	public $columns = [
		'Customer' => TRUE,
		'Factory' => TRUE,
		'Category' => TRUE,
		'Image' => TRUE,
		'RT Style No' => TRUE,
		'Customer Style No' => TRUE,
		'Description' => TRUE,
		'Version' => TRUE,
		'Colourway' => TRUE,
		'Date of Issue' => TRUE,
		'Carryover' => TRUE,
		'Yarn Ends' => TRUE,
        'Sample Yarn' => TRUE,
        'Sample Accessories' => TRUE,
		'Proto Approved' => TRUE,
		'Fit Approved' => TRUE,
		'Sealer Approved' => TRUE,
		'SMS Approved' => TRUE,
		'Photo Approved' => TRUE,
		'Shipment Approved' => TRUE,
		'Composition' => TRUE,
		'Yarns' => TRUE,
		'Gauge' => TRUE,
		'Weight' => TRUE,
		'Recent Sample Comment' => TRUE,
        'Colourway Comments' => TRUE,
	];

    public function updated($propertyName)
    {
        // Validate individual field
        // Lightweight validation per field (skip complex customer array)
        if ($propertyName !== 'customer') {
            $this->validateOnly($propertyName);
        }

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['season', 'customer', 'factory', 'category', 'department', 'search', 'paginate', 'orderBy'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
            // No additional state churn
        }
    }

    // Initialize colourway comments array with existing values from the current page
    private function initializeColourwayComments($colourways)
    {
        $this->colourwayComments = [];
        foreach ($colourways as $colourway) {
            $this->colourwayComments[$colourway->id] = $colourway->colourway_comments ?? '';
        }
    }

    // Persist colourway comments when they change via wire:model
    public function updatedColourwayComments($val, $cw_id){
        $cw = Colourways::find($cw_id);
        if ($cw) {
            $cw->comments = $val;
            $cw->save();
        }
    }

	#[Computed]
	public function seasons(){
		return Seasons::allCached();
	}
	#[Computed]
	public function customers(){
		return Customer::whereHas('styles')->get();
	}
	#[Computed]
	public function factories(){
		return Suppliers::whereHas('style_versions')->get();
	}
	#[Computed]
	public function departments(){
		return Departments::allCached();
	}

	#[On('close-modal')]
	public function closeModals(){
		$this->reset(['styleEditID', 'sampleEditID']);
	}

    public function mount()
    {
        $this->loadFilterSettings();

        if(!is_array($this->customer)){
            $this->customer = [];
        }

        // Normalize customer filter to keyed array for UI (id => 1)
        if (! empty($this->customer) && array_keys($this->customer) === range(0, count($this->customer) - 1)) {
            $normalized = [];
            foreach ($this->customer as $customerId) {
                $normalized[$customerId] = 1;
            }
            $this->customer = $normalized;
        }

        // Ensure other filters are initialized (no-op if already set)
    }

    // Methods for filter selection (match Shipment Schedule UX)
    public function select($filter, $value = 0)
    {
        if (!property_exists($this, $filter)) {
            return;
        }
        if (is_string($this->$filter)) {
            $this->$filter = [];
        }
        if ($value == 0) {
            $this->$filter = [];
        } else {
            if (($this->$filter[$value] ?? 0) == 0) {
                $this->$filter[$value] = 1;
            } else {
                unset($this->$filter[$value]);
            }
        }

        // Trigger pagination reset and re-render without running per-field validation on complex array
        if (in_array($filter, ['season', 'customer', 'factory', 'category', 'department'])) {
            $this->resetPage();
            $this->saveFilterSettings();
            return; // Livewire will re-render automatically due to state change
        }
        $this->updated($filter);
    }

    #[On('refreshData')]
    public function refreshData()
    {
        // Refresh to show updated style and sample data
        $this->resetPage();
    }

	#[On('refresh')]
    public function render()
    {
		Gate::authorize('style:read');
        // Support both legacy array-of-ids and keyed array (id=>1) for customers filter
        $customerFilter = $this->normalizeIdArray($this->customer);
        
        // Get user departments to avoid calling multiple times and handle empty array case
        $userDepartments = Auth::user()->user_departments->pluck('departments_id')->toArray();

        $colourways = Colourways::select([
			'colourways.id',
			'styles.id as styles_id',
			'customers.name as customer',
			'suppliers.name as factory',
			'styles.category',
			'designs.id as rt',
			'styles.customer_ref',
			'designs.description',
			'style_versions.name as version',
            'style_versions_id',
			'colourways.name as colourway',
			'designs.issued as date_of_issue',
			'styles.carryover',
			'style_versions.yarn_ends',
			'colourways.cancelled as cw_cancelled',
			'styles.cancelled as s_cancelled',
			'gauges.gauge',
            'colourways.img_thumb as image',
            'style_versions.sample_yarn',
            'style_versions.sample_yarn_date',
            'style_versions.sample_accessories',
            'style_versions.sample_accessories_date',
            'colourways.comments as colourway_comments',
		])
        ->selectRaw('seasons.locked season_locked')
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"1\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"1\"') AS UNSIGNED) END as proto_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"2\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"2\"') AS UNSIGNED) END as fit_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"3\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"3\"') AS UNSIGNED) END as sealer_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"5\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"5\"') AS UNSIGNED) END as sms_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"6\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"6\"') AS UNSIGNED) END as photo_required")
        ->selectRaw("CASE WHEN json_extract(samples_required, '$.\"7\"') IS NULL THEN false ELSE CAST(json_extract(samples_required, '$.\"7\"') AS UNSIGNED) END as shipment_required")


		// ->selectRaw("(select image from colourways c where c.style_versions_id=style_versions.id and image IS NOT NULL limit 0,1) as image")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=1 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as proto")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=2 order by created_at DESC limit 0,1) as fit")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=3 order by created_at DESC limit 0,1) as sealer")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=5 order by created_at DESC limit 0,1) as sms")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND (samples.sample_types_id=6 OR samples.sample_types_id=8) order by created_at DESC limit 0,1) as photo")
		->selectRaw("(SELECT JSON_OBJECT('id', samples.id, 'date_expected', samples.date_expected, 'date_sent', samples.date_sent, 'status', samples.status, 'approved_date', samples.approved_date, 'status', samples.status) from samples where samples.colourways_id=colourways.id AND samples.sample_types_id=7 order by created_at DESC limit 0,1) as shipment")
		->selectRaw("(SELECT
			CONCAT('[', GROUP_CONCAT(DISTINCT JSON_OBJECT('id', yc.id,'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_ref', yc.reference, 'colour_id', yc.id, 'yarn_id', y.id, 'percentage', cy.percentage, 'comp', CONCAT('[', (SELECT group_concat(json_object('material', material, 'percentage', percentage)) from yarn_compositions ycomp JOIN materials m on m.id=ycomp.materials_id where ycomp.yarn_id=y.id), ']') ) ORDER BY y.description, yc.reference), ']')
			FROM colourway_yarns cy
			JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id
			JOIN yarns y ON y.id = yc.yarn_id
			JOIN counts ON counts.id = y.counts_id
			WHERE cy.colourways_id=colourways.id
		) AS yarns")
		->selectRaw("(SELECT weight from samples where samples.colourways_id=colourways.id AND samples.weight IS NOT NULL order by created_at DESC limit 0,1) as weight")
		->selectRaw("(SELECT comments from samples where samples.colourways_id=colourways.id AND samples.comments IS NOT NULL order by created_at DESC limit 0,1) as comments")
		->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
		->join('styles', 'styles.id', '=', 'style_versions.styles_id')
		->join('seasons', 'seasons.id', '=', 'styles.seasons_id')
		->join('customers', 'customers.id', '=', 'styles.customers_id')
		->join('suppliers', 'suppliers.id', '=', 'style_versions.factory_id')
		->join('designs', 'designs.id', '=', 'styles.designs_id')
		->join('gauges', 'gauges.id', '=', 'style_versions.gauge_id')
        ->search($this->search)
        ->when(!empty($userDepartments), function($query) use ($userDepartments){
            $query->whereIn('styles.departments_id', $userDepartments);
        })
		->when(!empty($this->season),function($query){
			$query->whereRelation('style_versions.styles', 'seasons_id', $this->season);
		})
        ->when(!empty($customerFilter),function($query) use ($customerFilter){
            $query->whereRelationIn('style_versions.styles', 'customers_id', $customerFilter);
		})
		->when(!empty($this->factory),function($query){
			$query->whereRelation('style_versions', 'factory_id', $this->factory);
		})
		->when(!empty($this->category),function($query){
			$query->whereRelation('style_versions.styles', 'category', $this->category);
		})
		->when(!empty($this->department),function($query){
			$query->whereRelation('style_versions.styles', 'departments_id', $this->department);
		})
        ->when(empty($this->sortField) && !empty($this->orderBy), function ($query) {
            if ($this->orderBy === 'rt') {
                $query->orderBy('designs.id');
            } elseif ($this->orderBy === 'proto') {
                $query->orderBy('colourways.id');
            }
        })
        ->when(!empty($this->sortField), function ($query) {
            $columns = $this->sortableColumns();
            $field = $this->sortField;
            $direction = $this->sortDirection === 'desc' ? 'desc' : 'asc';
            if (!isset($columns[$field])) {
                return;
            }
            $config = $columns[$field];
            if ($config['type'] === 'column') {
                $query->orderBy($config['value'], $direction);
            } elseif ($config['type'] === 'raw') {
                $query->orderByRaw($config['value'] . ' ' . $direction);
            }
        })

        ->orderBy('styles.cancelled')
        ->orderBy('colourways.cancelled')
        ->orderBy('styles.id')
		->paginate($this->paginate);

        foreach($colourways as $colourway){

			$colourway->customer_settings = json_decode(str_replace('-', '', $colourway->customer_settings), true);
			$colourway->yarns = json_decode($colourway->yarns);

			$composition = [];
			if(!empty($colourway->yarns)){
				foreach($colourway->yarns as $yarn){
					if(!empty($yarn->comp)){
						foreach(json_decode($yarn->comp) as $material){
							if(!empty($composition[$material->material]))
								$composition[$material->material] = number_format($composition[$material->material] + (($material->percentage/100) * $yarn->percentage), 0);
							else
								$composition[$material->material] = number_format(($material->percentage/100) * $yarn->percentage, 0);
						}
					}
				}
			}
			$colourway->composition = collect($composition);

            // Do not mutate component state during render to keep payload small
		}

        // Initialize the colourway comments array with existing values
        $this->initializeColourwayComments($colourways);
        
        // Debug: Check if comments are being loaded
        // dd($colourways->first()->colourway_comments ?? 'No comments found');
        
        return view('livewire.development.reports.styles-overview', ['colourways' => $colourways]);
    }

    public function sortBy(string $field): void
    {
        $columns = $this->sortableColumns();
        if (!isset($columns[$field])) {
            return;
        }
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortField = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
        $this->saveFilterSettings();
    }

    private function sortableColumns(): array
    {
        // Custom sort order buckets for approvals:
        // 0 Approved, 1 Rejected, 2 Sent, 3 Not ordered (required & no sample), 4 Other/Not required
        $protoApprovedExpr = $this->approvalSortCase([1, 8], 1);
        $fitApprovedExpr = $this->approvalSortCase([2], 2);
        $sealerApprovedExpr = $this->approvalSortCase([3], 3);
        $smsApprovedExpr = $this->approvalSortCase([5], 5);
        $photoApprovedExpr = $this->approvalSortCase([6, 8], 6);
        $shipmentApprovedExpr = $this->approvalSortCase([7], 7);

        $latestCommentsExpr = "(SELECT comments FROM samples WHERE samples.colourways_id = colourways.id AND samples.comments IS NOT NULL ORDER BY created_at DESC LIMIT 1)";
        $latestWeightExpr = "(SELECT weight FROM samples WHERE samples.colourways_id = colourways.id AND samples.weight IS NOT NULL ORDER BY created_at DESC LIMIT 1)";
        $yarnsExpr = "(SELECT CONCAT('[', GROUP_CONCAT(DISTINCT JSON_OBJECT('id', yc.id,'yarn', y.description, 'count', counts.count, 'colour', yc.description, 'colour_ref', yc.reference, 'colour_id', yc.id, 'yarn_id', y.id, 'percentage', cy.percentage, 'comp', CONCAT('[', (SELECT group_concat(json_object('material', material, 'percentage', percentage)) from yarn_compositions ycomp JOIN materials m on m.id=ycomp.materials_id where ycomp.yarn_id=y.id), ']') ) ORDER BY y.description, yc.reference), ']') FROM colourway_yarns cy JOIN yarn_colours yc ON yc.id = cy.yarn_colours_id JOIN yarns y ON y.id = yc.yarn_id JOIN counts ON counts.id = y.counts_id WHERE cy.colourways_id=colourways.id)";

        return [
            'customer' => ['type' => 'column', 'value' => 'customers.name'],
            'factory' => ['type' => 'column', 'value' => 'suppliers.name'],
            'category' => ['type' => 'column', 'value' => 'styles.category'],
            'image' => ['type' => 'raw', 'value' => '(colourways.img_thumb IS NOT NULL)'],
            'rt' => ['type' => 'column', 'value' => 'designs.id'],
            'customer_ref' => ['type' => 'column', 'value' => 'styles.customer_ref'],
            'description' => ['type' => 'column', 'value' => 'designs.description'],
            'version' => ['type' => 'column', 'value' => 'style_versions.name'],
            'colourway' => ['type' => 'column', 'value' => 'colourways.name'],
            'date_of_issue' => ['type' => 'column', 'value' => 'designs.issued'],
            'carryover' => ['type' => 'column', 'value' => 'styles.carryover'],
            'yarn_ends' => ['type' => 'column', 'value' => 'style_versions.yarn_ends'],
            'sample_yarn' => ['type' => 'column', 'value' => 'style_versions.sample_yarn'],
            'sample_yarn_date' => ['type' => 'column', 'value' => 'style_versions.sample_yarn_date'],
            'sample_accessories' => ['type' => 'column', 'value' => 'style_versions.sample_accessories'],
            'sample_accessories_date' => ['type' => 'column', 'value' => 'style_versions.sample_accessories_date'],
            'proto_approved' => ['type' => 'raw', 'value' => $protoApprovedExpr],
            'fit_approved' => ['type' => 'raw', 'value' => $fitApprovedExpr],
            'sealer_approved' => ['type' => 'raw', 'value' => $sealerApprovedExpr],
            'sms_approved' => ['type' => 'raw', 'value' => $smsApprovedExpr],
            'photo_approved' => ['type' => 'raw', 'value' => $photoApprovedExpr],
            'shipment_approved' => ['type' => 'raw', 'value' => $shipmentApprovedExpr],
            'composition' => ['type' => 'raw', 'value' => $yarnsExpr],
            'yarns' => ['type' => 'raw', 'value' => $yarnsExpr],
            'gauge' => ['type' => 'column', 'value' => 'gauges.gauge'],
            'weight' => ['type' => 'raw', 'value' => $latestWeightExpr],
            'comments' => ['type' => 'raw', 'value' => $latestCommentsExpr],
            'colourway_comments' => ['type' => 'column', 'value' => 'colourways.comments'],
        ];
    }

    private function approvalSortCase(array $sampleTypeIds, int $requiredJsonKey): string
    {
        $typesList = implode(',', array_map('intval', $sampleTypeIds));
        $latestStatus = "(SELECT status FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $latestSent = "(SELECT date_sent FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $latestId = "(SELECT id FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN ($typesList) ORDER BY created_at DESC LIMIT 1)";
        $requiredExpr = "(CASE WHEN json_extract(samples_required, '$.\"$requiredJsonKey\"') IS NULL THEN 0 ELSE CAST(json_extract(samples_required, '$.\"$requiredJsonKey\"') AS UNSIGNED) END)";

        return "(CASE "
            . "WHEN $latestStatus = 'approved' THEN 0 "
            . "WHEN $latestStatus = 'rejected' THEN 1 "
            . "WHEN $latestSent IS NOT NULL THEN 2 "
            . "WHEN $latestId IS NULL AND $requiredExpr = 1 THEN 3 "
            . "ELSE 4 END)";
    }

    protected function normalizeIdArray($value): array
    {
        if (!is_array($value) || empty($value)) {
            return [];
        }
        // If associative (keyed id=>1), return keys; if numeric (list of ids), return as-is
        $isNumericList = array_keys($value) === range(0, count($value) - 1);
        return $isNumericList ? array_values($value) : array_keys($value);
    }

}
