<?php

namespace App\Http\Livewire\Designs;

use App\Models\Design;
use App\Models\Customer;
use App\Models\Suppliers;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Title;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

#[Title('Designs')]
class DesignsList extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    #[Validate([
        'customer' => ['nullable', 'exists:customers,id'],
        'factory'  => ['nullable', 'exists:suppliers,id'],
        'search'   => ['nullable', 'string', 'max:30'],
        'paginate' => ['required', 'numeric', 'min:10', 'max:1000'],
    ])]
    public $customer, $factory, $search, $paginate;

    protected function filterKey(): string
    {
        return 'designsfilters';
    }

    protected function columnsVersion(): float
    {
        return 1.2;
    }

    protected function filters(): array
    {
        return ['customer', 'factory', 'search', 'columns', 'paginate'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    #[On('refresh-columns')]
    public function refresh()
    {
        $this->loadFilterSettings();
    }


    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['customer', 'factory', 'search', 'paginate'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    public $columns = [
        'ID' => true,
        'Description' => true,
        'Customer' => true,
        'Factory' => true,
        'Gauge' => true,
        'Yarn' => true,
        'Date of Issue' => true,
    ];

    #[Computed()]
    public function customers(){
        return Customer::allCached();
    }

    #[Computed()]
    public function factories(){
        return Suppliers::allCached();
    }

    #[Computed]
    public function designs()
    {
        return Design::select('id', 'description', 'customers_id', 'factory_id', 'gauge_id', 'yarn', 'issued')
            ->with('customers:name,id', 'gauges:id,gauge', 'factories:id,name')
            ->search($this->search)
            ->when(!empty($this->customer), function ($query) {
                $query->where('customers_id', $this->customer);
            })
            ->when(!empty($this->factory), function ($query) {
                $query->where('factory_id', $this->factory);
            })
            ->orderBy('created_at', 'desc')
            ->paginate($this->paginate);
    }


    #[On('refresh')]
    public function render()
    {
        Gate::authorize('design:read');

        return view('livewire.designs.designs-list');
    }

    public function delete($id)
    {
        $design = Design::find($id);

        if (!$design) {
            session()->flash('message', 'Design not found.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }

        Gate::authorize('design:delete');
        $design->delete();
        session()->flash('message', 'Design deleted successfully.');
        session()->flash('alert-class', 'alert-success');
        $this->dispatch('refresh-designs');
    }

    public function go(){}

    public function allCols(){
        $this->reset('columns');
        $this->updatedColumns();
    }

    protected function getDefaultColumns(): array
    {
        return $this->columns;
    }
}

