<?php

namespace App\Http\Livewire\Designs;

use App\Models\Gauge;
use App\Models\Design;
use App\Models\Styles;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use Livewire\Attributes\On;
use App\Models\StyleVersions;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\BaseComponent;

class DesignQuickAdd extends BaseComponent
{
    #[Validate([
        'department_id' => ['required', 'exists:departments,id'],
        'season_id' => ['required', 'exists:seasons,id'],
        'customer_id' => ['required', 'exists:customers,id'],
        'category' => ['required', 'in:ladies,mens,childrens,accessories'],
        'description' => ['required', 'string', 'max:255'],
        'gauge_id' => ['required', 'exists:gauges,id'],
        'factory_id' => ['required', 'exists:suppliers,id'],
        'issued' => ['required', 'date'],
        'yarn' => ['nullable', 'string', 'max:100'],
    ])]
    public $department_id = '';
    public $season_id = '';
    public $customer_id = '';
    public $category = 'ladies';
    public $description = '';
    public $gauge_id = '';
    public $factory_id = '';
    public $issued;
    public $yarn = '';

    public function mount()
    {
        $this->issued = now()->format('Y-m-d');
    }

    #[Computed]
    public function departments()
    {
        return Departments::allCached();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::allCached();
    }

    #[Computed]
    public function customers()
    {
        return Customer::allCached();
    }

    #[Computed]
    public function gauges()
    {
        return Gauge::allCached();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::allCached();
    }

    #[On('close-modal')]
    public function clearForm()
    {
        $this->reset(['department_id', 'season_id', 'customer_id', 'description', 'gauge_id', 'factory_id', 'yarn']);
        $this->category = 'ladies';
        $this->issued = now()->format('Y-m-d');
        $this->resetValidation();
    }

    public function save()
    {
        Gate::authorize('design:create');

        $this->validate();

        // Create the Design
        $design = Design::create([
            'description' => $this->description,
            'customers_id' => $this->customer_id,
            'factory_id' => $this->factory_id,
            'gauge_id' => $this->gauge_id,
            'yarn' => $this->yarn ?? '',
            'issued' => $this->issued,
        ]);

        // Create the Style
        $style = Styles::create([
            'designs_id' => $design->id,
            'departments_id' => $this->department_id,
            'seasons_id' => $this->season_id,
            'customers_id' => $this->customer_id,
            'category' => $this->category,
        ]);

        // Create the Style Version (A)
        $styleVersion = StyleVersions::create([
            'styles_id' => $style->id,
            'factory_id' => $this->factory_id,
            'name' => 'A',
            'gauge_id' => $this->gauge_id,
            'issued' => $this->issued,
        ]);

        // Create the Colourway (TBC)
        Colourways::create([
            'style_versions_id' => $styleVersion->id,
            'name' => 'TBC',
            'colour_type' => 'solid',
        ]);

        session()->flash('message', 'Design, Style, Version and Colourway created successfully!');
        session()->flash('alert-class', 'alert-success');

        $this->dispatch('close-modal');
        $this->dispatch('refresh');
        $this->clearForm();
    }

    public function render()
    {
        Gate::authorize('design:read');
        return view('livewire.designs.design-quick-add');
    }
}

