<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Seasons;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class UnconfirmedPrices extends Component
{
    public $selectedSeason = null;
    public $selectedOrderType = 'all';
    public $pricesData = [];
    public $isLoading = true;
    
    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    protected $listeners = [
        'dashboardFiltersUpdated' => 'updateFilters',
        'refreshDashboard' => 'refreshData'
    ];

    public function mount()
    {
        // Get global dashboard filter values
        $this->selectedSeason = auth()->user()->setting('dashboard_season_filter');
        // Order type is always 'wholesale' for this dashboard - commission has its own
        $this->selectedOrderType = 'wholesale';
        
        $this->loadPricesData();
        $this->isLoading = false;
    }

    public function updateFilters($filters)
    {
        $this->selectedSeason = $filters['season'];
        // Order type stays hardcoded to 'wholesale'
        $this->isLoading = true;
        $this->loadPricesData();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadPricesData();
        $this->isLoading = false;
    }

    public function loadPricesData()
    {
        if (!$this->selectedSeason) {
            $this->pricesData = [];
            return;
        }

        // Create cache key
        $cacheKey = "dashboard_unconfirmed_prices_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        
        // Try to get from cache first
        $cachedData = Cache::get($cacheKey);
        if ($cachedData) {
            $this->pricesData = $cachedData;
            return;
        }

        // Get style versions that have orders in the selected season
        $styleVersionsWithOrdersQuery = DB::table('style_versions')
            ->join('colourways', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('customer_order_lines', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
            ->whereNull('style_versions.deleted_at')
            ->whereNull('colourways.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->whereNull('styles.deleted_at')
            ->where('colourways.cancelled', false)
            ->where('customer_order_lines.cancelled', false)
            ->where('customer_orders.cancelled', false)
            ->where('styles.cancelled', false)
            ->where('styles.seasons_id', $this->selectedSeason);

        // Filter by order type if not 'all'
        if ($this->selectedOrderType !== 'all') {
            $styleVersionsWithOrdersQuery->where('customer_orders.order_type', $this->selectedOrderType);
        }

        $styleVersionsWithOrders = $styleVersionsWithOrdersQuery->distinct()->pluck('style_versions.id');

        if ($styleVersionsWithOrders->isEmpty()) {
            $this->pricesData = [];
            Cache::put($cacheKey, $this->pricesData, now()->addHours(24));
            return;
        }

        // Query to get unconfirmed prices for only those style versions that have orders
        $query = DB::table('prices')
            ->join('style_versions', 'prices.style_versions_id', '=', 'style_versions.id')
            ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
            ->join('customers', 'styles.customers_id', '=', 'customers.id')
            ->whereNull('prices.deleted_at')
            ->whereNull('style_versions.deleted_at')
            ->whereNull('styles.deleted_at')
            ->where('styles.cancelled', false)
            ->where('styles.seasons_id', $this->selectedSeason)
            ->whereIn('style_versions.id', $styleVersionsWithOrders)
            ->where(function($q) {
                $q->where('prices.quote_status', '!=', 'confirmed')
                  ->orWhereNull('prices.quote_status')
                  ->orWhere('prices.quote_status', '');
            });

        $results = $query->select([
                'customers.id as customer_id',
                'customers.name as customer_name',
                DB::raw('COUNT(DISTINCT prices.id) as unconfirmed_count')
            ])
            ->groupBy('customers.id', 'customers.name')
            ->having('unconfirmed_count', '>', 0)
            ->orderBy('unconfirmed_count', 'desc')
            ->orderBy('customers.name', 'asc')
            ->get();

        $this->pricesData = $results->toArray();
        
        // Cache the results for 24 hours
        Cache::put($cacheKey, $this->pricesData, now()->addHours(24));
    }

    public function viewPrices($customerId)
    {
        // Update the user's pricing filter settings to show unconfirmed prices for this customer
        auth()->user()->settings([
            'whpricingfilters' => [
                'season' => $this->selectedSeason,
                'customer' => $customerId,
                'not_quoted' => 1,
                'cancelled' => 0,
                'department' => null,
                'factory' => null,
                'coo' => null,
                'category' => null,
                'search' => null,
            ]
        ]);
        
        // Redirect to pricing sheet
        return redirect()->route('pricing');
    }

    public function render()
    {
        return view('livewire.dashboard.unconfirmed-prices');
    }
}

