<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\ShipmentLine;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class ShipmentsLeavingThisWeek extends Component
{
    public $selectedOrderType = 'all';
    public $shipmentsData = [];
    public $isLoading = true;
    public $totalQuantity = 0;
    public $totalValue = 0;
    public $totalSaleValue = 0;
    
    private const CACHE_DURATION = 60; // 1 hour in minutes

    protected $listeners = [
        'dashboardFiltersUpdated' => 'updateFilters',
        'refreshDashboard' => 'refreshData'
    ];

    public function mount()
    {
        // Order type is always 'wholesale' for this dashboard - commission has its own
        $this->selectedOrderType = 'wholesale';
        
        $this->loadShipmentsData();
        $this->isLoading = false;
    }

    public function updateFilters($filters)
    {
        // Order type stays hardcoded to 'wholesale' - this widget doesn't use season filter
        $this->isLoading = true;
        $this->loadShipmentsData();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadShipmentsData();
        $this->isLoading = false;
    }

    public function loadShipmentsData()
    {
        // Calculate this week's date range (Monday to Sunday)
        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();

        // Create cache key (with version to invalidate old cache)
        $cacheKey = "dashboard_shipments_leaving_week_v6_{$startOfWeek->format('Y-m-d')}_order_type_{$this->selectedOrderType}";
        
        // Try to get from cache first
        $cachedData = Cache::get($cacheKey);
        if ($cachedData) {
            $this->shipmentsData = $cachedData['shipments'];
            $this->totalQuantity = $cachedData['totalQuantity'];
            $this->totalValue = $cachedData['totalValue'];
            $this->totalSaleValue = $cachedData['totalSaleValue'] ?? 0;
            return;
        }

        // Query shipment lines leaving this week (pending only) - grouped by style
        $query = DB::table('shipment_lines')
            ->select([
                'styles.id as style_id',
                'designs.id as design_id',
                'designs.description as design_name',
                'customers.name as customer_name',
                'seasons.description as season',
                'factory.name as factory_name',
                DB::raw('MIN(shipment_lines.exfty) as earliest_exfty'),
                DB::raw('MAX(shipment_lines.exfty) as latest_exfty'),
                DB::raw('COUNT(DISTINCT colourways.id) as colourway_count'),
                DB::raw('GROUP_CONCAT(DISTINCT colourways.name ORDER BY colourways.name SEPARATOR ", ") as colourways'),
                DB::raw('GROUP_CONCAT(DISTINCT shipment_lines.id ORDER BY shipment_lines.id SEPARATOR ",") as shipment_line_ids'),
                DB::raw('SUM((SELECT SUM(sls.qty) FROM shipment_line_sizes sls WHERE sls.shipment_line_id = shipment_lines.id AND sls.deleted_at IS NULL)) as total_qty'),
                DB::raw('SUM(
                    CASE 
                        WHEN customer_orders.order_type = "wholesale" THEN
                            COALESCE(
                                (SELECT JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, "$.quote"))
                                 FROM total_cache tc
                                 WHERE tc.entity_id = shipment_lines.id
                                 AND tc.entity_type = "shipment_line"
                                 AND tc.cache_key = "prices"
                                 AND tc.fresh_at IS NOT NULL), 0)
                        ELSE
                            (SELECT SUM(
                                CASE 
                                    WHEN COALESCE(sls.shipped_qty, 0) > 0 THEN sls.shipped_qty
                                    ELSE COALESCE(sls.qty, 0)
                                END * COALESCE(colq.price, 0)
                             )
                             FROM shipment_line_sizes sls
                             JOIN customer_order_line_quantities colq 
                                ON colq.customer_order_lines_id = customer_order_lines.id 
                                AND colq.sizes_id = sls.sizes_id
                             WHERE sls.shipment_line_id = shipment_lines.id 
                             AND sls.deleted_at IS NULL)
                    END
                ) as total_sale_value'),
                DB::raw('MAX(CASE WHEN shipment_lines.flag_issue = 1 THEN 1 ELSE 0 END) as has_flagged_issue'),
                DB::raw('MAX(CASE 
                    WHEN (
                        -- Row is NOT good if samples are rejected
                        EXISTS (
                            SELECT 1 FROM samples 
                            WHERE samples.colourways_id = colourways.id 
                            AND samples.sample_types_id IN (3, 7)
                            AND samples.deleted_at IS NULL
                            AND samples.status = "Rejected"
                        )
                    ) OR (
                        -- Row is NOT good if required samples are pending/required
                        (COALESCE(shipment_lines.shipment_status, 
                            (SELECT samples.status FROM samples 
                             WHERE samples.colourways_id = colourways.id 
                             AND samples.sample_types_id = 7 
                             AND samples.deleted_at IS NULL
                             ORDER BY created_at DESC LIMIT 1)
                        ) != "Approved") 
                        AND NOT (
                            -- Exception: Both samples are "Not Required"
                            COALESCE(
                                (SELECT samples.status FROM samples 
                                 WHERE samples.colourways_id = colourways.id 
                                 AND samples.sample_types_id = 7 
                                 AND samples.deleted_at IS NULL
                                 ORDER BY created_at DESC LIMIT 1),
                                CASE 
                                    WHEN JSON_EXTRACT(customers.samples_required, "$[7]") = false THEN "Not Required"
                                    ELSE "Required"
                                END
                            ) = "Not Required"
                            AND
                            COALESCE(
                                (SELECT samples.status FROM samples 
                                 WHERE samples.colourways_id = colourways.id 
                                 AND samples.sample_types_id = 3 
                                 AND samples.deleted_at IS NULL
                                 ORDER BY created_at DESC LIMIT 1),
                                CASE 
                                    WHEN JSON_EXTRACT(customers.samples_required, "$[3]") = false THEN "Not Required"
                                    ELSE "Required"
                                END
                            ) = "Not Required"
                        )
                    ) THEN 1 ELSE 0 END) as has_issue'),
            ])
            ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->join('customers', 'customer_orders.customers_id', '=', 'customers.id')
            ->join('seasons', 'customer_orders.seasons_id', '=', 'seasons.id')
            ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
            ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
            ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
            ->join('designs', 'styles.designs_id', '=', 'designs.id')
            ->join('suppliers as factory', 'style_versions.factory_id', '=', 'factory.id')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->whereNull('shipment_lines.deleted_at')
            ->where('customer_order_lines.cancelled', 0)
            ->where('customer_orders.cancelled', 0)
            ->where('shipment_lines.complete', 0)
            ->whereBetween('shipment_lines.exfty', [$startOfWeek->format('Y-m-d'), $endOfWeek->format('Y-m-d')]);

        // Filter by order type if not 'all'
        if ($this->selectedOrderType !== 'all') {
            $query->where('customer_orders.order_type', $this->selectedOrderType);
        }

        $results = $query
            ->groupBy('styles.id', 'designs.id', 'designs.description', 'customers.name', 'seasons.description', 'factory.name')
            ->orderBy('earliest_exfty', 'asc')
            ->orderBy('customers.name', 'asc')
            ->get();

        // Calculate totals and ensure issue flags have defaults
        $totalQty = 0;
        $totalSale = 0;

        foreach ($results as $shipment) {
            $totalQty += $shipment->total_qty ?? 0;
            $totalSale += $shipment->total_sale_value ?? 0;
            
            // Ensure issue flags are set with defaults
            $shipment->has_flagged_issue = $shipment->has_flagged_issue ?? 0;
            $shipment->has_issue = $shipment->has_issue ?? 0;
        }

        $this->shipmentsData = $results;
        $this->totalQuantity = $totalQty;
        $this->totalValue = 0;
        $this->totalSaleValue = $totalSale;
        
        // Cache the results
        Cache::put($cacheKey, [
            'shipments' => $this->shipmentsData,
            'totalQuantity' => $this->totalQuantity,
            'totalValue' => $this->totalValue,
            'totalSaleValue' => $this->totalSaleValue
        ], now()->addMinutes(self::CACHE_DURATION));
    }

    public function viewStyle($styleId, $shipmentLineIds)
    {
        // Set the search filter to all shipment line IDs (space-separated)
        $filterKey = $this->selectedOrderType === 'commission' 
            ? 'commissionshipmentschedulefilters' 
            : 'shipmentschedulefilters';
        
        // Get existing filters
        $filters = auth()->user()->setting($filterKey, []);
        
        // Convert comma-separated IDs to space-separated for search
        $searchIds = str_replace(',', ' ', $shipmentLineIds);
        $filters['search'] = $searchIds;
        
        // Save the updated filters
        auth()->user()->settings([$filterKey => $filters]);
        
        // Redirect to shipment schedule
        if ($this->selectedOrderType === 'commission') {
            return redirect()->route('commissionshipmentschedule');
        } else {
            return redirect()->route('shipmentschedule');
        }
    }

    public function render()
    {
        return view('livewire.dashboard.shipments-leaving-this-week');
    }
}

