<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Samples;
use App\Models\ShipmentLine;
use App\Models\ShipmentLineSizes;
use App\Models\CustomerOrderLineQuantities;
use Livewire\Component;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class DashboardMetrics extends Component
{
    public $totalStylesInSeason = 0;
    public $totalStylesInOrders = 0;
    public $totalQmtsShipped = 0;
    public $totalGmtsLeftToShip = 0;
    public $totalValueShipped = 0;
    public $totalValueOrdered = 0;
    public $totalSamplesSent = 0;
    public $totalInvoicesSent = 0;
    public $averageMargin = 0;
    public $isLoading = true;
    public $selectedSeason = null;
    public $selectedOrderType = 'wholesale'; // Hardcoded - commission has its own dashboard

    private const CACHE_DURATION = 24 * 60; // 24 hours in minutes

    protected $listeners = [
        'dashboardFiltersUpdated' => 'updateFilters',
        'refreshDashboard' => 'refreshData'
    ];

    public function mount()
    {
        // Get global dashboard filter values
        $this->selectedSeason = auth()->user()->setting('dashboard_season_filter');
        // Order type is always 'wholesale' for this dashboard
        
        $this->loadMetrics();
    }

    public function updateFilters($filters)
    {
        $this->selectedSeason = $filters['season'];
        // Order type stays hardcoded to 'wholesale'
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadMetrics();
        $this->isLoading = false;
    }

    public function loadMetrics()
    {
        // Load all metrics with caching
        $this->totalStylesInSeason = $this->getTotalStylesInSeason();
        $this->totalStylesInOrders = $this->getTotalStylesInOrders();
        $this->totalQmtsShipped = $this->getTotalQmtsShipped();
        $this->totalGmtsLeftToShip = $this->getTotalGmtsLeftToShip();
        $this->totalValueShipped = $this->getTotalValueShipped();
        $this->totalValueOrdered = $this->getTotalValueOrdered();
        $this->averageMargin = $this->getAverageMargin();
        $this->totalSamplesSent = $this->getTotalSamplesSent();
        $this->totalInvoicesSent = $this->getTotalInvoicesSent();
        
        $this->isLoading = false;
    }

    private function getTotalStylesInSeason()
    {
        $cacheKey = "dashboard_total_styles_season_{$this->selectedSeason}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            if (!$this->selectedSeason) {
                return 0;
            }

            return DB::table('styles')
                ->where('seasons_id', $this->selectedSeason)
                ->whereNull('deleted_at')
                ->where('cancelled', false)
                ->count();
        });
    }

    private function getTotalStylesInOrders()
    {
        $cacheKey = "dashboard_total_styles_in_orders_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            if (!$this->selectedSeason) {
                return 0;
            }

            $query = DB::table('styles')
                ->join('style_versions', 'styles.id', '=', 'style_versions.styles_id')
                ->join('colourways', 'style_versions.id', '=', 'colourways.style_versions_id')
                ->join('customer_order_lines', 'colourways.id', '=', 'customer_order_lines.colourways_id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->where('styles.seasons_id', $this->selectedSeason)
                ->whereNull('styles.deleted_at')
                ->where('styles.cancelled', false)
                ->whereNull('style_versions.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false)
                ->whereNull('customer_order_lines.deleted_at')
                ->where('customer_order_lines.cancelled', false)
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false);

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $query->where('customer_orders.order_type', $this->selectedOrderType);
            }

            return $query->distinct('styles.id')->count('styles.id');
        });
    }

    private function getTotalQmtsShipped()
    {
        $cacheKey = "dashboard_total_qmts_shipped_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $query->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $query->where('customer_orders.order_type', $this->selectedOrderType);
            }

            return $query->sum('shipment_line_sizes.shipped_qty');
        });
    }

    private function getTotalGmtsLeftToShip()
    {
        $cacheKey = "dashboard_total_gmts_left_to_ship_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Get total ordered quantities (only from non-cancelled, non-deleted orders)
            $totalOrderedQuery = DB::table('customer_order_line_quantities')
                ->join('customer_order_lines', 'customer_order_line_quantities.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNull('customer_order_line_quantities.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $totalOrderedQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $totalOrderedQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $totalOrdered = $totalOrderedQuery->sum('customer_order_line_quantities.qty');

            // Get total shipped quantities (only from non-cancelled, non-deleted orders)
            $totalShippedQuery = DB::table('shipment_line_sizes')
                ->join('shipment_lines', 'shipment_line_sizes.shipment_line_id', '=', 'shipment_lines.id')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_line_sizes.shipped_qty')
                ->where('shipment_line_sizes.shipped_qty', '>', 0)
                ->whereNull('shipment_line_sizes.deleted_at')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $totalShippedQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $totalShippedQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $totalShipped = $totalShippedQuery->sum('shipment_line_sizes.shipped_qty');

            return max(0, $totalOrdered - $totalShipped);
        });
    }

    private function getTotalValueShipped()
    {
        $cacheKey = "dashboard_total_value_shipped_season_{$this->selectedSeason}_order_type3_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('shipment_lines as sl')
                ->join('customer_order_lines as col', 'sl.customer_order_lines_id', '=', 'col.id')
                ->join('customer_orders as co', 'col.customer_orders_id', '=', 'co.id')
                ->join('colourways as cw', 'col.colourways_id', '=', 'cw.id')
                ->join('style_versions as sv', 'cw.style_versions_id', '=', 'sv.id')
                ->join('styles as s', 'sv.styles_id', '=', 's.id')
                ->join('total_cache as tc', function($join) {
                    $join->on('tc.entity_id', '=', 'sl.id')
                         ->where('tc.entity_type', '=', 'shipment_line')
                         ->where('tc.cache_key', '=', 'prices');
                })
                ->whereNull('sl.deleted_at')
                ->whereNull('col.deleted_at')
                ->whereNull('co.deleted_at')
                ->whereNull('cw.deleted_at')
                ->whereNull('sv.deleted_at')
                ->whereNull('s.deleted_at')
                ->where('co.cancelled', false)
                ->where('col.cancelled', false)
                ->where('cw.cancelled', false)
                ->where('s.cancelled', false);

            if ($this->selectedSeason) {
                $query->where('co.seasons_id', $this->selectedSeason);
            }

            if ($this->selectedOrderType !== 'all') {
                $query->where('co.order_type', $this->selectedOrderType);
            }

            $sumExpr = DB::raw("SUM(CASE WHEN sl.complete = 1 THEN CAST(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.quote_base')) AS DECIMAL(18,2)) ELSE 0 END) as total");

            return (float) ($query->select($sumExpr)->value('total') ?? 0);
        });
    }

    private function getTotalValueOrdered()
    {
        $cacheKey = "dashboard_total_value_ordered_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('shipment_lines as sl')
                ->join('customer_order_lines as col', 'sl.customer_order_lines_id', '=', 'col.id')
                ->join('customer_orders as co', 'col.customer_orders_id', '=', 'co.id')
                ->join('colourways as cw', 'col.colourways_id', '=', 'cw.id')
                ->join('style_versions as sv', 'cw.style_versions_id', '=', 'sv.id')
                ->join('styles as s', 'sv.styles_id', '=', 's.id')
                ->join('total_cache as tc', function($join) {
                    $join->on('tc.entity_id', '=', 'sl.id')
                         ->where('tc.entity_type', '=', 'shipment_line')
                         ->where('tc.cache_key', '=', 'prices');
                })
                ->whereNull('sl.deleted_at')
                ->whereNull('col.deleted_at')
                ->whereNull('co.deleted_at')
                ->whereNull('cw.deleted_at')
                ->whereNull('sv.deleted_at')
                ->whereNull('s.deleted_at')
                ->where('co.cancelled', false)
                ->where('col.cancelled', false)
                ->where('cw.cancelled', false)
                ->where('s.cancelled', false);

            if ($this->selectedSeason) {
                $query->where('co.seasons_id', $this->selectedSeason);
            }

            if ($this->selectedOrderType !== 'all') {
                $query->where('co.order_type', $this->selectedOrderType);
            }

            $sumExpr = DB::raw("SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, '$.quote_base')) AS DECIMAL(18,2))) as total");

            return (float) ($query->select($sumExpr)->value('total') ?? 0);
        });
    }

    private function getTotalSamplesSent()
    {
        $cacheKey = "dashboard_total_samples_sent_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            $query = DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
                ->whereNotNull('samples.date_sent')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $query->where('styles.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - samples don't have direct order type, so we'll skip this filter for samples
            // Samples are associated with styles, not customer orders directly

            return $query->count();
        });
    }

    private function getTotalInvoicesSent()
    {
        $cacheKey = "dashboard_total_invoices_sent_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            // Get distinct rt_invoice from samples (only non-cancelled, non-deleted)
            $samplesQuery = DB::table('samples')
                ->join('colourways', 'samples.colourways_id', '=', 'colourways.id')
                ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
                ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
                ->whereNotNull('samples.rt_invoice')
                ->where('samples.rt_invoice', '!=', '')
                ->whereNull('samples.deleted_at')
                ->whereNull('colourways.deleted_at')
                ->where('colourways.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $samplesQuery->where('styles.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter - samples don't have direct order type, so we'll skip this filter for samples
            // Samples are associated with styles, not customer orders directly

            $samplesInvoices = $samplesQuery->distinct()->count('samples.rt_invoice');

            // Get distinct rt_invoice from shipment_lines (only non-cancelled, non-deleted)
            $shipmentQuery = DB::table('shipment_lines')
                ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
                ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
                ->whereNotNull('shipment_lines.rt_invoice')
                ->where('shipment_lines.rt_invoice', '!=', '')
                ->whereNull('shipment_lines.deleted_at')
                ->whereNull('customer_order_lines.deleted_at')
                ->whereNull('customer_orders.deleted_at')
                ->where('customer_orders.cancelled', false)
                ->where('customer_order_lines.cancelled', false);

            // Apply season filter
            if ($this->selectedSeason) {
                $shipmentQuery->where('customer_orders.seasons_id', $this->selectedSeason);
            }

            // Apply order type filter
            if ($this->selectedOrderType !== 'all') {
                $shipmentQuery->where('customer_orders.order_type', $this->selectedOrderType);
            }

            $shipmentInvoices = $shipmentQuery->distinct()->count('shipment_lines.rt_invoice');

            return $samplesInvoices + $shipmentInvoices;
        });
    }

    private function getAverageMargin()
    {
        $cacheKey = "dashboard_average_margin_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        return Cache::remember($cacheKey, self::CACHE_DURATION, function () {
            if (!$this->selectedSeason) {
                return 0;
            }

            // Only calculate for wholesale orders (commission doesn't use margins)
            if ($this->selectedOrderType !== 'all' && $this->selectedOrderType !== 'wholesale') {
                return 0;
            }

            $query = DB::table('shipment_lines as sl')
                ->join('customer_order_lines as col', 'sl.customer_order_lines_id', '=', 'col.id')
                ->join('customer_orders as co', 'col.customer_orders_id', '=', 'co.id')
                ->join('colourways as cw', 'col.colourways_id', '=', 'cw.id')
                ->join('style_versions as sv', 'cw.style_versions_id', '=', 'sv.id')
                ->join('styles as s', 'sv.styles_id', '=', 's.id')
                ->join('total_cache as tc', function($join) {
                    $join->on('tc.entity_id', '=', 'sl.id')
                         ->where('tc.entity_type', '=', 'shipment_line')
                         ->where('tc.cache_key', '=', 'prices');
                })
                ->whereNull('sl.deleted_at')
                ->whereNull('col.deleted_at')
                ->whereNull('co.deleted_at')
                ->whereNull('cw.deleted_at')
                ->whereNull('sv.deleted_at')
                ->whereNull('s.deleted_at')
                ->where('co.cancelled', false)
                ->where('col.cancelled', false)
                ->where('cw.cancelled', false)
                ->where('s.cancelled', false)
                ->where('co.seasons_id', $this->selectedSeason)
                ->where('co.order_type', 'wholesale'); // Only wholesale orders

            // Only include lines with confirmed Subtotal and Quote prices
            $query->whereRaw('JSON_EXTRACT(tc.cached_data, "$.subtotal") IS NOT NULL')
                  ->whereRaw('JSON_EXTRACT(tc.cached_data, "$.subtotal") > 0')
                  ->whereRaw('JSON_EXTRACT(tc.cached_data, "$.quote") IS NOT NULL')
                  ->whereRaw('JSON_EXTRACT(tc.cached_data, "$.quote") > 0');

            // Calculate average margin: ((quote - subtotal) / quote) * 100
            $avgMargin = $query->selectRaw('
                AVG(
                    (CAST(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, "$.quote")) AS DECIMAL(18,2)) - 
                     CAST(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, "$.subtotal")) AS DECIMAL(18,2))) / 
                    CAST(JSON_UNQUOTE(JSON_EXTRACT(tc.cached_data, "$.quote")) AS DECIMAL(18,2)) * 100
                ) as avg_margin
            ')->value('avg_margin');

            return (float) ($avgMargin ?? 0);
        });
    }

    public function render()
    {
        return view('livewire.dashboard.dashboard-metrics');
    }
}
