<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Departments;
use App\Models\Seasons;
use App\Models\ShipmentLine;
use Livewire\Component;
use App\Helper\Conversions;
use Illuminate\Support\Facades\DB;

class CustomerSalesByDepartment extends Component
{
    private const CACHE_VERSION = 4;
    public $selectedSeason = null;
    public $seasons = [];
    public $salesData = [];
    public $totalSales = 0;
    public $selectedOrderType = 'all';
    public $orderTypes = [
        'all' => 'All Orders',
        'wholesale' => 'Wholesale',
        'commission' => 'Commission'
    ];
    public $isLoading = true;

    public function mount()
    {
        $this->seasons = Seasons::orderBy('created_at', 'desc')->get();
        
        // Get global dashboard filter values
        $this->selectedSeason = auth()->user()->setting('dashboard_season_filter');
        // Order type is always 'wholesale' for this dashboard - commission has its own
        $this->selectedOrderType = 'wholesale';
        
        // If no global filter is set, use defaults
        if (!$this->selectedSeason || !$this->seasons->contains('id', $this->selectedSeason)) {
            $this->selectedSeason = $this->seasons->skip(3)->first()?->id ?? $this->seasons->first()?->id;
        }
        
        if (!array_key_exists($this->selectedOrderType, $this->orderTypes)) {
            $this->selectedOrderType = 'all';
        }
        
        // Load data after mount
        $this->loadSalesData();
        $this->isLoading = false;
    }

    protected $listeners = [
        'dashboardFiltersUpdated' => 'updateFilters',
        'refreshDashboard' => 'refreshData'
    ];

    public function updateFilters($filters)
    {
        $this->selectedSeason = $filters['season'];
        // Order type stays hardcoded to 'wholesale'
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function refreshData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadData()
    {
        $this->isLoading = true;
        $this->loadSalesData();
        $this->isLoading = false;
    }

    public function loadSalesData()
    {
        if (!$this->selectedSeason) {
            $this->salesData = [];
            $this->totalSales = 0;
            return;
        }

        // Create cache key (include version to invalidate old caches)
        $cacheKey = "customer_sales_dashboard_v" . self::CACHE_VERSION . "_season_{$this->selectedSeason}_order_type_{$this->selectedOrderType}";
        
        // Try to get from cache first
        $cachedData = cache()->get($cacheKey);
        if ($cachedData) {
            $this->salesData = $cachedData['salesData'];
            $this->totalSales = $cachedData['totalSales'];
            return;
        }

        // Use total cache for accurate values
        $query = DB::table('shipment_lines')
            ->join('customer_order_lines', 'shipment_lines.customer_order_lines_id', '=', 'customer_order_lines.id')
            ->join('customer_orders', 'customer_order_lines.customer_orders_id', '=', 'customer_orders.id')
            ->join('customers', 'customer_orders.customers_id', '=', 'customers.id')
            ->join('departments', 'customer_orders.departments_id', '=', 'departments.id')
            ->join('colourways', 'customer_order_lines.colourways_id', '=', 'colourways.id')
            ->join('style_versions', 'colourways.style_versions_id', '=', 'style_versions.id')
            ->join('styles', 'style_versions.styles_id', '=', 'styles.id')
            ->join('total_cache', function($join) {
                $join->on('total_cache.entity_id', '=', 'shipment_lines.id')
                     ->where('total_cache.entity_type', '=', 'shipment_line')
                     ->where('total_cache.cache_key', '=', 'prices');
            })
            ->whereNull('shipment_lines.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->whereNull('colourways.deleted_at')
            ->whereNull('style_versions.deleted_at')
            ->whereNull('styles.deleted_at')
            ->where('customer_orders.cancelled', false)
            ->where('customer_order_lines.cancelled', false)
            ->where('colourways.cancelled', false)
            ->where('styles.cancelled', false)
            ->where('customer_orders.seasons_id', $this->selectedSeason);

        // Filter by order type if not 'all'
        if ($this->selectedOrderType !== 'all') {
            $query->where('customer_orders.order_type', $this->selectedOrderType);
        }

        $salesData = $query->select([
                'customers.id as customer_id',
                'customers.name as customer_name',
                'departments.description as department_name',
                DB::raw('SUM(CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote_base")) AS DECIMAL(18,2))) as total_value'),
                DB::raw('COUNT(shipment_lines.id) as total_drops'),
                DB::raw('SUM(CASE WHEN shipment_lines.complete = 1 THEN 1 ELSE 0 END) as shipped_drops'),
                DB::raw('SUM(CASE WHEN shipment_lines.complete = 1 THEN CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote_base")) AS DECIMAL(18,2)) ELSE 0 END) as shipped_value'),
                DB::raw('SUM(CASE WHEN shipment_lines.complete = 0 THEN CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote_base")) AS DECIMAL(18,2)) ELSE 0 END) as remaining_value'),
                DB::raw('AVG(
                    CASE 
                        WHEN JSON_EXTRACT(total_cache.cached_data, "$.subtotal") IS NOT NULL 
                        AND JSON_EXTRACT(total_cache.cached_data, "$.subtotal") > 0
                        AND JSON_EXTRACT(total_cache.cached_data, "$.quote") IS NOT NULL 
                        AND JSON_EXTRACT(total_cache.cached_data, "$.quote") > 0
                        THEN (CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote")) AS DECIMAL(18,2)) - 
                              CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.subtotal")) AS DECIMAL(18,2))) / 
                             CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote")) AS DECIMAL(18,2)) * 100
                        ELSE NULL 
                    END
                ) as avg_margin')
            ])
            ->groupBy('customers.id', 'customers.name', 'departments.id', 'departments.description')
            ->orderBy('customers.name', 'asc')
            ->get();

        // Debug: Log the total customers found
        \Log::info("Dashboard: Found " . $salesData->count() . " records for season {$this->selectedSeason}");

        $customerSales = [];

        foreach ($salesData as $row) {
            $customerId = $row->customer_id;
            $customerName = $row->customer_name;
            $departmentName = $row->department_name;

            // If customer already exists, add to their totals
            if (isset($customerSales[$customerId])) {
                $customerSales[$customerId]['shipped_sales'] += (float) ($row->shipped_value ?? 0);
                $customerSales[$customerId]['remaining_sales'] += (float) ($row->remaining_value ?? 0);
                $customerSales[$customerId]['total_drops'] += (int) ($row->total_drops ?? 0);
                $customerSales[$customerId]['shipped_drops'] += (int) ($row->shipped_drops ?? 0);
                // For margin, we'll recalculate the average after all data is processed
            } else {
                $customerSales[$customerId] = [
                    'name' => $customerName,
                    'department' => $departmentName,
                    'shipped_sales' => (float) ($row->shipped_value ?? 0),
                    'remaining_sales' => (float) ($row->remaining_value ?? 0),
                    'total_drops' => (int) ($row->total_drops ?? 0),
                    'shipped_drops' => (int) ($row->shipped_drops ?? 0),
                    'avg_margin' => (float) ($row->avg_margin ?? 0)
                ];
            }
        }

        // Calculate shipping percentage for each customer based on drops
        foreach ($customerSales as $customerId => &$data) {
            if ($data['total_drops'] > 0) {
                $data['shipping_percentage'] = ($data['shipped_drops'] / $data['total_drops']) * 100;
            } else {
                $data['shipping_percentage'] = 0;
            }
        }

        // Sort alphabetically by customer name
        uasort($customerSales, function ($a, $b) {
            return strcasecmp($a['name'], $b['name']);
        });

        // Limit to top 50 customers by total value
        $customerSales = array_slice($customerSales, 0, 50, true);

        $this->salesData = $customerSales;
        $this->totalSales = array_sum(array_column($customerSales, 'shipped_sales'));
        
        
        // Calculate additional stats for dashboard
        $pendingSales = array_sum(array_column($customerSales, 'remaining_sales'));
        $totalCustomers = count($customerSales);
        $avgProgress = $totalCustomers > 0 ? array_sum(array_column($customerSales, 'shipping_percentage')) / $totalCustomers : 0;
        
        // Dispatch stats to dashboard
        $this->dispatch('statsUpdated', [
            'totalSales' => $this->totalSales,
            'pendingSales' => $pendingSales,
            'totalCustomers' => $totalCustomers,
            'avgProgress' => round($avgProgress, 0)
        ]);
        
        // Cache the results for 24 hours
        cache()->put($cacheKey, [
            'salesData' => $this->salesData,
            'totalSales' => $this->totalSales
        ], now()->addHours(24));
    }

    public function render()
    {
        return view('livewire.dashboard.customer-sales-by-department');
    }
}
