<?php

namespace App\Http\Livewire\Dashboard;

use App\Models\Seasons;
use Livewire\Component;
use App\Models\Departments;
use Illuminate\Support\Carbon;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Cache;

class CommissionMarginOverview extends BaseComponent
{
    #[Validate([
        'department' => ['required', 'exists:departments,id'],
        'season' => ['required', 'exists:seasons,id'],
    ])]
    public $department, $season = 4;

    public function updated($propertyName)
    {
        try {
            $this->validateOnly($propertyName);
        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->reset($propertyName);
        }
    }

    public function raw(){
        $cacheKey = 'commision_margin_overview_raw_data_v2_' . $this->season . '_' . $this->department;

        return Cache::remember($cacheKey, 3600, function () {
            // Using new commission tables
            return DB::select("
                SELECT
                    customers.name,
                    SUM(CASE
                        WHEN cds.shipped_qty > 0 THEN cds.shipped_qty
                        ELSE cds.qty
                    END) AS qty,
                    SUM(CASE
                        WHEN cds.shipped_qty > 0 THEN cds.shipped_qty * CONVERT_CURRENCY(colq.price, customers.currency, '£', co.seasons_id)
                        ELSE cds.qty * CONVERT_CURRENCY(colq.price, customers.currency, '£', co.seasons_id)
                    END) AS revenue,
                    SUM(CASE
                        WHEN cds.shipped_qty > 0 THEN cds.shipped_qty * CONVERT_CURRENCY(colq.price, customers.currency, '£', co.seasons_id) * COALESCE(IF(col.discount_percent = 0, NULL, col.discount_percent / 100), 1) * (col.commission_percent / 100)
                        ELSE cds.qty * CONVERT_CURRENCY(colq.price, customers.currency, '£', co.seasons_id) * COALESCE(IF(col.discount_percent = 0, NULL, col.discount_percent / 100), 1) * (col.commission_percent / 100)
                    END) AS commission

                FROM commission_order_line_quantities colq
                JOIN commission_order_lines col ON col.id = colq.commission_order_lines_id AND col.cancelled = 0 AND col.deleted_at IS NULL
                JOIN commission_orders co ON co.id = col.commission_orders_id AND co.cancelled = 0 AND co.seasons_id = ? " . ($this->department ? " AND co.departments_id = ? " : "") . " AND co.deleted_at IS NULL
                JOIN commission_drops cd ON cd.commission_order_lines_id = col.id AND cd.deleted_at IS NULL
                JOIN commission_drop_sizes cds ON cds.commission_drops_id = cd.id AND cds.sizes_id = colq.sizes_id AND cds.deleted_at IS NULL
                JOIN customers ON customers.id = co.customers_id

                WHERE colq.deleted_at IS NULL
                AND co.status = 'confirmed'
                GROUP BY customers.name
            ", $this->department ? [$this->season, $this->department] : [$this->season]);
        });
    }


    #[Computed()]
    public function departments(){
        return Departments::allCached();
    }
    #[Computed()]
    public function seasons(){
        return Seasons::allCached();
    }
    public function render()
    {
        return view('livewire.dashboard.commission-margin-overview', ['data' => collect($this->raw())]);
    }
}
