<?php

namespace App\Http\Livewire\Commission;

use App\Models\Design;
use App\Models\Styles;
use App\Models\Seasons;
use Livewire\Component;
use App\Models\Customer;
use App\Models\Suppliers;
use App\Models\Colourways;
use App\Models\Departments;
use App\Models\StyleVersions;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;

class StylesQuickAdd extends Component
{
    public bool $isOpen = false;
    public array $rows = [];
    
    // ERDOS defaults
    protected ?int $erdosDeptId = null;
    protected ?int $erdosFactoryId = null;
    protected int $erdosGaugeId = 24;

    public function mount()
    {
        $this->initDefaults();
    }

    protected function initDefaults()
    {
        $erdosDept = Departments::where('description', 'like', '%ERDOS%')->first();
        $erdosFactory = Suppliers::where('name', 'like', '%ERDOS%')->where('type', 'factory')->first();
        
        $this->erdosDeptId = $erdosDept?->id;
        $this->erdosFactoryId = $erdosFactory?->id ?? 5;
    }

    #[On('open-quickadd')]
    public function open()
    {
        $this->initDefaults();
        $this->rows = [$this->newRow()];
        $this->isOpen = true;
    }

    public function close()
    {
        $this->isOpen = false;
        $this->reset(['rows']);
    }

    protected function newRow(): array
    {
        return [
            'description' => '',
            'season_id' => Seasons::orderBy('created_at', 'desc')->first()?->id,
            'customer_id' => null,
            'customer_ref' => '',
            'category' => '',
            'colourways' => '',
        ];
    }

    public function addRow()
    {
        $this->rows[] = $this->newRow();
    }

    public function removeRow(int $index)
    {
        unset($this->rows[$index]);
        $this->rows = array_values($this->rows);
        
        if (empty($this->rows)) {
            $this->rows = [$this->newRow()];
        }
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function categories()
    {
        return [
            'ladies' => 'Ladies',
            'mens' => 'Mens',
            'childrens' => 'Childrens',
            'accessories' => 'Accessories',
        ];
    }

    public function save()
    {
        // Validate rows
        $errors = [];
        foreach ($this->rows as $index => $row) {
            $rowNum = $index + 1;
            if (empty(trim($row['description'] ?? ''))) {
                $errors[] = "Row {$rowNum}: Description is required";
            }
            if (empty($row['customer_id'])) {
                $errors[] = "Row {$rowNum}: Customer is required";
            }
            if (empty($row['season_id'])) {
                $errors[] = "Row {$rowNum}: Season is required";
            }
            if (empty($row['category'])) {
                $errors[] = "Row {$rowNum}: Category is required";
            }
            if (empty(trim($row['customer_ref'] ?? ''))) {
                $errors[] = "Row {$rowNum}: Customer Ref is required";
            }
            if (empty(trim($row['colourways'] ?? ''))) {
                $errors[] = "Row {$rowNum}: At least one colourway is required";
            }
        }

        if (!empty($errors)) {
            session()->flash('quickadd-error', implode('<br>', $errors));
            return;
        }

        $this->initDefaults();
        $created = 0;

        try {
            \DB::beginTransaction();

            foreach ($this->rows as $row) {
                // Create design
                $design = Design::create([
                    'description' => trim($row['description']),
                    'customers_id' => $row['customer_id'],
                    'factory_id' => $this->erdosFactoryId,
                    'gauge_id' => $this->erdosGaugeId,
                    'yarn' => '',
                    'issued' => now()->format('Y-m-d'),
                ]);

                // Create style
                $style = Styles::create([
                    'designs_id' => $design->id,
                    'departments_id' => $this->erdosDeptId,
                    'seasons_id' => $row['season_id'],
                    'customers_id' => $row['customer_id'],
                    'customer_ref' => trim($row['customer_ref']),
                    'category' => $row['category'],
                ]);

                // Create style version
                $version = StyleVersions::create([
                    'styles_id' => $style->id,
                    'factory_id' => $this->erdosFactoryId,
                    'gauge_id' => $this->erdosGaugeId,
                    'name' => 'V1',
                ]);

                // Create colourways
                $colourwayNames = array_map('trim', str_getcsv($row['colourways']));
                foreach ($colourwayNames as $name) {
                    if (!empty($name)) {
                        Colourways::create([
                            'style_versions_id' => $version->id,
                            'name' => $name,
                        ]);
                    }
                }

                $created++;
            }

            \DB::commit();

            $this->dispatch('styles-created');
            session()->flash('message', $created . ' style(s) created successfully!');
            $this->close();

        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('QuickAdd error: ' . $e->getMessage());
            session()->flash('quickadd-error', 'Error creating styles: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.commission.styles-quick-add');
    }
}
