<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\Colourways;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Models\Samples;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;

class StylesOverview extends Component
{
    public string $search = '';
    public $season = '';
    public $customer = '';
    public $status = '';
    public string $sortBy = 'rt';
    public string $sortDir = 'desc';
    
    // Infinite scroll
    public int $perPage = 30;
    public array $loadedItems = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    // Inline editing
    public ?int $editingCommentId = null;
    public string $editingCommentValue = '';

    protected $queryString = [
        'search' => ['except' => ''],
        'season' => ['except' => ''],
        'customer' => ['except' => ''],
        'status' => ['except' => ''],
    ];

    public function mount()
    {
        $this->loadInitialItems();
    }

    public function loadInitialItems()
    {
        $this->loadedItems = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    public function updatedSearch()
    {
        $this->loadInitialItems();
    }

    public function updatedSeason()
    {
        $this->loadInitialItems();
    }

    public function updatedCustomer()
    {
        $this->loadInitialItems();
    }

    public function updatedStatus()
    {
        $this->loadInitialItems();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'asc';
        }
        $this->loadInitialItems();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'season', 'customer', 'status']);
        $this->loadInitialItems();
    }

    public function quickAddSample($colourwayId, $sampleType)
    {
        // Create a pending sample directly
        Samples::create([
            'colourways_id' => $colourwayId,
            'sample_types_id' => $sampleType,
            'status' => 'pending',
            'qty' => 1,
        ]);
        
        // Refresh the list to show the new sample
        $this->loadInitialItems();
    }

    public function startEditingComment(int $colourwayId, ?string $currentComment = null)
    {
        $this->editingCommentId = $colourwayId;
        $this->editingCommentValue = $currentComment ?? '';
    }

    public function saveComment()
    {
        if (!$this->editingCommentId) {
            return;
        }

        $colourway = Colourways::find($this->editingCommentId);
        if ($colourway) {
            $colourway->update(['comments' => $this->editingCommentValue]);
            
            // Update the loaded item in place
            foreach ($this->loadedItems as &$item) {
                if ($item['id'] === $this->editingCommentId) {
                    $item['comments'] = $this->editingCommentValue;
                    break;
                }
            }
        }

        $this->cancelEditingComment();
    }

    public function cancelEditingComment()
    {
        $this->editingCommentId = null;
        $this->editingCommentValue = '';
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function activeFiltersCount()
    {
        return collect([$this->season, $this->customer, $this->status])
            ->filter()
            ->count();
    }

    #[Computed]
    public function stats()
    {
        $erdosDepts = Departments::where('description', 'like', '%ERDOS%')->pluck('id');
        
        $baseQuery = Colourways::query()
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->whereIn('styles.departments_id', $erdosDepts)
            ->where('colourways.cancelled', false)
            ->where('styles.cancelled', false);

        $total = (clone $baseQuery)->count();

        // Count those with approved proto
        $protoApproved = (clone $baseQuery)
            ->whereExists(function ($q) {
                $q->selectRaw(1)
                    ->from('samples')
                    ->whereColumn('samples.colourways_id', 'colourways.id')
                    ->whereIn('samples.sample_types_id', [1, 8])
                    ->where('samples.status', 'approved');
            })
            ->count();

        // Count those with approved sealer
        $sealerApproved = (clone $baseQuery)
            ->whereExists(function ($q) {
                $q->selectRaw(1)
                    ->from('samples')
                    ->whereColumn('samples.colourways_id', 'colourways.id')
                    ->where('samples.sample_types_id', 3)
                    ->where('samples.status', 'approved');
            })
            ->count();

        return [
            'total' => $total,
            'proto' => $protoApproved,
            'sealer' => $sealerApproved,
        ];
    }

    protected function buildQuery()
    {
        $erdosDepts = Departments::where('description', 'like', '%ERDOS%')->pluck('id');

        $query = Colourways::query()
            ->select([
                'colourways.id',
                'colourways.name as colourway_name',
                'colourways.img_thumb as image',
                'colourways.composition',
                'colourways.comments',
                'colourways.cancelled as cw_cancelled',
                'styles.id as style_id',
                'styles.customer_ref',
                'styles.cancelled as style_cancelled',
                'designs.id as design_id',
                'designs.description as design_description',
                'customers.name as customer_name',
                'customers.samples_required as customer_samples_required',
                'seasons.description as season_description',
                'style_versions.id as version_id',
                'style_versions.name as version_name',
            ])
            ->selectRaw("(SELECT JSON_ARRAYAGG(JSON_OBJECT('id', s.id, 'date_expected', s.date_expected, 'date_sent', s.date_sent, 'status', s.status, 'approved_date', s.approved_date)) FROM (SELECT * FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN (1,8) AND samples.deleted_at IS NULL ORDER BY created_at DESC LIMIT 3) s) as proto_samples")
            ->selectRaw("(SELECT JSON_ARRAYAGG(JSON_OBJECT('id', s.id, 'date_expected', s.date_expected, 'date_sent', s.date_sent, 'status', s.status, 'approved_date', s.approved_date)) FROM (SELECT * FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id = 3 AND samples.deleted_at IS NULL ORDER BY created_at DESC LIMIT 3) s) as sealer_samples")
            ->selectRaw("(SELECT JSON_ARRAYAGG(JSON_OBJECT('id', s.id, 'date_expected', s.date_expected, 'date_sent', s.date_sent, 'status', s.status, 'approved_date', s.approved_date)) FROM (SELECT * FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id IN (6,8) AND samples.deleted_at IS NULL ORDER BY created_at DESC LIMIT 3) s) as photo_samples")
            ->selectRaw("(SELECT JSON_ARRAYAGG(JSON_OBJECT('id', s.id, 'date_expected', s.date_expected, 'date_sent', s.date_sent, 'status', s.status, 'approved_date', s.approved_date)) FROM (SELECT * FROM samples WHERE samples.colourways_id = colourways.id AND samples.sample_types_id = 7 AND samples.deleted_at IS NULL ORDER BY created_at DESC LIMIT 3) s) as shipment_samples")
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('seasons', 'seasons.id', '=', 'styles.seasons_id')
            ->join('customers', 'customers.id', '=', 'styles.customers_id')
            ->join('designs', 'designs.id', '=', 'styles.designs_id')
            ->whereIn('styles.departments_id', $erdosDepts)
            ->where('colourways.cancelled', false)
            ->where('styles.cancelled', false);

        // Apply search
        if ($this->search) {
            $search = $this->search;
            $query->where(function ($q) use ($search) {
                $q->where('designs.description', 'like', "%{$search}%")
                  ->orWhere('designs.id', 'like', "%{$search}%")
                  ->orWhere('customers.name', 'like', "%{$search}%")
                  ->orWhere('styles.customer_ref', 'like', "%{$search}%")
                  ->orWhere('colourways.name', 'like', "%{$search}%");
            });
        }

        // Apply filters
        if ($this->season) {
            $query->where('styles.seasons_id', $this->season);
        }

        if ($this->customer) {
            $query->where('styles.customers_id', $this->customer);
        }

        if ($this->status) {
            if ($this->status === 'proto_pending') {
                $query->whereNotExists(function ($q) {
                    $q->selectRaw(1)
                        ->from('samples')
                        ->whereColumn('samples.colourways_id', 'colourways.id')
                        ->whereIn('samples.sample_types_id', [1, 8])
                        ->where('samples.status', 'approved');
                });
            } elseif ($this->status === 'proto_approved') {
                $query->whereExists(function ($q) {
                    $q->selectRaw(1)
                        ->from('samples')
                        ->whereColumn('samples.colourways_id', 'colourways.id')
                        ->whereIn('samples.sample_types_id', [1, 8])
                        ->where('samples.status', 'approved');
                });
            } elseif ($this->status === 'sealer_approved') {
                $query->whereExists(function ($q) {
                    $q->selectRaw(1)
                        ->from('samples')
                        ->whereColumn('samples.colourways_id', 'colourways.id')
                        ->where('samples.sample_types_id', 3)
                        ->where('samples.status', 'approved');
                });
            }
        }

        // Apply sorting
        $sortColumn = match ($this->sortBy) {
            'rt' => 'designs.id',
            'description' => 'designs.description',
            'customer' => 'customers.name',
            'colourway' => 'colourways.name',
            default => 'designs.id',
        };

        $query->orderBy($sortColumn, $this->sortDir);

        return $query;
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedItems)->pluck('id')->toArray();
        
        $query = $this->buildQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('colourways.id', $loadedIds);
        }

        $newItems = $query->limit($this->perPage)->get();

        if ($newItems->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newItems as $item) {
            // Parse samples_required JSON from customer settings
            $samplesRequired = [];
            if ($item->customer_samples_required) {
                $samplesRequired = json_decode($item->customer_samples_required, true) ?? [];
            }
            
            // Parse sample arrays (up to 3 each)
            $protoSamples = $item->proto_samples ? json_decode($item->proto_samples, true) ?? [] : [];
            $sealerSamples = $item->sealer_samples ? json_decode($item->sealer_samples, true) ?? [] : [];
            $photoSamples = $item->photo_samples ? json_decode($item->photo_samples, true) ?? [] : [];
            $shipmentSamples = $item->shipment_samples ? json_decode($item->shipment_samples, true) ?? [] : [];
            
            $this->loadedItems[] = [
                'id' => $item->id,
                'style_id' => $item->style_id,
                'design_id' => $item->design_id,
                'design_description' => $item->design_description,
                'customer_name' => $item->customer_name,
                'customer_ref' => $item->customer_ref,
                'season_description' => $item->season_description,
                'version_name' => $item->version_name,
                'colourway_name' => $item->colourway_name,
                'image' => $item->image,
                'composition' => $item->composition,
                'comments' => $item->comments,
                'proto_samples' => $protoSamples,
                'sealer_samples' => $sealerSamples,
                'photo_samples' => $photoSamples,
                'shipment_samples' => $shipmentSamples,
                'proto_required' => !empty($samplesRequired['1']),
                'sealer_required' => !empty($samplesRequired['3']),
                'photo_required' => !empty($samplesRequired['6']),
                'shipment_required' => !empty($samplesRequired['7']),
            ];
        }

        $this->isLoading = false;
    }

    #[On('style-saved')]
    #[On('sample-saved')]
    public function refreshData()
    {
        unset($this->stats);
        $this->loadInitialItems();
    }

    public function render()
    {
        return view('livewire.commission.styles-overview')
            ->layout('layouts.commission');
    }
}

