<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\Styles as StyleModel;
use App\Models\Seasons;
use App\Models\Customer;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Collection;

class Styles extends Component
{
    #[Url]
    public string $search = '';
    
    #[Url]
    public string $season = '';
    
    #[Url]
    public string $customer = '';
    
    public int $perPage = 48; // Items per page for infinite scroll
    public int $page = 1;
    public int $totalCount = 0;

    public function mount()
    {
        $this->loadStyles();
    }

    #[On('styles-created')]
    #[On('style-saved')]
    public function refreshStyles()
    {
        $this->resetScroll();
    }

    public function updatedSearch()
    {
        $this->resetScroll();
    }

    public function updatedSeason()
    {
        $this->resetScroll();
    }

    public function updatedCustomer()
    {
        $this->resetScroll();
    }

    public function resetScroll()
    {
        $this->page = 1;
        unset($this->styles);
        $this->loadStyles();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'season', 'customer']);
        $this->resetScroll();
    }

    protected function buildQuery()
    {
        return StyleModel::query()
            ->with([
                'seasons:id,description',
                'customers:id,name',
                'designs:id,description',
                'firstImageColourway'
            ])
            // Only show ERDOS department styles for Commission section
            ->whereHas('departments', fn($q) => $q->where('description', 'like', '%ERDOS%'))
            ->when($this->search, function ($query) {
                $searchTerms = explode(' ', trim($this->search));
                $query->where(function ($q) use ($searchTerms) {
                    foreach ($searchTerms as $term) {
                        $q->where(function ($subQ) use ($term) {
                            $subQ->where('customer_ref', 'like', "%{$term}%")
                                ->orWhereHas('designs', fn($d) => $d->where('id', 'like', "%{$term}%")
                                    ->orWhere('description', 'like', "%{$term}%"));
                        });
                    }
                });
            })
            ->when($this->season, fn($q) => $q->where('seasons_id', $this->season))
            ->when($this->customer, fn($q) => $q->where('customers_id', $this->customer))
            ->orderBy('id', 'desc');
    }

    protected function loadStyles()
    {
        $this->totalCount = $this->buildQuery()->count();
    }

    #[Computed]
    public function styles(): Collection
    {
        return $this->buildQuery()
            ->take($this->page * $this->perPage)
            ->get();
    }

    public function hasMorePages(): bool
    {
        return $this->styles->count() < $this->totalCount;
    }

    public function loadMore()
    {
        if (!$this->hasMorePages()) {
            return;
        }
        
        $this->page++;
        unset($this->styles);
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        // Get customers that have commission-type orders (you may need to adjust this logic)
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if ($this->season) $count++;
        if ($this->customer) $count++;
        return $count;
    }

    public function render()
    {
        return view('livewire.commission.styles')->layout('layouts.commission');
    }
}

