<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\Styles;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Models\Suppliers;
use App\Models\StyleVersions;
use App\Models\Colourways;
use App\Models\User;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use App\Services\DesignSimilarityService;

class StyleEdit extends Component
{
    use WithFileUploads;

    public bool $isOpen = false;
    public ?int $styleId = null;
    public string $activeTab = 'details';
    public ?int $activeColourwayId = null;
    public bool $isDirty = false;
    
    // Style fields
    public $departments_id;
    public $seasons_id;
    public $customers_id;
    public $customer_ref;
    public $category;
    public $notes;
    public $cancelled = false;
    
    // Design fields (read-only display)
    public $design_id;
    public $design_description;
    
    // Version fields
    public ?int $activeVersionId = null;
    public $version_name;
    public $factory_id;
    
    // Colourway fields
    public $colourway_name;
    public $colourway_customer_description;
    public $colourway_composition;
    public $colourway_cancelled = false;
    public $colourway_image;
    public $new_image;

    protected function rules()
    {
        if ($this->isNewStyle) {
            // ALL fields required for new style
            return [
                'new_design_description' => 'required|string|min:3|max:255',
                'seasons_id' => 'required|exists:seasons,id',
                'customers_id' => 'required|exists:customers,id',
                'departments_id' => 'required|exists:departments,id',
                'factory_id' => 'required|exists:suppliers,id',
                'customer_ref' => 'required|string|min:1|max:255',
                'category' => 'required|in:ladies,mens,childrens,accessories',
                'notes' => 'nullable|string|max:5000',
            ];
        }
        
        return [
            'departments_id' => 'nullable|exists:departments,id',
            'seasons_id' => 'required|exists:seasons,id',
            'customers_id' => 'required|exists:customers,id',
            'customer_ref' => 'nullable|string|max:255',
            'category' => 'nullable|in:ladies,mens,childrens,accessories',
            'notes' => 'nullable|string|max:5000',
            'cancelled' => 'boolean',
            'design_description' => 'required|string|min:3|max:255',
            'version_name' => 'nullable|string|max:255',
            'factory_id' => 'nullable|exists:suppliers,id',
            'colourway_name' => 'nullable|string|max:255',
            'colourway_customer_description' => 'nullable|string|max:255',
            'colourway_composition' => 'nullable|string|max:5000',
            'colourway_cancelled' => 'boolean',
            'new_image' => 'nullable|image|max:2048',
        ];
    }

    protected $validationAttributes = [
        'new_design_description' => 'design description',
        'design_description' => 'description',
        'seasons_id' => 'season',
        'customers_id' => 'customer',
        'departments_id' => 'department',
        'factory_id' => 'factory',
        'customer_ref' => 'customer reference',
        'version_name' => 'version name',
        'colourway_name' => 'colourway name',
        'colourway_customer_description' => 'customer description',
        'colourway_composition' => 'composition',
        'new_image' => 'image',
    ];

    protected $messages = [
        'new_design_description.required' => 'Design description is required.',
        'new_design_description.min' => 'Design description must be at least 3 characters.',
        'design_description.required' => 'Description is required.',
        'design_description.min' => 'Description must be at least 3 characters.',
        'seasons_id.required' => 'Season is required.',
        'customers_id.required' => 'Customer is required.',
        'departments_id.required' => 'Department is required.',
        'factory_id.required' => 'Factory is required.',
        'customer_ref.required' => 'Customer reference is required.',
        'category.required' => 'Category is required.',
        'new_image.image' => 'The file must be an image.',
        'new_image.max' => 'The image must be less than 2MB.',
    ];

    public bool $isNewStyle = false;
    public $new_design_description;
    public bool $checkingSimilarity = false;
    public array $similarDesigns = [];

    #[On('open-style-edit')]
    public function open(int $styleId)
    {
        $this->styleId = $styleId;
        $this->isNewStyle = false;
        $this->loadStyle();
        $this->isOpen = true;
        $this->isDirty = false;
        $this->activeTab = 'details';
    }

    #[On('create-new-style')]
    public function createNew()
    {
        $this->reset();
        $this->resetErrorBag();
        $this->isNewStyle = true;
        $this->isOpen = true;
        $this->isDirty = false;
        $this->activeTab = 'details';
        
        // Set ERDOS defaults
        $erdosDept = Departments::where('description', 'like', '%ERDOS%')->first();
        $erdosFactory = Suppliers::where('name', 'like', '%ERDOS%')->where('type', 'factory')->first();
        
        $this->departments_id = $erdosDept?->id;
        $this->factory_id = $erdosFactory?->id;
        
        // Default to most recent season
        $this->seasons_id = Seasons::orderBy('created_at', 'desc')->first()?->id;
    }

    public function close()
    {
        $this->isOpen = false;
        $this->isDirty = false;
        $this->reset(['styleId', 'activeVersionId', 'activeColourwayId', 'new_image']);
    }

    public function confirmClose()
    {
        if ($this->isDirty) {
            $this->dispatch('confirm-close');
        } else {
            $this->close();
        }
    }

    public function updated($property)
    {
        // Mark as dirty when any editable field changes
        $editableFields = [
            'departments_id', 'seasons_id', 'customers_id', 'customer_ref', 
            'category', 'notes', 'cancelled', 'version_name', 'factory_id',
            'colourway_name', 'colourway_customer_description', 
            'colourway_composition', 'colourway_cancelled', 'new_image',
            'new_design_description', 'design_description'
        ];
        
        if (in_array($property, $editableFields)) {
            $this->isDirty = true;
        }

        // Real-time validation for ALL required fields in new style mode
        if ($this->isNewStyle) {
            $this->validateOnly($property, [
                'new_design_description' => ['required', 'string', 'min:3', 'max:255'],
                'seasons_id' => ['required'],
                'customers_id' => ['required'],
                'departments_id' => ['required'],
                'factory_id' => ['required'],
                'customer_ref' => ['required', 'min:1'],
                'category' => ['required', 'in:ladies,mens,childrens,accessories'],
            ], [
                'new_design_description.required' => 'Design description is required.',
                'new_design_description.min' => 'At least 3 characters required.',
                'seasons_id.required' => 'Season is required.',
                'customers_id.required' => 'Customer is required.',
                'departments_id.required' => 'Department is required.',
                'factory_id.required' => 'Factory is required.',
                'customer_ref.required' => 'Customer reference is required.',
                'category.required' => 'Category is required.',
            ]);
        }
    }

    public $similarityError = null;
    public bool $showDuplicateConfirm = false;

    public function checkForSimilarDesigns()
    {
        $this->similarityError = null;
        $this->similarDesigns = [];
        
        if (!$this->isNewStyle || empty($this->new_design_description) || strlen(trim($this->new_design_description)) < 3) {
            $this->similarityError = 'Please enter at least 3 characters';
            return;
        }

        $this->checkingSimilarity = true;
        
        try {
            $service = app(DesignSimilarityService::class);
            $result = $service->checkSimilarity(
                $this->new_design_description,
                $this->customers_id
            );
            
            if (isset($result['error'])) {
                $this->similarityError = $result['error'];
            } else {
                $this->similarDesigns = $result['matches'] ?? [];
                if (empty($this->similarDesigns)) {
                    $this->similarityError = 'No similar designs found - you\'re good to go!';
                }
            }
        } catch (\Exception $e) {
            $this->similarityError = 'Error checking: ' . $e->getMessage();
        }
        
        $this->checkingSimilarity = false;
    }

    public function loadStyle()
    {
        $style = Styles::with([
            'designs',
            'seasons',
            'customers',
            'departments',
            'style_versions.factories',
            'style_versions.colourways'
        ])->find($this->styleId);

        if (!$style) {
            $this->close();
            return;
        }

        // Style fields
        $this->departments_id = $style->departments_id;
        $this->seasons_id = $style->seasons_id;
        $this->customers_id = $style->customers_id;
        $this->customer_ref = $style->customer_ref;
        $this->category = $style->category;
        $this->notes = $style->notes;
        $this->cancelled = (bool) $style->cancelled;
        
        // Design info (read-only)
        $this->design_id = $style->designs?->id;
        $this->design_description = $style->designs?->description;
        
        // Load first version by default
        $firstVersion = $style->style_versions->first();
        if ($firstVersion) {
            $this->selectVersion($firstVersion->id);
        }
    }

    public function selectVersion(int $versionId)
    {
        $this->activeVersionId = $versionId;
        $version = StyleVersions::with('colourways', 'factories')->find($versionId);
        
        if ($version) {
            $this->version_name = $version->name;
            $this->factory_id = $version->factory_id;
            
            // Select first colourway
            $firstColourway = $version->colourways->first();
            if ($firstColourway) {
                $this->selectColourway($firstColourway->id);
            } else {
                $this->activeColourwayId = null;
                $this->resetColourwayFields();
            }
        }
    }

    public function selectColourway(int $colourwayId)
    {
        $this->activeColourwayId = $colourwayId;
        $colourway = Colourways::find($colourwayId);
        
        if ($colourway) {
            $this->colourway_name = $colourway->name;
            $this->colourway_customer_description = $colourway->customer_description;
            $this->colourway_composition = $colourway->composition;
            $this->colourway_cancelled = (bool) $colourway->cancelled;
            $this->colourway_image = $colourway->img_thumb;
            $this->new_image = null;
        }
    }

    protected function resetColourwayFields()
    {
        $this->colourway_name = null;
        $this->colourway_customer_description = null;
        $this->colourway_composition = null;
        $this->colourway_cancelled = false;
        $this->colourway_image = null;
        $this->new_image = null;
    }

    public function addVersion()
    {
        if (!$this->styleId) {
            return;
        }

        $style = Styles::find($this->styleId);
        if (!$style) {
            return;
        }

        // Get the next version number
        $existingVersions = $style->style_versions()->count();
        $versionName = 'V' . ($existingVersions + 1);

        // Get defaults from existing version or style
        $existingVersion = $style->style_versions()->first();
        $factoryId = $existingVersion?->factory_id;
        $gaugeId = $existingVersion?->gauge_id ?? 24; // Default ERDOS gauge

        $version = StyleVersions::create([
            'styles_id' => $this->styleId,
            'name' => $versionName,
            'factory_id' => $factoryId,
            'gauge_id' => $gaugeId,
        ]);

        // Create a default colourway for this version
        Colourways::create([
            'style_versions_id' => $version->id,
            'name' => 'Main',
        ]);

        // Select the new version
        $this->selectVersion($version->id);
        $this->isDirty = false; // Just created, no unsaved changes
        
        session()->flash('message', 'Version created successfully.');
    }

    public function cloneVersion()
    {
        if (!$this->activeVersionId) {
            return;
        }

        $original = StyleVersions::with('colourways')->find($this->activeVersionId);
        if (!$original) {
            return;
        }

        // Clone the version
        $clone = $original->replicate();
        $clone->name = ($original->name ?? 'Version') . ' (Copy)';
        $clone->save();

        // Clone all colourways
        foreach ($original->colourways as $colourway) {
            $cwClone = $colourway->replicate();
            $cwClone->style_versions_id = $clone->id;
            $cwClone->cancelled = false;
            $cwClone->save();
        }

        // Select the cloned version
        $this->selectVersion($clone->id);
        $this->isDirty = false;
        
        session()->flash('message', 'Version cloned successfully.');
    }

    public function deleteVersion()
    {
        if (!$this->activeVersionId) {
            return;
        }

        $version = StyleVersions::with('colourways')->find($this->activeVersionId);
        if (!$version) {
            return;
        }

        // Check if this version has orders linked
        $hasOrders = false;
        foreach ($version->colourways as $cw) {
            if ($cw->customer_order_lines()->exists()) {
                $hasOrders = true;
                break;
            }
        }

        if ($hasOrders) {
            session()->flash('error', 'Cannot delete: Version has orders linked to it.');
            return;
        }

        // Check we're not deleting the only version
        $style = Styles::find($this->styleId);
        if ($style && $style->style_versions()->count() <= 1) {
            session()->flash('error', 'Cannot delete: Style must have at least one version.');
            return;
        }

        $version->delete();

        // Select another version
        $this->activeVersionId = null;
        if ($style && $style->style_versions->isNotEmpty()) {
            $this->selectVersion($style->style_versions->first()->id);
        }
        
        session()->flash('message', 'Version deleted.');
    }

    public function addColourway()
    {
        if (!$this->activeVersionId) {
            return;
        }

        $colourway = Colourways::create([
            'style_versions_id' => $this->activeVersionId,
            'name' => 'New Colourway',
        ]);

        // Select the new colourway
        $this->selectColourway($colourway->id);
    }

    public function cloneColourway()
    {
        if (!$this->activeColourwayId) {
            return;
        }

        $original = Colourways::find($this->activeColourwayId);
        if (!$original) {
            return;
        }

        $clone = $original->replicate();
        $clone->name = ($original->name ?? 'Colourway') . ' (Copy)';
        $clone->cancelled = false;
        $clone->save();

        // Select the cloned colourway
        $this->selectColourway($clone->id);
    }

    public function deleteColourway()
    {
        if (!$this->activeColourwayId) {
            return;
        }

        $colourway = Colourways::find($this->activeColourwayId);
        if (!$colourway) {
            return;
        }

        // Check if this colourway has orders
        if ($colourway->customer_order_lines()->exists()) {
            session()->flash('error', 'Cannot delete: Colourway has orders linked to it.');
            return;
        }

        $colourway->delete();

        // Reset selection
        $this->activeColourwayId = null;
        $this->resetColourwayFields();

        // Select another colourway if available
        $version = StyleVersions::with('colourways')->find($this->activeVersionId);
        if ($version && $version->colourways->isNotEmpty()) {
            $this->selectColourway($version->colourways->first()->id);
        }
    }

    public function save()
    {
        // Clear previous errors
        $this->resetErrorBag();
        
        if ($this->isNewStyle) {
            // Validate ALL fields - every field must be filled
            $rules = [
                'new_design_description' => ['required', 'string', 'min:3', 'max:255'],
                'seasons_id' => ['required', 'exists:seasons,id'],
                'customers_id' => ['required', 'exists:customers,id'],
                'departments_id' => ['required', 'exists:departments,id'],
                'factory_id' => ['required', 'exists:suppliers,id'],
                'customer_ref' => ['required', 'string', 'min:1', 'max:255'],
                'category' => ['required', 'in:ladies,mens,childrens,accessories'],
            ];
            
            $messages = [
                'new_design_description.required' => 'Design description is required.',
                'new_design_description.min' => 'Design description must be at least 3 characters.',
                'seasons_id.required' => 'Season is required.',
                'customers_id.required' => 'Customer is required.',
                'departments_id.required' => 'Department is required.',
                'factory_id.required' => 'Factory is required.',
                'customer_ref.required' => 'Customer reference is required.',
                'category.required' => 'Category is required.',
            ];
            
            // Run validation - this throws ValidationException if it fails
            $this->validate($rules, $messages);
            
            // If we get here, validation passed - check for duplicates
            $this->checkForSimilarDesigns();
            
            if (!empty($this->similarDesigns)) {
                $this->showDuplicateConfirm = true;
                return;
            }
            
            $this->saveNewStyle();
        } else {
            $this->saveExistingStyle();
        }
    }

    public function confirmSaveWithDuplicates()
    {
        $this->showDuplicateConfirm = false;
        $this->saveNewStyle();
    }

    public function cancelDuplicateSave()
    {
        $this->showDuplicateConfirm = false;
    }

    protected function saveNewStyle()
    {
        // Double-check validation (save() already validated, but be safe)
        $rules = [
            'new_design_description' => ['required', 'string', 'min:3', 'max:255'],
            'seasons_id' => ['required', 'exists:seasons,id'],
            'customers_id' => ['required', 'exists:customers,id'],
            'departments_id' => ['required', 'exists:departments,id'],
            'factory_id' => ['required', 'exists:suppliers,id'],
            'customer_ref' => ['required', 'string', 'min:1', 'max:255'],
            'category' => ['required', 'in:ladies,mens,childrens,accessories'],
            'notes' => ['nullable', 'string', 'max:5000'],
        ];
        
        $this->validate($rules);

        try {
            \DB::beginTransaction();

            // Default factory and gauge for ERDOS
            $factoryId = $this->factory_id ?: 5; // ERDOS default
            $gaugeId = 24; // ERDOS default gauge

            // Create the Design first
            $design = \App\Models\Design::create([
                'description' => trim($this->new_design_description),
                'customers_id' => $this->customers_id,
                'gauge_id' => $gaugeId,
                'factory_id' => $factoryId,
                'issued' => now()->format('Y-m-d'),
                'yarn' => '',
            ]);

            // Create the Style
            $style = Styles::create([
                'designs_id' => $design->id,
                'departments_id' => $this->departments_id ?: null,
                'seasons_id' => $this->seasons_id,
                'customers_id' => $this->customers_id,
                'customer_ref' => $this->customer_ref ? trim($this->customer_ref) : null,
                'category' => $this->category ? trim($this->category) : null,
                'notes' => $this->notes ? trim($this->notes) : null,
            ]);

            // Create a default StyleVersion with gauge_id
            $version = StyleVersions::create([
                'styles_id' => $style->id,
                'name' => 'V1',
                'factory_id' => $factoryId,
                'gauge_id' => $gaugeId,
            ]);

            // Create a default Colourway
            Colourways::create([
                'style_versions_id' => $version->id,
                'name' => 'Main',
            ]);

            \DB::commit();

            $this->isDirty = false;
            $this->isNewStyle = false;
            $this->dispatch('style-saved');
            
            // Open the newly created style for editing
            $this->open($style->id);
            
            session()->flash('message', 'Style created successfully.');

        } catch (\Illuminate\Database\QueryException $e) {
            \DB::rollBack();
            \Log::error('Failed to create style (DB): ' . $e->getMessage(), [
                'description' => $this->new_design_description,
                'customer_id' => $this->customers_id,
                'season_id' => $this->seasons_id,
            ]);
            
            session()->flash('error', 'Database error: ' . $this->friendlyDbError($e->getMessage()));
        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Failed to create style: ' . $e->getMessage(), [
                'description' => $this->new_design_description,
                'customer_id' => $this->customers_id,
                'season_id' => $this->seasons_id,
            ]);
            session()->flash('error', 'Something went wrong. Please try again.');
        }
    }

    protected function friendlyDbError(string $message): string
    {
        if (str_contains($message, 'Duplicate entry')) {
            return 'A style with these details already exists.';
        }
        if (str_contains($message, 'foreign key constraint')) {
            return 'Invalid selection in one of the dropdowns.';
        }
        if (str_contains($message, "doesn't have a default value") || str_contains($message, 'cannot be null')) {
            return 'A required database field is missing. Please contact support.';
        }
        return 'An unexpected error occurred. Please contact support.';
    }

    protected function saveExistingStyle()
    {
        $this->validate();

        try {
            \DB::beginTransaction();

            // Save Style
            $style = Styles::find($this->styleId);
            if (!$style) {
                throw new \Exception('Style not found');
            }
            
            $style->update([
                'departments_id' => $this->departments_id,
                'seasons_id' => $this->seasons_id,
                'customers_id' => $this->customers_id,
                'customer_ref' => $this->customer_ref ? trim($this->customer_ref) : null,
                'category' => $this->category ? trim($this->category) : null,
                'notes' => $this->notes ? trim($this->notes) : null,
                'cancelled' => $this->cancelled,
            ]);

            // Save Design description
            if ($style->designs && $this->design_description) {
                $style->designs->update([
                    'description' => trim($this->design_description),
                ]);
            }

            // Save Version
            if ($this->activeVersionId) {
                $version = StyleVersions::find($this->activeVersionId);
                if ($version) {
                    $version->update([
                        'name' => $this->version_name ? trim($this->version_name) : null,
                        'factory_id' => $this->factory_id,
                    ]);
                }
            }

            // Save Colourway
            if ($this->activeColourwayId) {
                $colourway = Colourways::find($this->activeColourwayId);
                if ($colourway) {
                    $data = [
                        'name' => $this->colourway_name ? trim($this->colourway_name) : null,
                        'customer_description' => $this->colourway_customer_description ? trim($this->colourway_customer_description) : null,
                        'composition' => $this->colourway_composition ? trim($this->colourway_composition) : null,
                        'cancelled' => $this->colourway_cancelled,
                    ];

                    // Handle image upload
                    if ($this->new_image) {
                        $path = $this->new_image->store('colourways', 'public');
                        $data['img_thumb'] = $path;
                        $data['image'] = $path;
                    }

                    $colourway->update($data);
                }
            }

            \DB::commit();

            $this->isDirty = false;
            $this->new_image = null; // Clear the uploaded file
            $this->dispatch('style-saved');
            
            session()->flash('message', 'Style saved successfully.');

        } catch (\Illuminate\Database\QueryException $e) {
            \DB::rollBack();
            \Log::error('Failed to save style (DB): ' . $e->getMessage(), [
                'style_id' => $this->styleId,
            ]);
            
            if (str_contains($e->getMessage(), 'foreign key constraint')) {
                session()->flash('error', 'Invalid selection. Please ensure all dropdowns have valid options selected.');
            } elseif (str_contains($e->getMessage(), 'cannot be null')) {
                session()->flash('error', 'Required field missing. Please fill in all required fields.');
            } else {
                session()->flash('error', 'Database error occurred. Please contact support if this persists.');
            }
        } catch (\Exception $e) {
            \DB::rollBack();
            \Log::error('Failed to save style: ' . $e->getMessage(), [
                'style_id' => $this->styleId,
                'trace' => $e->getTraceAsString(),
            ]);
            session()->flash('error', 'Something went wrong saving the style. Please check all fields and try again.');
        }
    }

    #[Computed(persist: false)]
    public function style()
    {
        if (!$this->styleId) return null;
        return Styles::with([
            'style_versions.colourways',
            'style_versions.factories'
        ])->find($this->styleId);
    }

    #[Computed]
    public function departments()
    {
        return Departments::orderBy('description')->get();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function factories()
    {
        return Suppliers::where('type', 'factory')->orderBy('name')->get();
    }

    #[Computed]
    public function categories()
    {
        return [
            'ladies' => 'Ladies',
            'mens' => 'Mens',
            'childrens' => 'Childrens',
            'accessories' => 'Accessories',
        ];
    }

    #[Computed(persist: false)]
    public function auditHistory()
    {
        if (!$this->styleId) {
            return collect();
        }

        $style = Styles::find($this->styleId);
        if (!$style) {
            return collect();
        }

        // Get related IDs
        $designId = $style->designs_id;
        $versionIds = $style->style_versions()->pluck('id')->toArray();
        $colourwayIds = Colourways::whereIn('style_versions_id', $versionIds)->pluck('id')->toArray();

        // Get names for versions and colourways
        $versionNames = StyleVersions::whereIn('id', $versionIds)->pluck('name', 'id')->toArray();
        $colourwayNames = Colourways::whereIn('id', $colourwayIds)->pluck('name', 'id')->toArray();

        // Build queries for all related models
        $audits = DB::table('audits')
            ->where(function ($query) use ($style, $designId, $versionIds, $colourwayIds) {
                // Style audits
                $query->where(function ($q) use ($style) {
                    $q->where('auditable_type', 'App\\Models\\Styles')
                      ->where('auditable_id', $style->id);
                });
                
                // Design audits
                if ($designId) {
                    $query->orWhere(function ($q) use ($designId) {
                        $q->where('auditable_type', 'App\\Models\\Design')
                          ->where('auditable_id', $designId);
                    });
                }
                
                // Version audits
                if (!empty($versionIds)) {
                    $query->orWhere(function ($q) use ($versionIds) {
                        $q->where('auditable_type', 'App\\Models\\StyleVersions')
                          ->whereIn('auditable_id', $versionIds);
                    });
                }
                
                // Colourway audits
                if (!empty($colourwayIds)) {
                    $query->orWhere(function ($q) use ($colourwayIds) {
                        $q->where('auditable_type', 'App\\Models\\Colourways')
                          ->whereIn('auditable_id', $colourwayIds);
                    });
                }
            })
            ->orderBy('created_at', 'desc')
            ->orderBy('id', 'desc')
            ->limit(200)
            ->get();

        // Get user names
        $userIds = $audits->pluck('user_id')->unique()->filter()->toArray();
        $users = User::whereIn('id', $userIds)->pluck('name', 'id');

        return $audits->map(function ($audit) use ($users, $versionNames, $colourwayNames) {
            $oldValues = json_decode($audit->old_values, true) ?? [];
            $newValues = json_decode($audit->new_values, true) ?? [];
            
            // Get model type without namespace
            $modelType = class_basename($audit->auditable_type);
            
            // Add specific name for versions and colourways
            $itemName = null;
            if ($modelType === 'StyleVersions') {
                $modelType = 'Version';
                $itemName = $versionNames[$audit->auditable_id] ?? null;
                // If name was in the audit values, use that
                if (!$itemName && isset($newValues['name'])) {
                    $itemName = $newValues['name'];
                }
            } elseif ($modelType === 'Colourways') {
                $modelType = 'Colourway';
                $itemName = $colourwayNames[$audit->auditable_id] ?? null;
                // If name was in the audit values, use that
                if (!$itemName && isset($newValues['name'])) {
                    $itemName = $newValues['name'];
                }
            } elseif ($modelType === 'Styles') {
                $modelType = 'Style';
            }
            
            // Format changes
            $changes = [];
            foreach ($newValues as $field => $newValue) {
                $oldValue = $oldValues[$field] ?? null;
                
                // Skip if values are effectively the same (handle 0/false, 1/true comparisons)
                if ($this->areValuesEquivalent($oldValue, $newValue)) {
                    continue;
                }
                
                $changes[] = [
                    'field' => $this->formatFieldName($field),
                    'old' => $this->formatAuditValue($field, $oldValue),
                    'new' => $this->formatAuditValue($field, $newValue),
                ];
            }
            
            return (object) [
                'id' => $audit->id,
                'event' => ucfirst($audit->event),
                'model_type' => $modelType,
                'item_name' => $itemName,
                'user_name' => $users[$audit->user_id] ?? 'System',
                'created_at' => \Carbon\Carbon::parse($audit->created_at),
                'changes' => $changes,
            ];
        })->filter(function ($audit) {
            // For "updated" events, only show if there are actual changes
            // "created" and "deleted" events are always shown
            if ($audit->event === 'Updated' && empty($audit->changes)) {
                return false;
            }
            return true;
        })->values();
    }

    protected function formatFieldName(string $field): string
    {
        $fieldNames = [
            'designs_id' => 'Design',
            'departments_id' => 'Department',
            'seasons_id' => 'Season',
            'customers_id' => 'Customer',
            'customer_ref' => 'Customer Ref',
            'factory_id' => 'Factory',
            'style_versions_id' => 'Version',
            'img_thumb' => 'Image',
            'customer_description' => 'Customer Description',
        ];
        
        return $fieldNames[$field] ?? ucwords(str_replace('_', ' ', $field));
    }

    protected function formatAuditValue(string $field, $value): string
    {
        if ($value === null || $value === '') {
            return '—';
        }
        
        if (is_bool($value)) {
            return $value ? 'Yes' : 'No';
        }
        
        // Truncate long values
        if (is_string($value) && strlen($value) > 50) {
            return substr($value, 0, 47) . '...';
        }
        
        return (string) $value;
    }

    protected function areValuesEquivalent($oldValue, $newValue): bool
    {
        // Exact match
        if ($oldValue === $newValue) {
            return true;
        }
        
        // Handle boolean/integer equivalence (0/false, 1/true)
        $booleanFields = ['cancelled', 'carryover'];
        
        // Normalize both values to boolean for comparison
        $oldBool = $this->toBooleanValue($oldValue);
        $newBool = $this->toBooleanValue($newValue);
        
        // If both can be interpreted as booleans and are equal
        if ($oldBool !== null && $newBool !== null && $oldBool === $newBool) {
            return true;
        }
        
        // Handle null/empty string equivalence
        if (($oldValue === null || $oldValue === '') && ($newValue === null || $newValue === '')) {
            return true;
        }
        
        return false;
    }

    protected function toBooleanValue($value): ?bool
    {
        if ($value === true || $value === 1 || $value === '1') {
            return true;
        }
        if ($value === false || $value === 0 || $value === '0') {
            return false;
        }
        return null;
    }

    public function render()
    {
        return view('livewire.commission.style-edit');
    }
}

