<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\CommissionInvoiceGroup;
use App\Models\Customer;
use App\Services\ZohoService;
use App\Http\Livewire\Traits\HandlesZohoOAuth;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Log;

class SlInvoicing extends Component
{
    use HandlesZohoOAuth;
    
    protected ZohoService $zoho;
    
    // Infinite scroll state
    public array $loadedInvoiceGroups = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    public function boot()
    {
        $this->zoho = app(ZohoService::class);
    }
    
    public function mount()
    {
        $this->loadInitialInvoiceGroups();
    }

    #[On('invoice-updated')]
    public function refreshInvoices()
    {
        unset($this->stats);
        $this->loadInitialInvoiceGroups();
    }

    #[Computed]
    public function isZohoConnected(): bool
    {
        return $this->zoho->isConnected();
    }

    #[Url]
    public string $search = '';
    
    #[Url]
    public string $customer = '';
    
    #[Url]
    public string $status = '';
    
    public int $perPage = 25;
    
    // For expanded sections
    public array $expanded = [];
    
    // For Zoho ID input
    public array $zohoIds = [];
    
    // For manual Zoho entry
    public ?int $manualInvoiceId = null;
    public string $manualZohoNumber = '';

    public function updatingSearch()
    {
        $this->loadInitialInvoiceGroups();
    }

    public function updatingCustomer()
    {
        $this->loadInitialInvoiceGroups();
    }

    public function updatingStatus()
    {
        $this->loadInitialInvoiceGroups();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'customer', 'status']);
        $this->loadInitialInvoiceGroups();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function statusOptions()
    {
        return [
            'ready' => ['label' => 'Shipped & Approved', 'icon' => 'fa-check-circle', 'color' => 'primary'],
            'sent_to_zoho' => ['label' => 'Sent to Zoho', 'icon' => 'fa-paper-plane', 'color' => 'info'],
            'complete' => ['label' => 'Complete', 'icon' => 'fa-check-double', 'color' => 'success'],
        ];
    }

    /**
     * Build the base query for invoice groups
     */
    protected function buildInvoiceGroupsQuery()
    {
        return CommissionInvoiceGroup::query()
            ->with([
                'customer:id,name',
                'drops' => function ($q) {
                    $q->with([
                        'line.order:id,customer_po',
                        'line.colourway:id,name,style_versions_id',
                        'line.colourway.style_versions:id,styles_id',
                        'line.colourway.style_versions.styles:id,designs_id,customer_ref',
                        'line.colourway.style_versions.styles.designs:id,description',
                        'line.quantities:id,commission_order_lines_id,sizes_id,price',
                        'sizes.size',
                    ]);
                },
            ])
            ->withCount('drops')
            ->whereIn('status', ['ready', 'sent_to_zoho', 'complete'])
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('reference', 'like', "%{$this->search}%")
                      ->orWhere('id', 'like', "%{$this->search}%")
                      ->orWhere('zoho_invoice_id', 'like', "%{$this->search}%")
                      ->orWhereHas('customer', fn($cq) => $cq->where('name', 'like', "%{$this->search}%"));
                });
            })
            ->when($this->customer, fn($q) => $q->where('customers_id', $this->customer))
            ->when($this->status, fn($q) => $q->where('status', $this->status))
            ->orderByRaw("FIELD(status, 'ready', 'sent_to_zoho', 'complete')")
            ->orderBy('created_at', 'desc');
    }
    
    /**
     * Initialize the invoice groups list
     */
    public function loadInitialInvoiceGroups()
    {
        $this->loadedInvoiceGroups = [];
        $this->hasMore = true;
        $this->loadMore();
    }
    
    /**
     * Load more invoice groups for infinite scroll
     */
    public function loadMore()
    {
        if ($this->isLoading || !$this->hasMore) {
            return;
        }
        
        $this->isLoading = true;
        
        $invoiceGroups = $this->buildInvoiceGroupsQuery()
            ->skip(count($this->loadedInvoiceGroups))
            ->take($this->perPage)
            ->get();
        
        if ($invoiceGroups->isEmpty()) {
            $this->hasMore = false;
        } else {
            // Manually serialize to ensure style and colour data is properly included
            $serializedGroups = $invoiceGroups->map(function ($invoice) {
                $data = $invoice->toArray();
                
                // Enhance drops with style and colour information
                if (isset($data['drops']) && is_array($data['drops'])) {
                    $data['drops'] = array_map(function ($dropData) use ($invoice) {
                        // Find the actual drop model to access helper methods
                        $drop = $invoice->drops->firstWhere('id', $dropData['id'] ?? null);
                        
                        if ($drop && $drop->line) {
                            // Use helper methods that check both relationship and imported data
                            $dropData['style_ref'] = $drop->line->getStyleRef();
                            $dropData['colour_name'] = $drop->line->getColourName();
                            
                            // Calculate price info - check if all sizes have the same price
                            $prices = $drop->line->quantities->pluck('price')->filter(fn($p) => $p !== null && $p !== '')->unique()->values();
                            if ($prices->count() === 1) {
                                $unitPrice = (float)$prices->first();
                                $dropData['unit_price_display'] = number_format($unitPrice, 2);
                                // Calculate total price: sum of (qty * price) for all quantities
                                $dropData['total_price'] = $drop->line->quantities->sum(function ($qty) {
                                    return ($qty->qty ?? 0) * ((float)($qty->price ?? 0));
                                });
                            } elseif ($prices->count() > 1) {
                                $dropData['unit_price_display'] = 'DIFF';
                                // Still calculate total even if unit prices differ
                                $dropData['total_price'] = $drop->line->quantities->sum(function ($qty) {
                                    return ($qty->qty ?? 0) * ((float)($qty->price ?? 0));
                                });
                            } else {
                                $dropData['unit_price_display'] = null;
                                $dropData['total_price'] = null;
                            }
                            
                            // Also ensure line data includes these for backward compatibility
                            if (isset($dropData['line'])) {
                                $dropData['line']['style_ref'] = $drop->line->getStyleRef();
                                $dropData['line']['colour_name'] = $drop->line->getColourName();
                            }
                        }
                        
                        return $dropData;
                    }, $data['drops']);
                }
                
                return $data;
            })->toArray();
            
            $this->loadedInvoiceGroups = array_merge($this->loadedInvoiceGroups, $serializedGroups);
            
            if ($invoiceGroups->count() < $this->perPage) {
                $this->hasMore = false;
            }
        }
        
        $this->isLoading = false;
    }
    
    #[Computed]
    public function stats()
    {
        return [
            'ready' => CommissionInvoiceGroup::where('status', 'ready')->count(),
            'sent_to_zoho' => CommissionInvoiceGroup::where('status', 'sent_to_zoho')->count(),
            'complete' => CommissionInvoiceGroup::where('status', 'complete')->count(),
            'total_value' => CommissionInvoiceGroup::whereIn('status', ['ready', 'sent_to_zoho'])->sum('total_value'),
        ];
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if ($this->customer) $count++;
        if ($this->status) $count++;
        return $count;
    }

    public function toggleExpanded(int $invoiceId)
    {
        if (in_array($invoiceId, $this->expanded)) {
            $this->expanded = array_values(array_diff($this->expanded, [$invoiceId]));
        } else {
            $this->expanded[] = $invoiceId;
        }
    }

    /**
     * Generate Zoho Invoice via the API
     */
    public function sendToZoho(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::with('customer')->findOrFail($invoiceId);
        
        if ($invoice->status !== 'ready') {
            $this->dispatch('notify', type: 'error', message: 'Invoice must be in "ready" status to send to Zoho.');
            return;
        }

        // Check Zoho connection
        if (!$this->zoho->isConnected()) {
            $this->dispatch('notify', type: 'error', message: 'Please connect to Zoho first.');
            return;
        }

        $orgId = config('services.commission.zoho_org_id');
        $templateId = config('services.commission.zoho_template_id');
        $accountId = config('services.commission.zoho_account_id');
        $zohoCustomerId = config('services.commission.zoho_customer_id');

        if (!$zohoCustomerId) {
            $this->dispatch('notify', type: 'error', message: 'Zoho Customer ID not configured. Please set COMMISSION_ZOHO_CUSTOMER_ID in .env');
            return;
        }

        $customerName = $invoice->customer?->name ?? 'Unknown Customer';
        $supplierInvoiceRef = $invoice->supplier_invoice_ref ?? $invoice->reference ?? 'INV-' . $invoice->id;

        // Build the invoice payload with a single line item
        $invoicePayload = [
            'customer_id' => $zohoCustomerId,
            'template_id' => $templateId,
            'reference_number' => $invoice->reference ?? 'COMM-' . $invoice->id,
            'date' => $invoice->invoice_date?->format('Y-m-d') ?? now()->format('Y-m-d'),
            'status' => 'draft',
            'line_items' => [
                [
                    'name' => "Commissions due on your invoice {$supplierInvoiceRef} for {$customerName}",
                    'description' => "Commission invoice for {$customerName}",
                    'rate' => number_format($invoice->total_value ?? 0, 2, '.', ''),
                    'quantity' => 1,
                    'account_id' => $accountId,
                ],
            ],
        ];

        try {
            $zohoInvoice = $this->zoho->createInvoice($invoicePayload, $orgId);

            if (!$zohoInvoice || empty($zohoInvoice['invoice_number'])) {
                Log::error('Zoho commission invoice creation returned unexpected response', [
                    'invoice_id' => $invoiceId,
                    'response' => $zohoInvoice,
                ]);
                $this->dispatch('notify', type: 'error', message: 'Failed to create Zoho invoice. Check logs for details.');
                return;
            }

            $zohoInvoiceNumber = $zohoInvoice['invoice_number'];
            $zohoInvoiceId = $zohoInvoice['invoice_id'] ?? null;

            $invoice->update([
                'status' => 'sent_to_zoho',
                'sent_to_zoho_at' => now(),
                'zoho_invoice_id' => $zohoInvoiceNumber,
            ]);

            Log::info('Zoho commission invoice created', [
                'commission_invoice_id' => $invoiceId,
                'zoho_invoice_number' => $zohoInvoiceNumber,
                'zoho_invoice_id' => $zohoInvoiceId,
            ]);

            $this->dispatch('notify', type: 'success', message: "Zoho invoice created: {$zohoInvoiceNumber}");
            $this->refreshInvoices();

        } catch (\Exception $e) {
            Log::error('Failed to create Zoho commission invoice', [
                'invoice_id' => $invoiceId,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            $this->dispatch('notify', type: 'error', message: 'Error: ' . $e->getMessage());
        }
    }

    /**
     * Save Zoho Invoice ID
     */
    public function saveZohoId(int $invoiceId)
    {
        $zohoId = $this->zohoIds[$invoiceId] ?? null;
        
        if (empty($zohoId)) {
            $this->dispatch('notify', type: 'error', message: 'Please enter a Zoho Invoice ID.');
            return;
        }
        
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        $invoice->update(['zoho_invoice_id' => $zohoId]);
        
        $this->dispatch('notify', type: 'success', message: 'Zoho Invoice ID saved.');
        $this->refreshInvoices();
    }

    /**
     * Mark as complete
     */
    public function markComplete(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if ($invoice->status !== 'sent_to_zoho') {
            $this->dispatch('notify', type: 'error', message: 'Invoice must be sent to Zoho before marking complete.');
            return;
        }
        
        if (empty($invoice->zoho_invoice_id)) {
            $this->dispatch('notify', type: 'error', message: 'Please enter the Zoho Invoice ID before marking complete.');
            return;
        }
        
        $invoice->update(['status' => 'complete']);
        
        $this->dispatch('notify', type: 'success', message: 'Invoice marked as complete.');
        $this->refreshInvoices();
    }

    /**
     * Open manual Zoho entry modal
     */
    public function openManualEntry(int $invoiceId)
    {
        $this->manualInvoiceId = $invoiceId;
        $this->manualZohoNumber = '';
    }

    /**
     * Close manual entry modal
     */
    public function closeManualEntry()
    {
        $this->manualInvoiceId = null;
        $this->manualZohoNumber = '';
    }

    /**
     * Mark as sent manually (without API)
     */
    public function markAsSentManually()
    {
        if (!$this->manualInvoiceId) {
            return;
        }
        
        $invoice = CommissionInvoiceGroup::findOrFail($this->manualInvoiceId);
        
        if ($invoice->status !== 'ready') {
            $this->dispatch('notify', type: 'error', message: 'Invoice must be in "ready" status.');
            return;
        }
        
        if (empty($this->manualZohoNumber)) {
            $this->dispatch('notify', type: 'error', message: 'Please enter the Zoho Invoice Number.');
            return;
        }
        
        $invoice->update([
            'status' => 'sent_to_zoho',
            'sent_to_zoho_at' => now(),
            'zoho_invoice_id' => $this->manualZohoNumber,
        ]);
        
        $this->dispatch('notify', type: 'success', message: 'Invoice marked as sent to Zoho.');
        $this->closeManualEntry();
        $this->refreshInvoices();
    }

    /**
     * Revert to ready (undo send to zoho)
     */
    public function revertToReady(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if ($invoice->status !== 'sent_to_zoho') {
            $this->dispatch('notify', type: 'error', message: 'Only invoices sent to Zoho can be reverted.');
            return;
        }
        
        $invoice->update([
            'status' => 'ready',
            'sent_to_zoho_at' => null,
            'zoho_invoice_id' => null,
        ]);
        
        $this->dispatch('notify', type: 'success', message: 'Invoice reverted to ready status.');
        $this->refreshInvoices();
    }

    public function render()
    {
        return view('livewire.commission.sl-invoicing')->layout('layouts.commission');
    }
}



