<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\Attributes\Url;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;
use App\Models\CommissionOrder;
use App\Models\CommissionImportBatch;
use App\Models\User;
use App\Models\Seasons;
use App\Models\Customer;
use App\Notifications\SkeletonOrderForwardedToPd;
use App\Notifications\SkeletonOrderApprovedByPd;

class SkeletonOrders extends Component
{
    #[Url]
    public string $search = '';

    #[Url]
    public string $status = '';

    #[Url]
    public string $customer = '';

    #[Url]
    public ?int $batch = null;

    #[Url]
    public ?int $order = null;

    #[Url]
    public string $sortBy = 'imported_at';

    #[Url]
    public string $sortDir = 'desc';

    // Infinite scroll
    public int $perPage = 25;
    public array $loadedOrders = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    // For forward to PD modal
    public bool $showForwardModal = false;
    public ?int $forwardOrderId = null;
    public ?int $forwardToUserId = null;
    public string $forwardNotes = '';
    public bool $forwardSelfCertified = false;

    // For approval modal
    public bool $showApprovalModal = false;
    public ?int $approvalOrderId = null;
    public string $approvalType = ''; // 'pd' or 'merch'
    public string $approvalNotes = '';

    public function mount()
    {
        $this->loadInitialOrders();
    }

    public function loadInitialOrders()
    {
        $this->loadedOrders = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    public function updatedSearch()
    {
        $this->loadInitialOrders();
    }

    public function updatedStatus()
    {
        $this->loadInitialOrders();
    }

    public function updatedCustomer()
    {
        $this->loadInitialOrders();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'status', 'customer', 'batch', 'order']);
        $this->loadInitialOrders();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'desc';
        }
        $this->loadInitialOrders();
    }

    protected function buildQuery()
    {
        return CommissionOrder::query()
            ->with([
                'customer:id,name',
                'season:id,description',
                'importedBy:id,name',
                'pdReviewedBy:id,name',
                'confirmedBy:id,name',
            ])
            ->withCount([
                'lines as total_lines' => fn($q) => $q->where('cancelled', false),
                'lines as matched_lines' => fn($q) => $q->where('cancelled', false)->whereNotNull('colourways_id'),
                'lines as unmatched_lines' => fn($q) => $q->where('cancelled', false)->whereNull('colourways_id'),
            ])
            // Only show skeleton/pending orders (not confirmed)
            ->whereIn('status', ['skeleton', 'pending_pd', 'pending_merch'])
            ->where('cancelled', false)
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('customer_po', 'like', "%{$this->search}%")
                      ->orWhere('id', 'like', "%{$this->search}%")
                      ->orWhere('source_file', 'like', "%{$this->search}%")
                      ->orWhereHas('customer', fn($cq) => $cq->where('name', 'like', "%{$this->search}%"));
                });
            })
            ->when($this->status, fn($q) => $q->where('status', $this->status))
            ->when($this->customer, fn($q) => $q->where('customers_id', $this->customer))
            ->when($this->batch, function ($q) {
                $q->whereHas('importBatch', fn($bq) => $bq->where('id', $this->batch));
            })
            ->when($this->order, fn($q) => $q->where('id', $this->order))
            ->orderBy($this->sortBy, $this->sortDir);
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedOrders)->pluck('id')->toArray();
        
        $query = $this->buildQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('commission_orders.id', $loadedIds);
        }

        $newOrders = $query->limit($this->perPage)->get();

        if ($newOrders->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newOrders as $order) {
            $matchPercent = $order->total_lines > 0 
                ? (int) round(($order->matched_lines / $order->total_lines) * 100) 
                : 0;
            
            $this->loadedOrders[] = [
                'id' => $order->id,
                'customer_po' => $order->customer_po,
                'source_file' => $order->source_file,
                'source_files' => $order->source_files,
                'customer_name' => $order->customer?->name,
                'season_description' => $order->season?->description,
                'status' => $order->status,
                'total_lines' => $order->total_lines,
                'matched_lines' => $order->matched_lines,
                'unmatched_lines' => $order->unmatched_lines,
                'match_percent' => $matchPercent,
                'imported_at' => $order->imported_at?->format('d M Y H:i'),
                'imported_at_diff' => $order->imported_at?->diffForHumans(),
                'pd_reviewed_by_name' => $order->pdReviewedBy?->name,
                'pd_reviewed_by_id' => $order->pd_reviewed_by,
            ];
        }

        $this->isLoading = false;
    }

    #[Computed]
    public function stats()
    {
        $baseQuery = CommissionOrder::where('cancelled', false)
            ->whereIn('status', ['skeleton', 'pending_pd', 'pending_merch']);

        return [
            'total' => (clone $baseQuery)->count(),
            'skeleton' => (clone $baseQuery)->where('status', 'skeleton')->count(),
            'pending_pd' => (clone $baseQuery)->where('status', 'pending_pd')->count(),
            'pending_merch' => (clone $baseQuery)->where('status', 'pending_merch')->count(),
        ];
    }

    #[Computed]
    public function customers()
    {
        return Customer::whereHas('commissionOrders', function ($q) {
            $q->whereIn('status', ['skeleton', 'pending_pd', 'pending_merch']);
        })->orderBy('name')->get();
    }

    #[Computed]
    public function pdUsers()
    {
        // Get users with PD or Admin role
        return User::whereHas('functionalRoles', fn($q) => $q->whereIn('name', ['pd', 'admin']))
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if ($this->status) $count++;
        if ($this->customer) $count++;
        if ($this->batch) $count++;
        if ($this->order) $count++;
        return $count;
    }

    // ==================
    // Forward to PD
    // ==================

    public function openForwardModal(int $orderId)
    {
        $this->forwardOrderId = $orderId;
        $this->forwardToUserId = null;
        $this->forwardNotes = '';
        $this->forwardSelfCertified = false;
        $this->showForwardModal = true;
    }

    public function closeForwardModal()
    {
        $this->showForwardModal = false;
        $this->forwardOrderId = null;
        $this->forwardSelfCertified = false;
    }

    public function forwardToPd()
    {
        $this->validate([
            'forwardSelfCertified' => 'nullable|boolean',
        ]);

        if (!$this->forwardSelfCertified) {
            $this->validate([
                'forwardToUserId' => 'required|exists:users,id',
            ]);
        }

        $order = CommissionOrder::findOrFail($this->forwardOrderId);
        
        if ($order->status !== 'skeleton') {
            $this->dispatch('notify', type: 'error', message: 'Order must be in skeleton status.');
            return;
        }

        $forwardToUserId = $this->forwardSelfCertified ? auth()->id() : $this->forwardToUserId;

        $order->update([
            'status' => 'pending_pd',
            'pd_email_sent_at' => now(),
            'forwarded_to_user_id' => $forwardToUserId,
            'forward_notes' => $this->forwardNotes ?: null,
        ]);

        // Update batch if exists
        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();
        if ($batch) {
            $batch->forwardToPd($forwardToUserId, $this->forwardNotes);
        }

        // Send email notification to PD user
        $pdUser = User::find($forwardToUserId);
        if ($pdUser) {
            $pdUser->notify(new SkeletonOrderForwardedToPd($order, auth()->user(), $this->forwardNotes));
        }

        $this->closeForwardModal();
        $this->dispatch('notify', type: 'success', message: 'Order forwarded to ' . ($pdUser?->name ?? 'PD') . '.');
    }

    // ==================
    // Approvals
    // ==================

    public function openApprovalModal(int $orderId, string $type)
    {
        $this->approvalOrderId = $orderId;
        $this->approvalType = $type;
        $this->approvalNotes = '';
        $this->showApprovalModal = true;
    }

    public function closeApprovalModal()
    {
        $this->showApprovalModal = false;
        $this->approvalOrderId = null;
    }

    public function submitApproval()
    {
        $order = CommissionOrder::findOrFail($this->approvalOrderId);
        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();

        if ($this->approvalType === 'pd') {
            if ($order->status !== 'pending_pd') {
                $this->dispatch('notify', type: 'error', message: 'Order must be pending PD review.');
                return;
            }

            $order->update([
                'status' => 'pending_merch',
                'pd_reviewed_by' => auth()->id(),
                'pd_reviewed_at' => now(),
                'pd_notes' => $this->approvalNotes,
            ]);

            if ($batch) {
                $batch->approvePd($this->approvalNotes);
            }

            // Notify the person who imported/forwarded the order
            $originalUser = $order->importedBy;
            if ($originalUser) {
                $originalUser->notify(new SkeletonOrderApprovedByPd($order, auth()->user(), $this->approvalNotes));
            }

            $this->dispatch('notify', type: 'success', message: 'PD approval submitted. Order sent to Merch.');

        } elseif ($this->approvalType === 'merch') {
            if ($order->status !== 'pending_merch') {
                $this->dispatch('notify', type: 'error', message: 'Order must be pending Merch approval.');
                return;
            }

            // Check if all active lines have commission percentage set
            $linesWithoutCommission = $order->lines()
                ->where('cancelled', false)
                ->where(function ($q) {
                    $q->whereNull('commission_percent')
                      ->orWhere('commission_percent', '<=', 0);
                })
                ->count();
            if ($linesWithoutCommission > 0) {
                $this->dispatch('notify', type: 'error', message: "Cannot confirm: {$linesWithoutCommission} line(s) missing Commission %. Edit the order to set it on each line.");
                return;
            }

            // Check if all lines are matched
            $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
            if ($unmatchedCount > 0) {
                $this->dispatch('notify', type: 'error', message: "Cannot confirm: {$unmatchedCount} lines still unmatched.");
                return;
            }

            $order->update([
                'status' => 'confirmed',
                'confirmed_by' => auth()->id(),
                'confirmed_at' => now(),
                'merch_notes' => $this->approvalNotes,
            ]);

            if ($batch) {
                $batch->approveMerch($this->approvalNotes);
            }

            $this->dispatch('notify', type: 'success', message: 'Order confirmed! It is now visible in Orders and Shipment Schedule.');
        }

        $this->closeApprovalModal();
    }

    // ==================
    // Quick Actions
    // ==================

    public function viewOrder(int $orderId)
    {
        $this->dispatch('open-order-edit', orderId: $orderId);
    }

    public function deleteOrder(int $orderId)
    {
        $order = CommissionOrder::findOrFail($orderId);
        
        if ($order->status === 'confirmed') {
            $this->dispatch('notify', type: 'error', message: 'Cannot delete confirmed orders.');
            return;
        }

        $order->update(['cancelled' => true]);
        $this->dispatch('notify', type: 'success', message: 'Order deleted.');
    }

    #[On('order-saved')]
    public function refresh()
    {
        unset($this->stats);
        $this->loadInitialOrders();
    }

    public function render()
    {
        return view('livewire.commission.skeleton-orders')->layout('layouts.commission');
    }
}


