<?php

namespace App\Http\Livewire\Commission;

use App\Models\CommissionDrop;
use App\Models\CommissionInvoiceGroup;
use App\Models\Customer;
use App\Models\Seasons;
use Livewire\Component;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Carbon;

class ShipmentSchedule extends Component
{

    // ==================
    // Filters
    // ==================

    #[Url]
    public string $view = 'unshipped'; // unshipped, shipped, all, financeNC, overdue, needsInvoice

    #[Url]
    public string $search = '';
    
    // Expanded invoice groups (for shipped view)
    public array $expandedGroups = [];

    #[Url]
    public string $fromDate = '';

    #[Url]
    public string $toDate = '';

    #[Url]
    public array $customers = [];

    #[Url]
    public array $seasons = [];

    #[Url]
    public string $shipped = '';

    #[Url]
    public string $invoiceStatus = '';

    #[Url]
    public string $sortBy = 'exfty';

    #[Url]
    public string $sortDir = 'asc';

    // ==================
    // Selection
    // ==================

    public array $selected = [];
    public bool $selectAll = false;

    // ==================
    // Infinite Scroll
    // ==================

    public int $perPage = 50;
    public array $loadedDrops = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    // ==================
    // Inline Editing State
    // ==================

    public ?int $editingDropId = null;
    public ?string $editingField = null;
    
    // ==================
    // Invoice Group Modal
    // ==================
    
    public bool $showInvoiceModal = false;
    public ?int $selectedInvoiceGroup = null;
    public string $newInvoiceRef = '';

    // ==================
    // Lifecycle
    // ==================

    public function mount()
    {
        // Default date range: last 30 days to next 90 days
        // Only set defaults if dates are not present in the URL at all
        // This allows users to explicitly clear dates and have them stay cleared
        if (!request()->has('fromDate') && empty($this->fromDate)) {
            $this->fromDate = now()->subDays(30)->format('Y-m-d');
        }
        if (!request()->has('toDate') && empty($this->toDate)) {
            $this->toDate = now()->addDays(90)->format('Y-m-d');
        }
        $this->loadInitialDrops();
    }

    public function loadInitialDrops()
    {
        $this->loadedDrops = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedDrops)->pluck('id')->toArray();
        
        $query = $this->buildDropsQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('commission_drops.id', $loadedIds);
        }

        // Apply sorting
        $sortDirection = $this->sortDir === 'asc' ? 'ASC' : 'DESC';
        
        if ($this->sortBy === 'exfty') {
            $query->selectRaw('commission_drops.*, 
                COALESCE(
                    (SELECT MIN(cd2.exfty) 
                     FROM commission_drops cd2 
                     WHERE cd2.commission_invoice_groups_id = commission_drops.commission_invoice_groups_id 
                     AND cd2.commission_invoice_groups_id IS NOT NULL
                     AND cd2.deleted_at IS NULL),
                    commission_drops.exfty
                ) as group_sort_date')
                ->orderByRaw("group_sort_date {$sortDirection}")
                ->orderByRaw('COALESCE(commission_invoice_groups_id, 999999999) ASC')
                ->orderBy('exfty', $this->sortDir);
        } else {
            $query->orderByRaw('COALESCE(commission_invoice_groups_id, 999999999) ASC')
                  ->orderBy($this->sortBy, $this->sortDir);
        }

        $newDrops = $query->limit($this->perPage)->get();

        if ($newDrops->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newDrops as $drop) {
            $order = $drop->line?->order;
            $colourway = $drop->line?->colourway;
            $style = $colourway?->style_versions?->styles;
            $design = $style?->designs;
            $totalQty = $drop->sizes->sum('qty');
            $totalShipped = $drop->sizes->sum('shipped_qty') ?: 0;
            $isOverdue = !$drop->shipped && $drop->exfty && Carbon::parse($drop->exfty)->lt(now());
            
            // Calculate price info - check if all sizes have the same price
            $unitPriceDisplay = null;
            $totalPrice = null;
            if ($drop->line && $drop->line->quantities) {
                $prices = $drop->line->quantities->pluck('price')->filter(fn($p) => $p !== null && $p !== '')->unique()->values();
                if ($prices->count() === 1) {
                    $unitPrice = (float)$prices->first();
                    $unitPriceDisplay = number_format($unitPrice, 2);
                    // Calculate total price: sum of (qty * price) for all quantities
                    $totalPrice = $drop->line->quantities->sum(function ($qty) {
                        return ($qty->qty ?? 0) * ((float)($qty->price ?? 0));
                    });
                } elseif ($prices->count() > 1) {
                    $unitPriceDisplay = 'DIFF';
                    // Still calculate total even if unit prices differ
                    $totalPrice = $drop->line->quantities->sum(function ($qty) {
                        return ($qty->qty ?? 0) * ((float)($qty->price ?? 0));
                    });
                }
            }
            
            $this->loadedDrops[] = [
                'id' => $drop->id,
                'exfty' => $drop->exfty?->format('Y-m-d'),
                'exfty_formatted' => $drop->exfty?->format('d M Y'),
                'shipped' => $drop->shipped,
                'shipped_date' => $drop->shipped_date?->format('d M Y'),
                'fn_notes' => $drop->fn_notes,
                'commission_invoice_groups_id' => $drop->commission_invoice_groups_id,
                'invoice_ref' => $drop->invoiceGroup?->reference,
                'total_qty' => $totalQty,
                'total_shipped' => $totalShipped,
                'season_description' => $order?->season?->description,
                'customer_name' => $order?->customer?->name,
                'customer_po' => $order?->customer_po,
                'customer_currency' => $order?->customer?->currency ?? '$',
                'design_id' => $design?->id,
                'design_description' => $design?->description,
                'customer_ref' => $style?->customer_ref,
                'colourway_name' => $colourway?->name,
                'colourway_thumb_url' => $colourway?->thumb_url,
                'is_overdue' => $isOverdue,
                'status_class' => $drop->shipped ? 'shipped' : ($isOverdue ? 'overdue' : 'pending'),
                'unit_price_display' => $unitPriceDisplay,
                'total_price' => $totalPrice,
                'sizes' => $drop->sizes->sortBy('size.order')->unique('sizes_id')->filter(fn($s) => $s->size)->map(function ($s) use ($drop) {
                    // Find matching quantity to get price
                    $quantity = $drop->line?->quantities->firstWhere('sizes_id', $s->sizes_id);
                    $price = $quantity ? (float)($quantity->price ?? 0) : null;
                    
                    return [
                        'id' => $s->id,
                        'name' => $s->size->name,
                        'qty' => (int)$s->qty,
                        'shipped_qty' => (int)($s->shipped_qty ?? 0),
                        'price' => $price,
                        'total_price' => $price !== null ? ($s->qty * $price) : null,
                    ];
                })->values()->toArray(),
            ];
        }

        $this->isLoading = false;
    }

    // ==================
    // Computed Properties
    // ==================

    /**
     * Build the base query for drops
     */
    private function buildDropsQuery()
    {
        $query = CommissionDrop::query()
            ->with([
                'line.order.customer:id,name',
                'line.order.season:id,description',
                'line.colourway:id,name,img_thumb,style_versions_id',
                'line.colourway.style_versions:id,styles_id',
                'line.colourway.style_versions.styles:id,designs_id,customer_ref',
                'line.colourway.style_versions.styles.designs:id,description',
                'line.quantities.size:id,name,order',
                'invoiceGroup:id,reference,status,customers_id',
                'invoiceGroup.customer:id,name',
                'sizes.size:id,name,order',
            ])
            ->whereHas('line.order', function ($q) {
                $q->where('cancelled', false)
                  ->where('status', 'confirmed');
            })
            ->whereHas('line', function ($q) {
                $q->where('cancelled', false);
            });

        // View filter
        if ($this->view === 'unshipped') {
            $query->where('shipped', false);
        } elseif ($this->view === 'shipped') {
            $query->where('shipped', true);
        } elseif ($this->view === 'financeNC') {
            $query->where('invoiced', false);
        } elseif ($this->view === 'overdue') {
            $query->where('shipped', false)
                  ->whereNotNull('exfty')
                  ->where('exfty', '<', now()->toDateString());
        } elseif ($this->view === 'needsInvoice') {
            $query->where('shipped', true)
                  ->where('invoiced', false)
                  ->where(function ($q) {
                      $q->whereNull('rt_invoice')
                        ->orWhere('rt_invoice', '');
                  });
        }

        // Date range filter (skip for needsInvoice view - show all ready for invoicing)
        if ($this->view !== 'needsInvoice') {
        if ($this->fromDate) {
            $query->where('exfty', '>=', $this->fromDate);
        }
        if ($this->toDate) {
            $query->where('exfty', '<=', $this->toDate);
            }
        }

        // Customer filter
        if (!empty($this->customers)) {
            $query->whereHas('line.order', function ($q) {
                $q->whereIn('customers_id', $this->customers);
            });
        }

        // Season filter
        if (!empty($this->seasons)) {
            $query->whereHas('line.order', function ($q) {
                $q->whereIn('seasons_id', $this->seasons);
            });
        }

        // Shipped filter
        if ($this->shipped === 'shipped') {
            $query->where('shipped', true);
        } elseif ($this->shipped === 'unshipped') {
            $query->where('shipped', false);
        }

        // Invoice status filter
        if ($this->invoiceStatus === 'has_invoice') {
            $query->whereNotNull('commission_invoice_groups_id');
        } elseif ($this->invoiceStatus === 'no_invoice') {
            $query->whereNull('commission_invoice_groups_id');
        }

        // Search
        if ($this->search) {
            $searchTerm = '%' . $this->search . '%';
            $query->where(function ($q) use ($searchTerm) {
                $q->where('id', 'like', $searchTerm)
                  ->orWhereHas('line.order', function ($oq) use ($searchTerm) {
                      $oq->where('customer_po', 'like', $searchTerm);
                  })
                  ->orWhereHas('line.colourway.style_versions.styles', function ($sq) use ($searchTerm) {
                      $sq->where('customer_ref', 'like', $searchTerm);
                  })
                  ->orWhereHas('line.colourway.style_versions.styles.designs', function ($dq) use ($searchTerm) {
                      $dq->where('id', 'like', $searchTerm)
                         ->orWhere('description', 'like', $searchTerm);
                  })
                  ->orWhereHas('invoiceGroup', function ($iq) use ($searchTerm) {
                      $iq->where('reference', 'like', $searchTerm);
                  });
            });
        }

        return $query;
    }

    /**
     * Get drops organized for display with group info
     */
    #[Computed]
    public function organizedDrops()
    {
        $organized = [];
        $seenInvoices = [];
        $lastInvoiceId = null;
        $groupIndex = 0;
        
        foreach ($this->loadedDrops as $drop) {
            $invoiceId = $drop['commission_invoice_groups_id'];
            
            // Track when invoice group changes for alternating colors
            if ($invoiceId !== $lastInvoiceId) {
                $groupIndex++;
                $lastInvoiceId = $invoiceId;
            }
            
            // Only show header if we haven't seen this invoice before
            $showHeader = false;
            if ($invoiceId && !in_array($invoiceId, $seenInvoices)) {
                $showHeader = true;
                $seenInvoices[] = $invoiceId;
            }
            
            $organized[] = [
                'drop' => $drop,
                'groupIndex' => $groupIndex,
                'invoiceId' => $invoiceId,
                'isFirstInGroup' => $showHeader,
                'invoiceRef' => $drop['invoice_ref'],
            ];
        }
        
        return $organized;
    }

    /**
     * Check if we should show grouped view (shipped view)
     */
    #[Computed]
    public function showGroupedView(): bool
    {
        return $this->view === 'shipped';
    }

    /**
     * Get drops grouped by invoice for shipped view
     */
    #[Computed]
    public function groupedDrops()
    {
        if (!$this->showGroupedView) {
            return collect();
        }

        $query = $this->buildDropsQuery();
        
        // For shipped view, order by invoice group first
        $query->orderByRaw('COALESCE(commission_invoice_groups_id, 999999999) ASC')
              ->orderBy('shipped_date', 'desc')
              ->orderBy($this->sortBy, $this->sortDir);

        $drops = $query->get();

        // Group by invoice
        $grouped = $drops->groupBy(function ($drop) {
            return $drop->commission_invoice_groups_id ?? 'unassigned';
        });

        // Transform to include invoice info
        return $grouped->map(function ($drops, $groupKey) {
            $invoiceGroup = $groupKey !== 'unassigned' ? $drops->first()->invoiceGroup : null;
            
            return [
                'id' => $groupKey,
                'invoice' => $invoiceGroup,
                'reference' => $invoiceGroup?->reference ?? 'No Invoice Assigned',
                'customer' => $invoiceGroup?->customer?->name ?? $drops->first()->line?->order?->customer?->name ?? 'Unknown',
                'status' => $invoiceGroup?->status ?? 'none',
                'drops' => $drops,
                'total_qty' => $drops->sum(fn($d) => $d->sizes->sum('qty')),
                'total_shipped' => $drops->sum(fn($d) => $d->sizes->sum('shipped_qty') ?: $d->sizes->sum('qty')),
                'drop_count' => $drops->count(),
            ];
        })->sortByDesc(function ($group) {
            // Sort: invoices with reference first, then unassigned
            return $group['id'] === 'unassigned' ? 0 : 1;
        });
    }

    /**
     * Toggle expanded state for an invoice group
     */
    public function toggleGroup($groupId)
    {
        if (in_array($groupId, $this->expandedGroups)) {
            $this->expandedGroups = array_values(array_diff($this->expandedGroups, [$groupId]));
        } else {
            $this->expandedGroups[] = $groupId;
        }
    }

    /**
     * Expand all groups
     */
    public function expandAllGroups()
    {
        $this->expandedGroups = $this->groupedDrops->pluck('id')->map(fn($id) => (string)$id)->toArray();
    }

    /**
     * Collapse all groups
     */
    public function collapseAllGroups()
    {
        $this->expandedGroups = [];
    }

    #[Computed]
    public function customerList()
    {
        return Customer::whereHas('commissionOrders', function ($q) {
            $q->where('cancelled', false)->where('status', 'confirmed');
        })
        ->orderBy('name')
        ->get(['id', 'name']);
    }

    #[Computed]
    public function seasonList()
    {
        return Seasons::whereHas('commissionOrders', function ($q) {
            $q->where('cancelled', false)->where('status', 'confirmed');
        })
        ->orderBy('created_at', 'desc')
        ->get(['id', 'description']);
    }

    #[Computed]
    public function invoiceGroups()
    {
        return CommissionInvoiceGroup::whereIn('status', ['draft', 'ready'])
            ->orderBy('created_at', 'desc')
            ->get(['id', 'reference', 'status']);
    }

    #[Computed]
    public function stats()
    {
        $baseQuery = CommissionDrop::whereHas('line.order', function ($q) {
            $q->where('cancelled', false)->where('status', 'confirmed');
        })->whereHas('line', fn($q) => $q->where('cancelled', false));

        // Apply date filters for stats
        if ($this->fromDate) {
            $baseQuery->where('exfty', '>=', $this->fromDate);
        }
        if ($this->toDate) {
            $baseQuery->where('exfty', '<=', $this->toDate);
        }

        return [
            'total' => (clone $baseQuery)->count(),
            'unshipped' => (clone $baseQuery)->where('shipped', false)->count(),
            'shipped' => (clone $baseQuery)->where('shipped', true)->count(),
            'overdue' => (clone $baseQuery)
                ->where('shipped', false)
                ->where('exfty', '<', now()->format('Y-m-d'))
                ->count(),
        ];
    }

    #[Computed]
    public function selectedCount()
    {
        return count(array_filter($this->selected));
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if (!empty($this->customers)) $count++;
        if (!empty($this->seasons)) $count++;
        if ($this->shipped) $count++;
        if ($this->invoiceStatus) $count++;
        return $count;
    }

    // ==================
    // Filter Actions
    // ==================

    public function updatedSearch()
    {
        $this->loadInitialDrops();
    }

    public function updatedView()
    {
        $this->loadInitialDrops();
    }

    public function updatedFromDate()
    {
        $this->loadInitialDrops();
    }

    public function updatedToDate()
    {
        $this->loadInitialDrops();
    }

    public function toggleCustomer(int $customerId)
    {
        if (in_array($customerId, $this->customers)) {
            $this->customers = array_values(array_diff($this->customers, [$customerId]));
        } else {
            $this->customers[] = $customerId;
        }
        $this->loadInitialDrops();
    }

    public function toggleSeason(int $seasonId)
    {
        if (in_array($seasonId, $this->seasons)) {
            $this->seasons = array_values(array_diff($this->seasons, [$seasonId]));
        } else {
            $this->seasons[] = $seasonId;
        }
        $this->loadInitialDrops();
    }

    public function clearFilters()
    {
        $this->reset([
            'search', 'customers', 'seasons', 'shipped', 'invoiceStatus'
        ]);
        $this->fromDate = now()->subDays(30)->format('Y-m-d');
        $this->toDate = now()->addDays(90)->format('Y-m-d');
        $this->loadInitialDrops();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'asc';
        }
        $this->loadInitialDrops();
    }

    // ==================
    // Selection Actions
    // ==================

    public function toggleSelectAll()
    {
        if ($this->selectAll) {
            // Select all visible drops
            foreach ($this->loadedDrops as $drop) {
                $this->selected[$drop['id']] = true;
            }
        } else {
            $this->selected = [];
        }
    }

    public function toggleSelect(int $dropId)
    {
        if (isset($this->selected[$dropId]) && $this->selected[$dropId]) {
            unset($this->selected[$dropId]);
        } else {
            $this->selected[$dropId] = true;
        }
        
        // Update selectAll state
        $this->selectAll = count(array_filter($this->selected)) === count($this->loadedDrops);
    }

    public function clearSelection()
    {
        $this->selected = [];
        $this->selectAll = false;
    }

    // ==================
    // Inline Editing
    // ==================

    public function startEditing(int $dropId, string $field)
    {
        $this->editingDropId = $dropId;
        $this->editingField = $field;
    }

    public function stopEditing()
    {
        $this->editingDropId = null;
        $this->editingField = null;
    }

    #[Renderless]
    public function updateExfty(int $dropId, ?string $value)
    {
        $validated = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|date']
        )->validate();

        if (Gate::check('order:update')) {
            CommissionDrop::where('id', $dropId)->update([
                'exfty' => $validated['value'],
                'updated_at' => now(),
            ]);
            $this->dispatch('notify', type: 'success', message: 'Ex-factory date updated');
        }
    }

    public function updateShipped(int $dropId, bool $value)
    {
        if (Gate::check('order:update')) {
            // If marking as shipped, check that there are shipped quantities
            if ($value) {
                $hasShippedQty = DB::table('commission_drop_sizes')
                    ->where('commission_drops_id', $dropId)
                    ->where('shipped_qty', '>', 0)
                    ->exists();
                
                if (!$hasShippedQty) {
                    // Dispatch event to reset the specific checkbox
                    $this->dispatch('shipped-reset-' . $dropId);
                    $this->dispatch('notify', type: 'error', message: 'Cannot mark as shipped - enter shipped quantities first');
                    return;
                }
            }
            
            CommissionDrop::where('id', $dropId)->update([
                'shipped' => $value,
                'shipped_date' => $value ? now()->format('Y-m-d') : null,
                'updated_at' => now(),
            ]);
            
            // Update the local loaded drops array to reflect the change
            foreach ($this->loadedDrops as &$drop) {
                if ($drop['id'] === $dropId) {
                    $drop['shipped'] = $value;
                    $drop['shipped_date'] = $value ? now()->format('d M Y') : null;
                    $drop['status_class'] = $value ? 'shipped' : ($drop['is_overdue'] ? 'overdue' : 'pending');
                    break;
                }
            }
            unset($drop);
            
            // Clear cached computed property
            unset($this->organizedDrops);
            
            // Update the checkbox state via event
            $this->dispatch($value ? 'shipped-set-' . $dropId : 'shipped-reset-' . $dropId);
            $this->dispatch('notify', type: 'success', message: $value ? 'Marked as shipped' : 'Marked as unshipped');
        }
    }

    #[Renderless]
    public function updateFnNotes(int $dropId, ?string $value)
    {
        $validated = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|string|max:500']
        )->validate();

        if (Gate::check('order:update') || Gate::check('finance:update')) {
            CommissionDrop::where('id', $dropId)->update([
                'fn_notes' => $validated['value'],
                'updated_at' => now(),
            ]);
            $this->dispatch('notify', type: 'success', message: 'Notes updated');
        }
    }

    #[Renderless]
    public function updateDropSizeQty(int $sizeId, $value)
    {
        $validated = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            DB::table('commission_drop_sizes')
                ->where('id', $sizeId)
                ->update([
                    'qty' => empty($validated['value']) ? 0 : $validated['value'],
                    'updated_at' => now(),
                ]);
        }
    }

    #[Renderless]
    public function updateShippedQty(int $sizeId, $value)
    {
        $validated = Validator::make(
            ['value' => $value],
            ['value' => 'nullable|numeric|min:0']
        )->validate();

        if (Gate::check('order:update')) {
            DB::table('commission_drop_sizes')
                ->where('id', $sizeId)
                ->update([
                    'shipped_qty' => empty($validated['value']) ? 0 : $validated['value'],
                    'updated_at' => now(),
                ]);
        }
    }

    // ==================
    // Bulk Actions
    // ==================

    public function markSelectedShipped()
    {
        $selectedIds = array_keys(array_filter($this->selected));
        
        if (empty($selectedIds)) {
            $this->dispatch('notify', type: 'error', message: 'No drops selected');
            return;
        }

        if (Gate::check('order:update')) {
            // Only mark drops that have shipped quantities > 0
            $dropsWithShippedQty = CommissionDrop::whereIn('id', $selectedIds)
                ->whereHas('sizes', function ($q) {
                    $q->where('shipped_qty', '>', 0);
                })
                ->pluck('id')
                ->toArray();
            
            if (empty($dropsWithShippedQty)) {
                $this->dispatch('notify', type: 'error', message: 'Cannot mark as shipped - no shipped quantities entered');
                return;
            }
            
            $skipped = count($selectedIds) - count($dropsWithShippedQty);
            
            CommissionDrop::whereIn('id', $dropsWithShippedQty)->update([
                'shipped' => true,
                'shipped_date' => now()->format('Y-m-d'),
                'updated_at' => now(),
            ]);
            
            $this->clearSelection();
            $this->loadInitialDrops();
            
            $message = count($dropsWithShippedQty) . ' drops marked as shipped';
            if ($skipped > 0) {
                $message .= " ($skipped skipped - no shipped qty)";
            }
            $this->dispatch('notify', type: 'success', message: $message);
        }
    }

    public function openInvoiceModal()
    {
        if ($this->selectedCount === 0) {
            $this->dispatch('notify', type: 'error', message: 'No drops selected');
            return;
        }
        $this->showInvoiceModal = true;
    }

    public function closeInvoiceModal()
    {
        $this->showInvoiceModal = false;
        $this->selectedInvoiceGroup = null;
        $this->newInvoiceRef = '';
    }

    public function assignToInvoiceGroup()
    {
        $selectedIds = array_keys(array_filter($this->selected));
        
        if (empty($selectedIds)) {
            $this->dispatch('notify', type: 'error', message: 'No drops selected');
            return;
        }

        $invoiceGroupId = $this->selectedInvoiceGroup;

        // Create new invoice group if needed
        if (!$invoiceGroupId && $this->newInvoiceRef) {
            // Get customer from first selected drop
            $firstDrop = CommissionDrop::with('line.order')->find($selectedIds[0]);
            $customerId = $firstDrop?->line?->order?->customers_id;
            
            if (!$customerId) {
                $this->dispatch('notify', type: 'error', message: 'Could not determine customer for invoice group');
                return;
            }
            
            $invoiceGroup = CommissionInvoiceGroup::create([
                'customers_id' => $customerId,
                'reference' => $this->newInvoiceRef,
                'status' => 'draft',
            ]);
            $invoiceGroupId = $invoiceGroup->id;
        }

        if (!$invoiceGroupId) {
            $this->dispatch('notify', type: 'error', message: 'Please select or create an invoice group');
            return;
        }

        if (Gate::check('order:update')) {
            CommissionDrop::whereIn('id', $selectedIds)->update([
                'commission_invoice_groups_id' => $invoiceGroupId,
                'updated_at' => now(),
            ]);
            
            $this->closeInvoiceModal();
            $this->clearSelection();
            $this->dispatch('notify', type: 'success', message: count($selectedIds) . ' drops assigned to invoice group');
        }
    }

    // ==================
    // Export
    // ==================

    public function exportCsv()
    {
        // This would be implemented with a controller route
        $this->dispatch('notify', type: 'info', message: 'Export feature coming soon');
    }

    // ==================
    // Helpers
    // ==================

    public function getDropStatus(CommissionDrop $drop): array
    {
        $isOverdue = !$drop->shipped && $drop->exfty && Carbon::parse($drop->exfty)->lt(now());
        
        return [
            'class' => $drop->shipped ? 'shipped' : ($isOverdue ? 'overdue' : 'pending'),
            'label' => $drop->shipped ? 'Shipped' : ($isOverdue ? 'Overdue' : 'Pending'),
        ];
    }

    /**
     * Refresh the view
     */
    #[On('refresh-schedule')]
    public function refreshSchedule()
    {
        unset($this->stats);
        $this->loadInitialDrops();
    }

    // ==================
    // Render
    // ==================

    public function render()
    {
        Gate::authorize('shipment:read');

        return view('livewire.commission.shipment-schedule')
            ->layout('layouts.commission');
    }
}

