<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\Samples as SampleModel;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\SampleTypes;
use App\Models\Departments;
use Livewire\Attributes\Computed;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\Auth;

class Samples extends Component
{
    public string $search = '';
    public $season = '';
    public $customer = '';
    public $sampleType = '';
    public $status = '';
    public string $sortBy = 'date_sent';
    public string $sortDir = 'desc';
    
    // Infinite scroll
    public int $perPage = 30;
    public array $loadedSamples = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    // Selection for bulk edit
    public array $selected = [];
    public bool $selectAll = false;

    protected $queryString = [
        'search' => ['except' => ''],
        'season' => ['except' => ''],
        'customer' => ['except' => ''],
        'sampleType' => ['except' => ''],
        'status' => ['except' => ''],
    ];

    public function mount()
    {
        $this->loadInitialSamples();
    }

    public function loadInitialSamples()
    {
        $this->loadedSamples = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    public function updatedSearch()
    {
        $this->loadInitialSamples();
    }

    public function updatedSeason()
    {
        $this->loadInitialSamples();
    }

    public function updatedCustomer()
    {
        $this->loadInitialSamples();
    }

    public function updatedSampleType()
    {
        $this->loadInitialSamples();
    }

    public function updatedStatus()
    {
        $this->loadInitialSamples();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'asc';
        }
        $this->loadInitialSamples();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'season', 'customer', 'sampleType', 'status']);
        $this->loadInitialSamples();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function sampleTypes()
    {
        return SampleTypes::orderBy('name')->get();
    }

    #[Computed]
    public function activeFiltersCount()
    {
        return collect([$this->season, $this->customer, $this->sampleType, $this->status])
            ->filter()
            ->count();
    }

    #[Computed]
    public function stats()
    {
        $erdosDepts = Departments::where('description', 'like', '%ERDOS%')->pluck('id');
        
        $baseQuery = SampleModel::query()
            ->whereHas('colourways.style_versions.styles', function ($q) use ($erdosDepts) {
                $q->whereIn('departments_id', $erdosDepts);
            });

        return [
            'total' => (clone $baseQuery)->count(),
            'pending' => (clone $baseQuery)->where('status', 'pending')->orWhereNull('status')->count(),
            'approved' => (clone $baseQuery)->where('status', 'approved')->count(),
            'sent' => (clone $baseQuery)->whereNotNull('date_sent')->count(),
        ];
    }

    protected function buildQuery()
    {
        $erdosDepts = Departments::where('description', 'like', '%ERDOS%')->pluck('id');

        $query = SampleModel::query()
            ->with([
                'colourways.style_versions.styles.designs',
                'colourways.style_versions.styles.customers',
                'colourways.style_versions.styles.seasons',
                'colourways.style_versions.factories',
                'sample_types',
                'sizes',
            ])
            ->whereHas('colourways.style_versions.styles', function ($q) use ($erdosDepts) {
                $q->whereIn('departments_id', $erdosDepts);
            });

        // Apply filters
        if ($this->search) {
            $query->search($this->search);
        }

        if ($this->season) {
            $query->whereHas('colourways.style_versions.styles', function ($q) {
                $q->where('seasons_id', $this->season);
            });
        }

        if ($this->customer) {
            $query->whereHas('colourways.style_versions.styles', function ($q) {
                $q->where('customers_id', $this->customer);
            });
        }

        if ($this->sampleType) {
            $query->where('sample_types_id', $this->sampleType);
        }

        if ($this->status) {
            if ($this->status === 'pending') {
                $query->where(function ($q) {
                    $q->where('status', 'pending')->orWhereNull('status');
                });
            } else {
                $query->where('status', $this->status);
            }
        }

        // Apply sorting
        $sortColumn = match ($this->sortBy) {
            'date_sent' => 'samples.date_sent',
            'date_expected' => 'samples.date_expected',
            'created_at' => 'samples.created_at',
            default => 'samples.date_sent',
        };

        $query->orderBy($sortColumn, $this->sortDir);

        return $query;
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedSamples)->pluck('id')->toArray();
        
        $query = $this->buildQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('samples.id', $loadedIds);
        }

        $newSamples = $query->limit($this->perPage)->get();

        if ($newSamples->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newSamples as $sample) {
            $this->loadedSamples[] = [
                'id' => $sample->id,
                'colourway_name' => $sample->colourways?->name,
                'design_id' => $sample->colourways?->style_versions?->styles?->designs?->id,
                'design_description' => $sample->colourways?->style_versions?->styles?->designs?->description,
                'customer_name' => $sample->colourways?->style_versions?->styles?->customers?->name,
                'season_description' => $sample->colourways?->style_versions?->styles?->seasons?->description,
                'sample_type_name' => $sample->sample_types?->name,
                'size_name' => $sample->sizes?->name,
                'qty' => $sample->qty,
                'date_expected' => $sample->date_expected?->format('d M'),
                'date_sent' => $sample->date_sent?->format('d M'),
                'tracking' => $sample->tracking,
                'status' => $sample->status ?? 'pending',
            ];
        }

        $this->isLoading = false;
    }

    #[On('sample-saved')]
    public function refreshData()
    {
        unset($this->stats);
        $this->selected = [];
        $this->selectAll = false;
        $this->loadInitialSamples();
    }

    public function updatedSelected()
    {
        // Ensure selected is always an array of strings for consistent comparison
        $this->selected = array_values(array_map('strval', $this->selected));
        
        // Update selectAll state based on current selection
        $loadedIds = collect($this->loadedSamples)->pluck('id')->map(fn($id) => (string) $id)->toArray();
        $this->selectAll = !empty($loadedIds) && count(array_intersect($this->selected, $loadedIds)) === count($loadedIds);
    }

    public function updatedSelectAll($value)
    {
        if ($value) {
            $this->selected = collect($this->loadedSamples)->pluck('id')->map(fn($id) => (string) $id)->toArray();
        } else {
            $this->selected = [];
        }
    }

    public function openBulkEdit()
    {
        if (empty($this->selected)) {
            return;
        }
        $this->dispatch('open-bulk-edit', sampleIds: array_map('intval', $this->selected));
    }

    public function clearSelection()
    {
        $this->selected = [];
        $this->selectAll = false;
    }

    #[Computed]
    public function selectedCount()
    {
        return count($this->selected);
    }

    public function render()
    {
        return view('livewire.commission.samples')
            ->layout('layouts.commission');
    }
}

