<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\Samples;
use App\Models\Sizes;
use App\Models\SampleTypes;
use App\Models\Colourways;
use App\Models\Styles;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\Departments;
use App\Models\User;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;

class SampleEdit extends Component
{
    public bool $isOpen = false;
    public ?int $sampleId = null;
    public bool $isDirty = false;
    public bool $isNewSample = false;
    public string $activeTab = 'details';

    // Sample fields
    public $sample_types_id = '';
    public $colourways_id = '';
    public $sizes_id = '';
    public $date_expected = '';
    public $date_sent = '';
    public $date_received = '';
    public $tracking = '';
    public $weight = '';
    public $qty = 1;
    public $status = 'pending';
    public $comments = '';
    public $po = '';
    public $do_not_charge = false;
    public $do_not_charge_dev = false;
    public $fn_notes = '';

    // For new sample - style picker
    public bool $showStylePicker = false;
    public string $styleSearch = '';
    public $selectedSeasonId = '';
    public $selectedCustomerId = '';

    protected function rules()
    {
        return [
            'sample_types_id' => 'required|exists:sample_types,id',
            'colourways_id' => 'required|exists:colourways,id',
            'sizes_id' => 'required|exists:sizes,id',
            'date_expected' => 'nullable|date',
            'date_sent' => 'nullable|date',
            'date_received' => 'nullable|date',
            'tracking' => 'nullable|string|max:255',
            'weight' => 'nullable|numeric',
            'qty' => 'required|integer|min:1',
            'status' => 'nullable|in:pending,approved,rejected',
            'comments' => 'nullable|string|max:1000',
            'po' => 'nullable|string|max:255',
            'do_not_charge' => 'boolean',
            'do_not_charge_dev' => 'boolean',
            'fn_notes' => 'nullable|string|max:1000',
        ];
    }

    #[On('open-sample-edit')]
    public function open(int $sampleId)
    {
        $this->resetState();
        $this->sampleId = $sampleId;
        $this->isNewSample = false;
        $this->loadSample();
        $this->isOpen = true;
    }

    #[On('create-new-sample')]
    public function createNew()
    {
        $this->resetState();
        $this->isNewSample = true;
        $this->isOpen = true;
        $this->qty = 1;
        $this->status = 'pending';
        
        // Default to most recent season
        $this->selectedSeasonId = Seasons::orderBy('created_at', 'desc')->first()?->id ?? '';
    }

    protected function resetState()
    {
        $this->reset([
            'sampleId', 'isDirty', 'isNewSample', 'activeTab',
            'sample_types_id', 'colourways_id', 'sizes_id',
            'date_expected', 'date_sent', 'date_received',
            'tracking', 'weight', 'qty', 'status', 'comments',
            'po', 'do_not_charge', 'do_not_charge_dev', 'fn_notes',
            'showStylePicker', 'styleSearch', 'selectedSeasonId', 'selectedCustomerId'
        ]);
        $this->activeTab = 'details';
    }

    protected function loadSample()
    {
        $sample = Samples::with([
            'colourways.style_versions.styles.designs',
            'colourways.style_versions.styles.customers',
            'colourways.style_versions.styles.seasons',
            'sample_types',
            'sizes',
        ])->find($this->sampleId);

        if (!$sample) {
            $this->close();
            return;
        }

        $this->sample_types_id = $sample->sample_types_id ?? '';
        $this->colourways_id = $sample->colourways_id ?? '';
        $this->sizes_id = $sample->sizes_id ?? '';
        $this->date_expected = $sample->date_expected?->format('Y-m-d') ?? '';
        $this->date_sent = $sample->date_sent?->format('Y-m-d') ?? '';
        $this->date_received = $sample->date_received?->format('Y-m-d') ?? '';
        $this->tracking = $sample->tracking ?? '';
        $this->weight = $sample->weight ?? '';
        $this->qty = $sample->qty ?? 1;
        $this->status = $sample->status ?? 'pending';
        $this->comments = $sample->comments ?? '';
        $this->po = $sample->po ?? '';
        $this->do_not_charge = (bool) $sample->do_not_charge;
        $this->do_not_charge_dev = (bool) $sample->do_not_charge_dev;
        $this->fn_notes = $sample->fn_notes ?? '';
    }

    public function updated($property)
    {
        $nonDataProperties = ['activeTab', 'showStylePicker', 'styleSearch', 'selectedSeasonId', 'selectedCustomerId'];
        
        if (!in_array($property, $nonDataProperties)) {
            $this->isDirty = true;
        }
    }

    public function close()
    {
        if ($this->isDirty) {
            $this->dispatch('confirm-close');
        } else {
            $this->forceClose();
        }
    }

    public function forceClose()
    {
        $this->isOpen = false;
        $this->resetState();
    }

    public function openStylePicker()
    {
        $this->styleSearch = '';
        $this->showStylePicker = true;
    }

    public function closeStylePicker()
    {
        $this->showStylePicker = false;
        $this->styleSearch = '';
    }

    public function selectColourway(int $colourwayId)
    {
        $this->colourways_id = $colourwayId;
        $this->isDirty = true;
        $this->closeStylePicker();
    }

    #[Computed]
    public function sample()
    {
        if (!$this->sampleId) return null;
        return Samples::with([
            'colourways.style_versions.styles.designs',
            'colourways.style_versions.styles.customers',
            'sample_types',
            'sizes',
        ])->find($this->sampleId);
    }

    #[Computed]
    public function selectedColourway()
    {
        if (!$this->colourways_id) return null;
        return Colourways::with([
            'style_versions.styles.designs',
            'style_versions.styles.customers',
        ])->find($this->colourways_id);
    }

    #[Computed]
    public function sampleTypes()
    {
        return SampleTypes::orderBy('name')->get();
    }

    #[Computed]
    public function sizes()
    {
        return Sizes::orderBy('order')->get();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::orderBy('name')->get();
    }

    #[Computed]
    public function availableStyles()
    {
        if (!$this->selectedCustomerId && !$this->selectedSeasonId && !$this->styleSearch) {
            return collect();
        }

        $erdosDepts = Departments::where('description', 'like', '%ERDOS%')->pluck('id');

        return Styles::with(['designs', 'style_versions.colourways'])
            ->whereIn('departments_id', $erdosDepts)
            ->when($this->selectedCustomerId, fn($q) => $q->where('customers_id', $this->selectedCustomerId))
            ->when($this->selectedSeasonId, fn($q) => $q->where('seasons_id', $this->selectedSeasonId))
            ->when($this->styleSearch, function ($q) {
                $q->where(function ($sq) {
                    $sq->where('customer_ref', 'like', "%{$this->styleSearch}%")
                       ->orWhereHas('designs', fn($dq) => $dq->where('description', 'like', "%{$this->styleSearch}%")
                           ->orWhere('id', 'like', "%{$this->styleSearch}%"));
                });
            })
            ->limit(20)
            ->get();
    }

    #[Computed(persist: false)]
    public function auditHistory()
    {
        if (!$this->sampleId) {
            return collect();
        }

        $audits = DB::table('audits')
            ->where('auditable_type', 'App\\Models\\Samples')
            ->where('auditable_id', $this->sampleId)
            ->orderBy('created_at', 'desc')
            ->orderBy('id', 'desc')
            ->limit(100)
            ->get();

        $userIds = $audits->pluck('user_id')->unique()->filter()->toArray();
        $users = User::whereIn('id', $userIds)->pluck('name', 'id');

        return $audits->map(function ($audit) use ($users) {
            $oldValues = json_decode($audit->old_values, true) ?? [];
            $newValues = json_decode($audit->new_values, true) ?? [];
            
            $changes = [];
            $skipFields = ['id', 'created_at', 'updated_at', 'deleted_at'];
            
            foreach ($newValues as $field => $newValue) {
                if (in_array($field, $skipFields)) continue;
                $oldValue = $oldValues[$field] ?? null;
                if ($this->areValuesEquivalent($oldValue, $newValue)) continue;
                
                $changes[] = [
                    'field' => $this->formatFieldName($field),
                    'old' => $this->formatAuditValue($field, $oldValue),
                    'new' => $this->formatAuditValue($field, $newValue),
                ];
            }
            
            return (object) [
                'id' => $audit->id,
                'event' => ucfirst($audit->event),
                'model_type' => 'Sample',
                'user_name' => $users[$audit->user_id] ?? 'System',
                'created_at' => \Carbon\Carbon::parse($audit->created_at),
                'changes' => $changes,
            ];
        })->filter(function ($audit) {
            if ($audit->event === 'Updated' && empty($audit->changes)) return false;
            return true;
        })->values();
    }

    protected function formatFieldName(string $field): string
    {
        $fieldNames = [
            'sample_types_id' => 'Sample Type',
            'colourways_id' => 'Colourway',
            'sizes_id' => 'Size',
            'date_expected' => 'Expected Date',
            'date_sent' => 'Sent Date',
            'date_received' => 'Received Date',
            'do_not_charge' => 'Don\'t Charge',
            'do_not_charge_dev' => 'Don\'t Charge Dev',
            'fn_notes' => 'Finance Notes',
        ];
        
        return $fieldNames[$field] ?? ucwords(str_replace('_', ' ', $field));
    }

    protected function formatAuditValue(string $field, $value): string
    {
        if ($value === null || $value === '') return '—';
        if (is_bool($value)) return $value ? 'Yes' : 'No';
        if (is_string($value) && strlen($value) > 50) return substr($value, 0, 47) . '...';
        return (string) $value;
    }

    protected function areValuesEquivalent($oldValue, $newValue): bool
    {
        if ($oldValue === $newValue) return true;
        
        $oldBool = $this->toBooleanValue($oldValue);
        $newBool = $this->toBooleanValue($newValue);
        if ($oldBool !== null && $newBool !== null && $oldBool === $newBool) return true;
        
        if (($oldValue === null || $oldValue === '') && ($newValue === null || $newValue === '')) return true;
        
        return false;
    }

    protected function toBooleanValue($value): ?bool
    {
        if ($value === true || $value === 1 || $value === '1') return true;
        if ($value === false || $value === 0 || $value === '0') return false;
        return null;
    }

    public function save()
    {
        $this->validate();

        try {
            DB::beginTransaction();

            if ($this->isNewSample) {
                $sample = Samples::create([
                    'sample_types_id' => $this->sample_types_id,
                    'colourways_id' => $this->colourways_id,
                    'sizes_id' => $this->sizes_id,
                    'date_expected' => $this->date_expected ?: null,
                    'date_sent' => $this->date_sent ?: null,
                    'date_received' => $this->date_received ?: null,
                    'tracking' => $this->tracking ?: null,
                    'weight' => $this->weight ?: null,
                    'qty' => $this->qty,
                    'status' => $this->status,
                    'comments' => $this->comments ?: null,
                    'po' => $this->po ?: null,
                    'do_not_charge' => $this->do_not_charge,
                    'do_not_charge_dev' => $this->do_not_charge_dev,
                    'fn_notes' => $this->fn_notes ?: null,
                ]);
                
                $this->sampleId = $sample->id;
                $this->isNewSample = false;
                session()->flash('message', 'Sample created successfully.');
            } else {
                // Get current sample to check if status is changing to approved
                $currentSample = Samples::find($this->sampleId);
                $approvedDate = $currentSample->approved_date;
                
                // If status is changing to 'approved' and no approved_date set, set it now
                if ($this->status === 'approved' && $currentSample->status !== 'approved') {
                    $approvedDate = now();
                }
                // If status is changing FROM 'approved' to something else, clear the date
                elseif ($this->status !== 'approved' && $currentSample->status === 'approved') {
                    $approvedDate = null;
                }
                
                Samples::where('id', $this->sampleId)->update([
                    'sample_types_id' => $this->sample_types_id,
                    'colourways_id' => $this->colourways_id,
                    'sizes_id' => $this->sizes_id,
                    'date_expected' => $this->date_expected ?: null,
                    'date_sent' => $this->date_sent ?: null,
                    'date_received' => $this->date_received ?: null,
                    'tracking' => $this->tracking ?: null,
                    'weight' => $this->weight ?: null,
                    'qty' => $this->qty,
                    'status' => $this->status,
                    'approved_date' => $approvedDate,
                    'comments' => $this->comments ?: null,
                    'po' => $this->po ?: null,
                    'do_not_charge' => $this->do_not_charge,
                    'do_not_charge_dev' => $this->do_not_charge_dev,
                    'fn_notes' => $this->fn_notes ?: null,
                ]);
                
                session()->flash('message', 'Sample saved successfully.');
            }

            DB::commit();
            
            $this->isDirty = false;
            $this->dispatch('sample-saved');

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Sample save error: ' . $e->getMessage());
            session()->flash('error', 'Error saving sample: ' . $e->getMessage());
        }
    }

    public function deleteSample()
    {
        if (!$this->sampleId) {
            session()->flash('error', 'No sample to delete.');
            return;
        }

        try {
            $sample = Samples::find($this->sampleId);
            
            if (!$sample) {
                session()->flash('error', 'Sample not found.');
                return;
            }
            
            $sample->delete();
            
            $this->dispatch('sample-saved');
            $this->isOpen = false;
            $this->resetState();
            
        } catch (\Exception $e) {
            \Log::error('Sample delete error: ' . $e->getMessage());
            session()->flash('error', 'Error deleting sample: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.commission.sample-edit');
    }
}

