<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\Samples;
use App\Models\Sizes;
use App\Models\SampleTypes;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\DB;

class SampleBulkEdit extends Component
{
    public bool $isOpen = false;
    public array $sampleIds = [];

    // Fields to update (toggle switches)
    public bool $update_sample_type = false;
    public bool $update_size = false;
    public bool $update_qty = false;
    public bool $update_status = false;
    public bool $update_date_expected = false;
    public bool $update_date_sent = false;
    public bool $update_tracking = false;
    public bool $update_weight = false;
    public bool $update_po = false;
    public bool $update_comments = false;
    public bool $update_do_not_charge = false;
    public bool $update_do_not_charge_dev = false;
    public bool $update_fn_notes = false;

    // Field values
    public $sample_types_id = '';
    public $sizes_id = '';
    public $qty = 1;
    public $status = 'pending';
    public $date_expected = '';
    public $date_sent = '';
    public $tracking = '';
    public $weight = '';
    public $po = '';
    public $comments = '';
    public $do_not_charge = false;
    public $do_not_charge_dev = false;
    public $fn_notes = '';

    #[On('open-bulk-edit')]
    public function open(array $sampleIds)
    {
        $this->resetState();
        $this->sampleIds = $sampleIds;
        $this->isOpen = true;
    }

    protected function resetState()
    {
        $this->reset([
            'sampleIds',
            'update_sample_type', 'update_size', 'update_qty', 'update_status',
            'update_date_expected', 'update_date_sent', 'update_tracking', 'update_weight',
            'update_po', 'update_comments', 'update_do_not_charge', 'update_do_not_charge_dev',
            'update_fn_notes',
            'sample_types_id', 'sizes_id', 'qty', 'status',
            'date_expected', 'date_sent', 'tracking', 'weight',
            'po', 'comments', 'do_not_charge', 'do_not_charge_dev', 'fn_notes',
        ]);
        $this->qty = 1;
        $this->status = 'pending';
    }

    public function close()
    {
        $this->isOpen = false;
        $this->resetState();
    }

    #[Computed]
    public function sampleTypes()
    {
        return SampleTypes::orderBy('name')->get();
    }

    #[Computed]
    public function sizes()
    {
        return Sizes::orderBy('order')->get();
    }

    #[Computed]
    public function selectedSamples()
    {
        return Samples::with([
            'colourways.style_versions.styles.designs',
            'sample_types',
        ])->whereIn('id', $this->sampleIds)->get();
    }

    #[Computed]
    public function fieldsToUpdate()
    {
        $count = 0;
        if ($this->update_sample_type) $count++;
        if ($this->update_size) $count++;
        if ($this->update_qty) $count++;
        if ($this->update_status) $count++;
        if ($this->update_date_expected) $count++;
        if ($this->update_date_sent) $count++;
        if ($this->update_tracking) $count++;
        if ($this->update_weight) $count++;
        if ($this->update_po) $count++;
        if ($this->update_comments) $count++;
        if ($this->update_do_not_charge) $count++;
        if ($this->update_do_not_charge_dev) $count++;
        if ($this->update_fn_notes) $count++;
        return $count;
    }

    public function save()
    {
        if ($this->fieldsToUpdate === 0) {
            session()->flash('error', 'Please select at least one field to update.');
            return;
        }

        try {
            DB::beginTransaction();

            $updateData = [];

            if ($this->update_sample_type && $this->sample_types_id) {
                $updateData['sample_types_id'] = $this->sample_types_id;
            }
            if ($this->update_size && $this->sizes_id) {
                $updateData['sizes_id'] = $this->sizes_id;
            }
            if ($this->update_qty) {
                $updateData['qty'] = $this->qty;
            }
            if ($this->update_status) {
                $updateData['status'] = $this->status;
            }
            if ($this->update_date_expected) {
                $updateData['date_expected'] = $this->date_expected ?: null;
            }
            if ($this->update_date_sent) {
                $updateData['date_sent'] = $this->date_sent ?: null;
            }
            if ($this->update_tracking) {
                $updateData['tracking'] = $this->tracking ?: null;
            }
            if ($this->update_weight) {
                $updateData['weight'] = $this->weight ?: null;
            }
            if ($this->update_po) {
                $updateData['po'] = $this->po ?: null;
            }
            if ($this->update_comments) {
                $updateData['comments'] = $this->comments ?: null;
            }
            if ($this->update_do_not_charge) {
                $updateData['do_not_charge'] = $this->do_not_charge;
            }
            if ($this->update_do_not_charge_dev) {
                $updateData['do_not_charge_dev'] = $this->do_not_charge_dev;
            }
            if ($this->update_fn_notes) {
                $updateData['fn_notes'] = $this->fn_notes ?: null;
            }

            if (!empty($updateData)) {
                // Check for locked seasons
                $samples = Samples::with('colourways.style_versions.styles.seasons')
                    ->whereIn('id', $this->sampleIds)
                    ->get();

                $lockedCount = 0;
                $updatedCount = 0;

                foreach ($samples as $sample) {
                    $isLocked = $sample->colourways?->style_versions?->styles?->seasons?->locked ?? false;
                    
                    if (!$isLocked) {
                        $sampleUpdateData = $updateData;
                        
                        // Handle approved_date when status is being updated
                        if ($this->update_status) {
                            if ($this->status === 'approved' && $sample->status !== 'approved') {
                                $sampleUpdateData['approved_date'] = now();
                            } elseif ($this->status !== 'approved' && $sample->status === 'approved') {
                                $sampleUpdateData['approved_date'] = null;
                            }
                        }
                        
                        $sample->update($sampleUpdateData);
                        $updatedCount++;
                    } else {
                        $lockedCount++;
                    }
                }

                if ($lockedCount > 0 && $updatedCount > 0) {
                    session()->flash('message', "Updated {$updatedCount} samples. {$lockedCount} samples were skipped (locked season).");
                } elseif ($lockedCount > 0) {
                    session()->flash('error', 'All selected samples are in locked seasons and cannot be updated.');
                    DB::rollBack();
                    return;
                } else {
                    session()->flash('message', "Successfully updated {$updatedCount} samples.");
                }
            }

            DB::commit();
            $this->dispatch('sample-saved');
            $this->close();

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Bulk sample update error: ' . $e->getMessage());
            session()->flash('error', 'Error updating samples: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.commission.sample-bulk-edit');
    }
}


