<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\CommissionOrder;
use App\Models\CommissionImportBatch;
use App\Models\CommissionDrop;
use App\Models\CommissionDropSize;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\User;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use App\Notifications\SkeletonOrderForwardedToPd;
use App\Notifications\SkeletonOrderApprovedByPd;
use App\Models\CommissionOrderWorkflowHistory;

class Orders extends Component
{
    #[Url]
    public string $search = '';
    
    #[Url]
    public string $season = '';
    
    #[Url]
    public string $customer = '';
    
    #[Url]
    public array $status = [];
    
    #[Url]
    public string $matchStatus = '';
    
    #[Url]
    public string $dateFrom = '';
    
    #[Url]
    public string $dateTo = '';
    
    #[Url]
    public string $sortBy = 'imported_at';
    
    #[Url]
    public string $sortDir = 'desc';

    #[Url]
    public ?int $batch = null;

    #[Url]
    public ?int $order = null;
    
    // Infinite scroll
    public int $perPage = 25;
    public array $loadedOrders = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    #[On('order-saved')]
    #[On('order-imported')]
    #[On('order-updated')]
    public function refreshOrders()
    {
        unset($this->stats);
        $this->loadInitialOrders();
    }

    public function mount()
    {
        $this->loadInitialOrders();
    }

    public function loadInitialOrders()
    {
        $this->loadedOrders = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    // For forward to PD modal
    public bool $showForwardModal = false;
    public ?int $forwardOrderId = null;
    public ?int $forwardToUserId = null;
    public string $forwardNotes = '';
    public bool $forwardSelfCertified = false;

    // For approval modal
    public bool $showApprovalModal = false;
    public ?int $approvalOrderId = null;
    public string $approvalType = ''; // 'pd' or 'merch'
    public string $approvalNotes = '';
    public string $approvalError = '';

    // For send back to PD modal
    public bool $showSendBackModal = false;
    public ?int $sendBackOrderId = null;
    public ?int $sendBackToUserId = null;
    public string $sendBackNotes = '';

    public function updatedSearch()
    {
        $this->loadInitialOrders();
    }

    public function updatedSeason()
    {
        $this->loadInitialOrders();
    }

    public function updatedCustomer()
    {
        $this->loadInitialOrders();
    }

    public function updatedStatus()
    {
        $this->loadInitialOrders();
    }

    public function updatedMatchStatus()
    {
        $this->loadInitialOrders();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'season', 'customer', 'status', 'matchStatus', 'dateFrom', 'dateTo', 'batch', 'order']);
        $this->loadInitialOrders();
    }

    public function toggleStatus(string $statusKey)
    {
        if (in_array($statusKey, $this->status)) {
            $this->status = array_values(array_diff($this->status, [$statusKey]));
        } else {
            $this->status[] = $statusKey;
        }
        $this->loadInitialOrders();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'desc';
        }
        $this->loadInitialOrders();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function pdUsers()
    {
        return User::whereHas('functionalRoles', fn($q) => $q->whereIn('name', ['pd', 'admin']))
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function statusOptions()
    {
        return [
            'skeleton' => ['label' => 'Imported', 'icon' => 'fa-file-import', 'color' => 'danger'],
            'pending_pd' => ['label' => 'Pending PD', 'icon' => 'fa-user-clock', 'color' => 'warning'],
            'pending_merch' => ['label' => 'Pending Merch', 'icon' => 'fa-clipboard-check', 'color' => 'info'],
            'confirmed' => ['label' => 'Approved', 'icon' => 'fa-check-circle', 'color' => 'success'],
        ];
    }

    protected function buildQuery()
    {
        return CommissionOrder::query()
            ->with([
                'customer:id,name',
                'season:id,description',
                'department:id,description',
                'importedBy:id,name',
                'pdReviewedBy:id,name',
                'confirmedBy:id,name',
                'lines' => function ($q) {
                    $q->select('id', 'commission_orders_id', 'colourways_id', 'match_status', 'cancelled')
                      ->where('cancelled', false);
                },
            ])
            ->withCount([
                'lines as total_lines' => fn($q) => $q->where('cancelled', false),
                'lines as matched_lines' => fn($q) => $q->where('cancelled', false)->whereNotNull('colourways_id'),
                'lines as unmatched_lines' => fn($q) => $q->where('cancelled', false)->whereNull('colourways_id'),
            ])
            ->where('cancelled', false)
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('customer_po', 'like', "%{$this->search}%")
                      ->orWhere('id', 'like', "%{$this->search}%")
                      ->orWhere('source_file', 'like', "%{$this->search}%")
                      ->orWhereHas('customer', fn($cq) => $cq->where('name', 'like', "%{$this->search}%"));
                });
            })
            ->when($this->season, fn($q) => $q->where('seasons_id', $this->season))
            ->when($this->customer, fn($q) => $q->where('customers_id', $this->customer))
            ->when(!empty($this->status), fn($q) => $q->whereIn('status', $this->status))
            ->when($this->matchStatus === 'all_matched', function ($q) {
                $q->whereDoesntHave('lines', fn($lq) => $lq->where('cancelled', false)->whereNull('colourways_id'));
            })
            ->when($this->matchStatus === 'has_unmatched', function ($q) {
                $q->whereHas('lines', fn($lq) => $lq->where('cancelled', false)->whereNull('colourways_id'));
            })
            ->when($this->dateFrom, fn($q) => $q->whereDate('imported_at', '>=', $this->dateFrom))
            ->when($this->dateTo, fn($q) => $q->whereDate('imported_at', '<=', $this->dateTo))
            ->when($this->batch, function ($q) {
                $q->whereHas('importBatch', fn($bq) => $bq->where('id', $this->batch));
            })
            ->when($this->order, fn($q) => $q->where('id', $this->order))
            ->orderBy($this->sortBy, $this->sortDir);
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedOrders)->pluck('id')->toArray();
        
        $query = $this->buildQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('commission_orders.id', $loadedIds);
        }

        $newOrders = $query->limit($this->perPage)->get();

        if ($newOrders->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newOrders as $order) {
            $matchPercent = $order->total_lines > 0 
                ? (int) round(($order->matched_lines / $order->total_lines) * 100) 
                : 0;
            
            $this->loadedOrders[] = [
                'id' => $order->id,
                'customer_po' => $order->customer_po,
                'customer_name' => $order->customer?->name,
                'season_description' => $order->season?->description,
                'status' => $order->status,
                'total_lines' => $order->total_lines,
                'matched_lines' => $order->matched_lines,
                'unmatched_lines' => $order->unmatched_lines,
                'match_percent' => $matchPercent,
                'imported_at' => $order->imported_at?->format('d M'),
                'imported_at_diff' => $order->imported_at?->diffForHumans(),
            ];
        }

        $this->isLoading = false;
    }
    
    #[Computed]
    public function stats()
    {
        $baseQuery = CommissionOrder::where('cancelled', false);
        
        return [
            'total' => (clone $baseQuery)->count(),
            'skeleton' => (clone $baseQuery)->where('status', 'skeleton')->count(),
            'pending_pd' => (clone $baseQuery)->where('status', 'pending_pd')->count(),
            'pending_merch' => (clone $baseQuery)->where('status', 'pending_merch')->count(),
            'confirmed' => (clone $baseQuery)->where('status', 'confirmed')->count(),
        ];
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if ($this->season) $count++;
        if ($this->customer) $count++;
        if (!empty($this->status)) $count++;
        if ($this->matchStatus) $count++;
        if ($this->dateFrom || $this->dateTo) $count++;
        if ($this->batch) $count++;
        if ($this->order) $count++;
        return $count;
    }

    // ==================
    // Forward to PD Modal
    // ==================

    public function openForwardModal(int $orderId)
    {
        $this->forwardOrderId = $orderId;
        $this->forwardToUserId = null;
        $this->forwardNotes = '';
        $this->forwardSelfCertified = false;
        $this->showForwardModal = true;
    }

    public function closeForwardModal()
    {
        $this->showForwardModal = false;
        $this->forwardOrderId = null;
        $this->forwardSelfCertified = false;
    }

    public function forwardToPd()
    {
        $this->validate([
            'forwardSelfCertified' => 'nullable|boolean',
        ]);

        if (!$this->forwardSelfCertified) {
            $this->validate([
                'forwardToUserId' => 'required|exists:users,id',
            ]);
        }

        $order = CommissionOrder::findOrFail($this->forwardOrderId);
        
        if ($order->status !== 'skeleton') {
            $this->dispatch('notify', type: 'error', message: 'Order must be in skeleton status.');
            return;
        }

        $forwardToUserId = $this->forwardSelfCertified ? auth()->id() : $this->forwardToUserId;

        $order->update([
            'status' => 'pending_pd',
            'pd_email_sent_at' => now(),
            'forwarded_to_user_id' => $forwardToUserId,
            'forward_notes' => $this->forwardNotes ?: null,
        ]);

        // Update batch if exists
        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();
        if ($batch) {
            $batch->forwardToPd($forwardToUserId, $this->forwardNotes);
        }

        // Send email notification to PD user
        $pdUser = User::find($forwardToUserId);
        if ($pdUser) {
            $pdUser->notify(new SkeletonOrderForwardedToPd($order, auth()->user(), $this->forwardNotes));
        }

        $this->closeForwardModal();
        $this->dispatch('notify', type: 'success', message: 'Order forwarded to ' . ($pdUser?->name ?? 'PD') . '.');
        $this->refreshOrders();
    }

    // ==================
    // Approval Modal
    // ==================

    public function openApprovalModal(int $orderId, string $type)
    {
        $this->approvalOrderId = $orderId;
        $this->approvalType = $type;
        $this->approvalNotes = '';
        $this->approvalError = '';
        $this->showApprovalModal = true;
    }

    public function closeApprovalModal()
    {
        $this->showApprovalModal = false;
        $this->approvalOrderId = null;
        $this->approvalError = '';
    }

    public function submitApproval()
    {
        if (!$this->approvalOrderId) {
            $this->dispatch('notify', type: 'error', message: 'No order selected.');
            $this->closeApprovalModal();
            return;
        }

        try {
            $order = CommissionOrder::findOrFail($this->approvalOrderId);
        } catch (\Exception $e) {
            $this->dispatch('notify', type: 'error', message: 'Order not found.');
            $this->closeApprovalModal();
            return;
        }

        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();

        if ($this->approvalType === 'pd') {
            if ($order->status !== 'pending_pd') {
                $this->dispatch('notify', type: 'error', message: 'Order must be pending PD review.');
                return;
            }

            // Check if all lines are matched
            $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
            if ($unmatchedCount > 0) {
                $this->dispatch('notify', type: 'error', message: "Cannot approve: {$unmatchedCount} line(s) not matched to a colourway.");
                return;
            }

            $order->update([
                'status' => 'pending_merch',
                'pd_reviewed_by' => auth()->id(),
                'pd_reviewed_at' => now(),
                'pd_notes' => $this->approvalNotes,
            ]);

            if ($batch) {
                $batch->approvePd($this->approvalNotes);
            }

            // Notify the person who imported/forwarded the order
            $originalUser = $order->importedBy;
            if ($originalUser) {
                $originalUser->notify(new SkeletonOrderApprovedByPd($order, auth()->user(), $this->approvalNotes));
            }

            $this->dispatch('notify', type: 'success', message: 'PD approval submitted. Order sent to Merch.');

        } elseif ($this->approvalType === 'merch') {
            if ($order->status !== 'pending_merch') {
                $this->approvalError = 'Order must be pending Merch approval.';
                return;
            }

            // Check if all active lines have commission percentage set
            $linesWithoutCommission = $order->lines()
                ->where('cancelled', false)
                ->where(function ($q) {
                    $q->whereNull('commission_percent')
                      ->orWhere('commission_percent', '<=', 0);
                })
                ->count();
            if ($linesWithoutCommission > 0) {
                $this->approvalError = "Cannot confirm: {$linesWithoutCommission} line(s) missing Commission %. Edit the order to set it on each line.";
                return;
            }

            // Check if all lines are matched
            $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
            if ($unmatchedCount > 0) {
                $this->approvalError = "Cannot confirm: {$unmatchedCount} lines still unmatched.";
                return;
            }

            // Check if all lines have drops with ex-factory dates
            $order->load('lines.drops');
            $linesWithoutExfty = 0;
            foreach ($order->lines as $line) {
                if ($line->cancelled) {
                    continue;
                }
                $hasExfty = $line->drops->contains(fn($drop) => !empty($drop->exfty));
                if (!$hasExfty) {
                    $linesWithoutExfty++;
                }
            }
            if ($linesWithoutExfty > 0) {
                $this->approvalError = "Cannot confirm: {$linesWithoutExfty} line(s) missing ex-factory date. Edit the order to add drops with dates.";
                return;
            }

            $order->update([
                'status' => 'confirmed',
                'confirmed_by' => auth()->id(),
                'confirmed_at' => now(),
                'merch_notes' => $this->approvalNotes,
            ]);

            if ($batch) {
                $batch->approveMerch($this->approvalNotes);
            }

            $this->dispatch('notify', type: 'success', message: 'Order confirmed!');
        }

        $this->closeApprovalModal();
        $this->refreshOrders();
    }

    // ==================
    // Quick Actions
    // ==================

    /**
     * Open send back to PD modal
     */
    public function openSendBackModal(int $orderId)
    {
        $this->sendBackOrderId = $orderId;
        $this->sendBackToUserId = null;
        $this->sendBackNotes = '';
        $this->showSendBackModal = true;
    }

    public function closeSendBackModal()
    {
        $this->showSendBackModal = false;
        $this->sendBackOrderId = null;
    }

    /**
     * Send order back to PD (from Merch)
     */
    public function sendBackToPd()
    {
        $this->validate([
            'sendBackToUserId' => 'required|exists:users,id',
        ], [
            'sendBackToUserId.required' => 'Please select a PD user.',
        ]);

        $order = CommissionOrder::findOrFail($this->sendBackOrderId);
        
        if ($order->status !== 'pending_merch') {
            $this->dispatch('notify', type: 'error', message: 'Order must be pending Merch approval.');
            return;
        }

        $previousStatus = $order->status;

        // Don't clear pd_reviewed_by/at - keep the last approval info for history
        $order->update([
            'status' => 'pending_pd',
            'forwarded_to_user_id' => $this->sendBackToUserId,
        ]);

        // Record workflow history
        CommissionOrderWorkflowHistory::create([
            'commission_orders_id' => $order->id,
            'action' => 'sent_back_to_pd',
            'from_status' => $previousStatus,
            'to_status' => 'pending_pd',
            'from_user_id' => auth()->id(),
            'to_user_id' => $this->sendBackToUserId,
            'notes' => $this->sendBackNotes ?: null,
        ]);

        $this->closeSendBackModal();

        $pdUser = User::find($this->sendBackToUserId);
        $this->dispatch('notify', type: 'success', message: 'Order sent back to ' . ($pdUser?->name ?? 'PD') . ' for review.');
        $this->refreshOrders();
    }

    /**
     * Quick action: Send to PD (generates email)
     */
    public function sendToPd(int $orderId)
    {
        $order = CommissionOrder::findOrFail($orderId);
        
        if ($order->status !== 'skeleton') {
            $this->dispatch('notify', type: 'error', message: 'Order must be in skeleton status to send to PD.');
            return;
        }
        
        // Update status
        $order->update([
            'status' => 'pending_pd',
            'pd_email_sent_at' => now(),
        ]);
        
        // Generate mailto link with attachment info
        $subject = urlencode("Order Review Required: {$order->customer_po} - {$order->customer?->name}");
        $body = urlencode("Please review the attached order.\n\nOrder: {$order->customer_po}\nCustomer: {$order->customer?->name}\nSeason: {$order->season?->description}\n\nSource file: {$order->source_file}");
        
        $this->dispatch('open-mailto', mailto: "mailto:?subject={$subject}&body={$body}");
        $this->dispatch('notify', type: 'success', message: 'Order sent to PD for review.');
        
        $this->refreshOrders();
    }

    /**
     * Quick action: Mark as reviewed by PD
     */
    public function markPdReviewed(int $orderId)
    {
        $order = CommissionOrder::findOrFail($orderId);
        
        if ($order->status !== 'pending_pd') {
            $this->dispatch('notify', type: 'error', message: 'Order must be pending PD review.');
            return;
        }
        
        // Check if all lines are matched to a colourway
        $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
        if ($unmatchedCount > 0) {
            $this->dispatch('notify', type: 'error', message: "Cannot approve: {$unmatchedCount} line(s) not matched to a colourway.");
            return;
        }
        
        $order->update([
            'status' => 'pending_merch',
            'pd_reviewed_by' => auth()->id(),
            'pd_reviewed_at' => now(),
        ]);
        
        $this->dispatch('notify', type: 'success', message: 'Order marked as PD reviewed.');
        $this->refreshOrders();
    }

    /**
     * Quick action: Confirm order
     */
    public function confirmOrder(int $orderId)
    {
        $order = CommissionOrder::findOrFail($orderId);
        
        if ($order->status !== 'pending_merch') {
            $this->dispatch('notify', type: 'error', message: 'Order must be pending merch confirmation.');
            return;
        }
        
        // Check if all lines are matched
        $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
        if ($unmatchedCount > 0) {
            $this->dispatch('notify', type: 'error', message: "Cannot confirm: {$unmatchedCount} lines still unmatched.");
            return;
        }
        
        $order->update([
            'status' => 'confirmed',
            'confirmed_by' => auth()->id(),
            'confirmed_at' => now(),
        ]);
        
        $this->dispatch('notify', type: 'success', message: 'Order confirmed successfully.');
        $this->refreshOrders();
    }

    /**
     * Create drops for order lines that don't have any drops yet
     * This is called when an order is confirmed to ensure it appears on shipment schedule
     */
    protected function createDropsForOrder(CommissionOrder $order)
    {
        $order->load('lines.quantities.size', 'lines.drops');
        
        foreach ($order->lines as $line) {
            // Skip cancelled lines
            if ($line->cancelled) {
                continue;
            }
            
            // Skip lines that already have drops
            if ($line->drops->count() > 0) {
                continue;
            }
            
            // Create a drop for this line
            $drop = CommissionDrop::create([
                'commission_order_lines_id' => $line->id,
                'exfty' => null, // Ex-factory date to be set later on shipment schedule
            ]);
            
            // Create drop sizes based on line quantities
            foreach ($line->quantities as $qty) {
                if ($qty->size_id && $qty->qty > 0) {
                    CommissionDropSize::create([
                        'commission_drops_id' => $drop->id,
                        'sizes_id' => $qty->size_id,
                        'qty' => $qty->qty,
                    ]);
                }
            }
        }
    }

    /**
     * Delete order
     */
    public function deleteOrder(int $orderId)
    {
        $order = CommissionOrder::findOrFail($orderId);
        
        if ($order->status === 'confirmed') {
            $this->dispatch('notify', type: 'error', message: 'Cannot delete confirmed orders.');
            return;
        }
        
        $order->update(['cancelled' => true]);
        
        $this->dispatch('notify', type: 'success', message: 'Order deleted.');
        $this->refreshOrders();
    }

    public function render()
    {
        return view('livewire.commission.orders')->layout('layouts.commission');
    }
}
