<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\CommissionOrder;
use App\Models\CommissionOrderLine;
use App\Models\CommissionOrderLineQuantity;
use App\Models\CommissionDrop;
use App\Models\CommissionDropSize;
use App\Models\Seasons;
use App\Models\Customer;
use App\Models\CustomerAddress;
use App\Models\Colourways;
use App\Models\Styles;
use App\Models\Sizes;
use App\Models\User;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Renderless;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use App\Mail\OrderCreated;
use App\Models\CommissionImportBatch;
use App\Models\CommissionOrderWorkflowHistory;
use App\Notifications\SkeletonOrderForwardedToPd;
use App\Notifications\SkeletonOrderApprovedByPd;

class OrderEdit extends Component
{
    use WithFileUploads;

    public bool $isOpen = false;
    public ?int $orderId = null;
    public bool $isDirty = false;
    public bool $showConfirmClose = false;
    public bool $isNewOrder = false;
    public string $orderStatus = 'confirmed';
    public string $activeTab = 'details';
    
    // Order header fields
    public $season_id = null;
    public $customer_id = null;
    public $customer_po = '';
    public $order_date = '';
    public $status = 'confirmed';
    
    // Default commission/discount for new lines (can be set at order level as default)
    public $default_commission_percent = null;
    public $default_discount_percent = null;
    
    // Import & Approval tracking (read-only display)
    public $imported_by_name = null;
    public $imported_at = null;
    public $forwarded_to_name = null;
    public $forwarded_at = null;
    public $forward_notes = null;
    public $pd_reviewed_by_name = null;
    public $pd_reviewed_at = null;
    public $pd_notes = null;
    public $confirmed_by_name = null;
    public $confirmed_at = null;
    public $merch_notes = null;
    
    // Source files
    public $source_files = [];
    public bool $showPdfViewer = false;
    
    // Forward/Approval modals
    public bool $showForwardModal = false;
    public $forwardToUserId = null;
    public $forwardNotes = '';
    public bool $forwardSelfCertified = false;
    public bool $showApprovalModal = false;
    public string $approvalType = '';
    public string $approvalNotes = '';
    public string $approvalError = '';
    
    // Send back to PD modal
    public bool $showSendBackModal = false;
    public ?int $sendBackToUserId = null;
    public string $sendBackNotes = '';
    
    // Line items - comprehensive structure
    public array $lines = [];
    
    // For style/colourway picker
    public bool $showStylePicker = false;
    public string $styleSearch = '';
    
    // For changing colourway on a line
    public bool $showColourwayPicker = false;
    public ?int $editingLineIndex = null;
    public string $colourwaySearch = '';
    
    // For creating new colourway
    public bool $showNewColourwayModal = false;
    public ?int $newColourwayStyleId = null;
    public string $newColourwayName = '';
    public string $newColourwayCode = '';
    
    // For creating new style
    public bool $showNewStyleModal = false;
    public string $newStyleCustomerRef = '';
    public string $newStyleDescription = '';
    public string $newStyleYarn = '';
    public $newStyleFactoryId = 5; // Default to Erdos
    public string $newStyleColourName = '';
    
    // For quick adding sizes inline - each entry has 'name' and 'values' (per line: qty, price)
    public array $quickAddSizeInputs = [];

    protected function rules()
    {
        return [
            'season_id' => 'required',
            'customer_id' => 'required',
            'customer_po' => 'required|string|max:255',
            'order_date' => 'required|date',
        ];
    }

    #[On('open-order-edit')]
    public function openOrder(int $orderId)
    {
        $this->resetState();
        $this->orderId = $orderId;
        $this->isNewOrder = false;
        $this->loadOrder();
        $this->isOpen = true;
    }

    #[On('create-new-order')]
    public function createNew()
    {
        $this->resetState();
        $this->orderId = null;
        $this->isNewOrder = true;
        
        $this->season_id = Seasons::orderBy('created_at', 'desc')->first()?->id;
        $this->order_date = now()->format('Y-m-d');
        $this->status = 'confirmed';
        $this->lines = [];
        
        $this->isOpen = true;
    }

    protected function resetState()
    {
        $this->reset([
            'orderId', 'isDirty', 'showConfirmClose', 'isNewOrder', 'orderStatus',
            'season_id', 'customer_id',
            'customer_po', 'order_date', 'status', 'lines',
            'default_discount_percent', 'default_commission_percent',
            'imported_by_name', 'imported_at',
            'forwarded_to_name', 'forwarded_at', 'forward_notes',
            'pd_reviewed_by_name', 'pd_reviewed_at', 'pd_notes',
            'confirmed_by_name', 'confirmed_at', 'merch_notes',
            'source_files',
            'showForwardModal', 'forwardToUserId', 'forwardNotes',
            'forwardSelfCertified',
            'showApprovalModal', 'approvalType', 'approvalNotes',
            'showSendBackModal', 'sendBackToUserId', 'sendBackNotes',
            'showStylePicker', 'styleSearch', 'activeTab',
            'quickAddSizeInputs'
        ]);
        $this->activeTab = 'details';
    }

    protected function loadOrder()
    {
        $order = CommissionOrder::with([
            'lines.colourway.style_versions.styles.designs',
            'lines.quantities.size',
            'lines.drops.sizes.size',
            'importedBy',
            'forwardedTo',
            'pdReviewedBy',
            'confirmedBy',
        ])->find($this->orderId);

        if (!$order) return;

        $this->season_id = $order->seasons_id;
        $this->customer_id = $order->customers_id;
        $this->customer_po = $order->customer_po ?? '';
        $this->order_date = $order->order_date?->format('Y-m-d') ?? '';
        $this->status = $order->status;
        $this->orderStatus = $order->status;
        
        // Set default commission/discount from first line (for display/new lines)
        $firstLine = $order->lines->first();
        $this->default_commission_percent = $firstLine?->commission_percent;
        $this->default_discount_percent = $firstLine?->discount_percent;
        
        // Load import & approval tracking
        $this->imported_by_name = $order->importedBy?->name;
        $this->imported_at = $order->imported_at?->format('d M Y H:i');
        $this->forwarded_to_name = $order->forwardedTo?->name;
        $this->forwarded_at = $order->pd_email_sent_at?->format('d M Y H:i');
        $this->forward_notes = $order->forward_notes;
        $this->pd_reviewed_by_name = $order->pdReviewedBy?->name;
        $this->pd_reviewed_at = $order->pd_reviewed_at?->format('d M Y H:i');
        $this->pd_notes = $order->pd_notes;
        $this->confirmed_by_name = $order->confirmedBy?->name;
        $this->confirmed_at = $order->confirmed_at?->format('d M Y H:i');
        $this->merch_notes = $order->merch_notes;
        
        // Load source files
        $this->source_files = $order->source_files;

        // Load lines with full structure
        $this->lines = [];
        foreach ($order->lines as $line) {
            $colourway = $line->colourway;
            $style = $colourway?->style_versions?->styles;
            $design = $style?->designs;
            
            // Build sizes array with ordered quantities
            $sizes = [];
            foreach ($line->quantities as $qtyRecord) {
                $sizes[] = [
                    'id' => $qtyRecord->id,
                    'size_id' => $qtyRecord->sizes_id,
                    'size_name' => $qtyRecord->size?->name ?? '',
                    'qty' => $qtyRecord->qty ?? 0,
                    'price' => $qtyRecord->price ?? 0,
                ];
            }
            
            // Build drops array with size breakdowns
            $drops = [];
            foreach ($line->drops as $drop) {
                $dropSizes = [];
                foreach ($drop->sizes as $dropSize) {
                    $dropSizes[] = [
                        'id' => $dropSize->id,
                        'size_id' => $dropSize->sizes_id,
                        'size_name' => $dropSize->size?->name ?? '',
                        'qty' => $dropSize->qty ?? 0,
                    ];
                }
                
                $drops[] = [
                    'id' => $drop->id,
                    'due_date' => $drop->exfty?->format('Y-m-d') ?? '',
                    'sizes' => $dropSizes,
                ];
            }
            
            $this->lines[] = [
                'id' => $line->id,
                'colourway_id' => $colourway?->id,
                'rt_number' => $design?->id,
                'description' => $design?->description ?? '',
                'colourway_name' => $colourway?->name ?? '',
                'customer_ref' => $style?->customer_ref ?? '',
                'imported_style_ref' => $line->imported_style_ref,
                'imported_description' => $line->imported_style_description,
                'imported_colour' => $line->imported_colour,
                'match_status' => $line->match_status,
                'ai_confidence' => $line->ai_match_confidence,
                'imported_raw_data' => $line->imported_raw_data,
                'commission_percent' => $line->commission_percent,
                'discount_percent' => $line->discount_percent,
                'cancelled' => $line->cancelled ?? false,
                'sizes' => $sizes,
                'drops' => $drops,
            ];
        }
    }

    #[Computed]
    public function order()
    {
        if (!$this->orderId) return null;
        return CommissionOrder::with(['lines'])->find($this->orderId);
    }

    #[Computed]
    public function isSkeletonOrder(): bool
    {
        return in_array($this->orderStatus, ['skeleton', 'pending_pd', 'pending_merch']);
    }

    #[Computed]
    public function isPdReviewer(): bool
    {
        // User is PD if order is pending_pd and they have PD role
        if ($this->orderStatus !== 'pending_pd') {
            return false;
        }
        return auth()->user()?->hasRole('pd') ?? false;
    }

    #[Computed]
    public function canEditQuantities(): bool
    {
        // PD can't edit quantities, only Merch/Admin can
        if ($this->isPdReviewer) {
            return false;
        }
        return true;
    }

    #[Computed]
    public function canEditStyleColourway(): bool
    {
        // Both PD and Merch can edit style/colourway on skeleton orders
        return $this->isSkeletonOrder;
    }

    #[Computed]
    public function pdUsers()
    {
        return User::whereNull('deleted_at')
            ->whereHas('functionalRoles', function ($q) {
                $q->whereIn('name', ['pd', 'admin']);
            })
            ->orderBy('name')
            ->get();
    }

    // ==================
    // Forward to PD
    // ==================

    public function openForwardModal()
    {
        $this->forwardToUserId = null;
        $this->forwardNotes = '';
        $this->forwardSelfCertified = false;
        $this->showForwardModal = true;
    }

    public function closeForwardModal()
    {
        $this->showForwardModal = false;
        $this->forwardSelfCertified = false;
    }

    public function saveAndForwardToPd()
    {
        // First save
        $this->save();
        
        if ($this->orderStatus !== 'skeleton') {
            $this->dispatch('notify', type: 'error', message: 'Order must be in skeleton status to forward.');
            return;
        }

        $this->openForwardModal();
    }

    public function forwardToPd()
    {
        $this->validate([
            'forwardSelfCertified' => 'nullable|boolean',
        ]);

        if (!$this->forwardSelfCertified) {
            $this->validate([
                'forwardToUserId' => 'required|exists:users,id',
            ]);
        }

        $order = CommissionOrder::findOrFail($this->orderId);
        
        if ($order->status !== 'skeleton') {
            $this->dispatch('notify', type: 'error', message: 'Order must be in skeleton status.');
            return;
        }

        $previousStatus = $order->status;

        $forwardToUserId = $this->forwardSelfCertified ? auth()->id() : $this->forwardToUserId;

        $order->update([
            'status' => 'pending_pd',
            'pd_email_sent_at' => now(),
            'forwarded_to_user_id' => $forwardToUserId,
            'forward_notes' => $this->forwardNotes ?: null,
        ]);

        // Record workflow history
        CommissionOrderWorkflowHistory::create([
            'commission_orders_id' => $order->id,
            'action' => 'forward_to_pd',
            'from_status' => $previousStatus,
            'to_status' => 'pending_pd',
            'from_user_id' => auth()->id(),
            'to_user_id' => $forwardToUserId,
            'notes' => $this->forwardNotes ?: null,
        ]);

        // Update batch if exists
        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();
        if ($batch) {
            $batch->forwardToPd($forwardToUserId, $this->forwardNotes);
        }

        // Send email notification to PD user
        $pdUser = User::find($forwardToUserId);
        if ($pdUser) {
            $pdUser->notify(new SkeletonOrderForwardedToPd($order, auth()->user(), $this->forwardNotes));
        }

        $this->closeForwardModal();
        $this->orderStatus = 'pending_pd';
        $this->forwarded_to_name = $pdUser?->name;
        $this->forwarded_at = now()->format('d M Y H:i');
        $this->forward_notes = $this->forwardNotes ?: null;
        
        $this->dispatch('notify', type: 'success', message: 'Order forwarded to ' . ($pdUser?->name ?? 'PD') . '.');
        $this->dispatch('order-updated');
    }

    // ==================
    // Approvals
    // ==================

    public function openApprovalModal(string $type)
    {
        $this->approvalType = $type;
        $this->approvalNotes = '';
        $this->approvalError = '';
        $this->showApprovalModal = true;
    }

    public function closeApprovalModal()
    {
        $this->showApprovalModal = false;
        $this->approvalError = '';
    }

    public function openSendBackModal()
    {
        $this->sendBackToUserId = null;
        $this->sendBackNotes = '';
        $this->showSendBackModal = true;
    }

    public function closeSendBackModal()
    {
        $this->showSendBackModal = false;
    }

    public function sendBackToPd()
    {
        $this->validate([
            'sendBackToUserId' => 'required|exists:users,id',
        ], [
            'sendBackToUserId.required' => 'Please select a PD user to send this order to.',
        ]);

        $order = CommissionOrder::findOrFail($this->orderId);
        
        if ($order->status !== 'pending_merch') {
            $this->dispatch('notify', type: 'error', message: 'Order must be pending Merch approval.');
            return;
        }

        $previousStatus = $order->status;

        // Don't clear pd_reviewed_by/at - keep the last approval info for history
        $order->update([
            'status' => 'pending_pd',
            'forwarded_to_user_id' => $this->sendBackToUserId,
        ]);

        // Record workflow history
        CommissionOrderWorkflowHistory::create([
            'commission_orders_id' => $order->id,
            'action' => 'sent_back_to_pd',
            'from_status' => $previousStatus,
            'to_status' => 'pending_pd',
            'from_user_id' => auth()->id(),
            'to_user_id' => $this->sendBackToUserId,
            'notes' => $this->sendBackNotes ?: null,
        ]);

        $this->orderStatus = 'pending_pd';

        $this->closeSendBackModal();

        $pdUser = User::find($this->sendBackToUserId);
        $this->dispatch('notify', type: 'success', message: 'Order sent back to ' . ($pdUser?->name ?? 'PD') . ' for review.');
        $this->dispatch('order-updated');
    }

    public function submitApproval()
    {
        $order = CommissionOrder::findOrFail($this->orderId);
        $batch = CommissionImportBatch::where('commission_order_id', $order->id)->first();

        if ($this->approvalType === 'pd') {
            if ($order->status !== 'pending_pd') {
                $this->dispatch('notify', type: 'error', message: 'Order must be pending PD review.');
                return;
            }

            $previousStatus = $order->status;

            $order->update([
                'status' => 'pending_merch',
                'pd_reviewed_by' => auth()->id(),
                'pd_reviewed_at' => now(),
                'pd_notes' => $this->approvalNotes,
            ]);

            // Record workflow history
            CommissionOrderWorkflowHistory::create([
                'commission_orders_id' => $order->id,
                'action' => 'pd_approved',
                'from_status' => $previousStatus,
                'to_status' => 'pending_merch',
                'from_user_id' => auth()->id(),
                'to_user_id' => null,
                'notes' => $this->approvalNotes ?: null,
            ]);

            if ($batch) {
                $batch->approvePd($this->approvalNotes);
            }

            // Notify the person who imported/forwarded the order
            $originalUser = $order->importedBy;
            if ($originalUser) {
                $originalUser->notify(new SkeletonOrderApprovedByPd($order, auth()->user(), $this->approvalNotes));
            }

            $this->orderStatus = 'pending_merch';
            $this->pd_reviewed_by_name = auth()->user()->name;
            $this->pd_reviewed_at = now()->format('d M Y H:i');
            $this->pd_notes = $this->approvalNotes;

            $this->dispatch('notify', type: 'success', message: 'PD approval submitted. Order sent to Merch.');

        } elseif ($this->approvalType === 'merch') {
            if ($order->status !== 'pending_merch') {
                $this->approvalError = 'Order must be pending Merch approval.';
                return;
            }

            // Check if all active lines have commission percentage set
            $linesWithoutCommission = 0;
            foreach ($this->lines as $lineData) {
                if ($lineData['cancelled'] ?? false) {
                    continue;
                }
                $commissionValue = $lineData['commission_percent'] ?? 0;
                if (!$commissionValue || $commissionValue <= 0) {
                    $linesWithoutCommission++;
                }
            }
            if ($linesWithoutCommission > 0) {
                $this->approvalError = "Cannot confirm: {$linesWithoutCommission} line(s) missing Commission %. Set commission on each line.";
                return;
            }

            // Check if all lines are matched
            $unmatchedCount = $order->lines()->where('cancelled', false)->whereNull('colourways_id')->count();
            if ($unmatchedCount > 0) {
                $this->approvalError = "Cannot confirm: {$unmatchedCount} lines still unmatched.";
                return;
            }

            // Check if all lines have drops with ex-factory dates
            $linesWithoutExfty = 0;
            foreach ($this->lines as $lineData) {
                if ($lineData['cancelled'] ?? false) {
                    continue;
                }
                $hasExfty = false;
                foreach ($lineData['drops'] ?? [] as $drop) {
                    if (!empty($drop['due_date'])) {
                        $hasExfty = true;
                        break;
                    }
                }
                if (!$hasExfty) {
                    $linesWithoutExfty++;
                }
            }
            if ($linesWithoutExfty > 0) {
                $this->approvalError = "Cannot confirm: {$linesWithoutExfty} line(s) missing ex-factory date. Add drops with dates on the Lines tab.";
                return;
            }
            
            // Save any unsaved changes first (including lines and drops)
            if ($this->isDirty) {
                $this->save();
                // Reload the order after save
                $order = CommissionOrder::findOrFail($this->orderId);
            }

            $previousStatus = $order->status;

            $order->update([
                'status' => 'confirmed',
                'confirmed_by' => auth()->id(),
                'confirmed_at' => now(),
                'merch_notes' => $this->approvalNotes,
            ]);

            // Record workflow history
            CommissionOrderWorkflowHistory::create([
                'commission_orders_id' => $order->id,
                'action' => 'merch_approved',
                'from_status' => $previousStatus,
                'to_status' => 'confirmed',
                'from_user_id' => auth()->id(),
                'to_user_id' => null,
                'notes' => $this->approvalNotes ?: null,
            ]);

            if ($batch) {
                $batch->approveMerch($this->approvalNotes);
            }

            $this->orderStatus = 'confirmed';
            $this->confirmed_by_name = auth()->user()->name;
            $this->confirmed_at = now()->format('d M Y H:i');
            $this->merch_notes = $this->approvalNotes;

            $this->dispatch('notify', type: 'success', message: 'Order confirmed! It is now visible in main Orders list.');
        }

        $this->closeApprovalModal();
        $this->dispatch('order-updated');
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }

    #[Computed]
    public function workflowHistory()
    {
        if (!$this->orderId) {
            return collect();
        }
        
        return CommissionOrderWorkflowHistory::where('commission_orders_id', $this->orderId)
            ->with(['fromUser', 'toUser'])
            ->orderBy('created_at', 'asc')
            ->get();
    }

    #[Computed]
    public function commissionFactories()
    {
        // Commission factories - Erdos, Ugur, and other commission-related factories
        return \App\Models\Suppliers::where('type', 'factory')
            ->whereIn('id', [5, 71]) // Erdos and Ugur
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function hasLines()
    {
        return count($this->lines) > 0;
    }

    #[Computed]
    public function totalQty()
    {
        $total = 0;
        foreach ($this->lines as $line) {
            foreach ($line['sizes'] ?? [] as $size) {
                $total += intval($size['qty'] ?? 0);
            }
        }
        return $total;
    }

    #[Computed]
    public function totalValue()
    {
        $total = 0;
        foreach ($this->lines as $line) {
            foreach ($line['sizes'] ?? [] as $size) {
                $price = floatval($size['price'] ?? 0);
                $qty = intval($size['qty'] ?? 0);
                $total += $price * $qty;
            }
        }
        return $total;
    }
    
    #[Computed]
    public function availableSizes()
    {
        return Sizes::orderBy('order')->get();
    }

    #[Computed]
    public function availableStyles()
    {
        if (!$this->customer_id || !$this->season_id) return collect();
        
        return Styles::with(['designs', 'style_versions.colourways'])
            ->where('customers_id', $this->customer_id)
            ->where('seasons_id', $this->season_id)
            ->when($this->styleSearch, function ($q) {
                $q->where(function ($sq) {
                    $sq->where('customer_ref', 'like', "%{$this->styleSearch}%")
                       ->orWhereHas('designs', fn($dq) => $dq->where('description', 'like', "%{$this->styleSearch}%")
                           ->orWhere('id', 'like', "%{$this->styleSearch}%"));
                });
            })
            ->limit(20)
            ->get();
    }

    public function updated($property)
    {
        $nonDataProperties = [
            'activeTab', 'showStylePicker', 'styleSearch', 'showConfirmClose',
            // Modal properties - these shouldn't trigger dirty state
            'showForwardModal', 'forwardToUserId', 'forwardNotes',
            'showApprovalModal', 'approvalType', 'approvalNotes',
            'showSendBackModal', 'sendBackToUserId', 'sendBackNotes',
            // Colourway picker
            'showColourwayPicker', 'colourwaySearch', 'editingLineIndex',
            // Quick add size inputs
            'quickAddSizeInputs',
        ];
        
        if (!in_array($property, $nonDataProperties)) {
            $this->isDirty = true;
        }
    }

    public function close()
    {
        if ($this->isDirty) {
            $this->dispatch('confirm-close');
        } else {
            $this->forceClose();
        }
    }

    public function forceClose()
    {
        $this->isOpen = false;
        $this->showConfirmClose = false;
        $this->showPdfViewer = false;
        $this->resetState();
        $this->dispatch('order-updated');
    }

    public function togglePdfViewer()
    {
        $this->showPdfViewer = !$this->showPdfViewer;
    }

    #[Computed]
    public function pdfUrl()
    {
        if (empty($this->source_files) || !$this->orderId) {
            return null;
        }
        // Use the preview route which serves the file inline
        return route('commission.orders.preview', ['order' => $this->orderId, 'fileIndex' => 0]);
    }

    public function cancelClose()
    {
        $this->showConfirmClose = false;
    }

    public function openStylePicker()
    {
        $this->styleSearch = '';
        $this->showStylePicker = true;
    }

    public function closeStylePicker()
    {
        $this->showStylePicker = false;
        $this->styleSearch = '';
    }

    // === Colourway Picker for changing a line's colourway ===
    
    public function openColourwayPicker(int $lineIndex)
    {
        $this->editingLineIndex = $lineIndex;
        
        // Pre-populate with RT number for quick colourway selection
        $line = $this->lines[$lineIndex] ?? null;
        if ($line && $line['rt_number']) {
            $this->colourwaySearch = (string) $line['rt_number'];
        } else {
            $this->colourwaySearch = '';
        }
        
        $this->showColourwayPicker = true;
    }
    
    public function closeColourwayPicker()
    {
        $this->showColourwayPicker = false;
        $this->editingLineIndex = null;
        $this->colourwaySearch = '';
    }
    
    public function changeLineColourway(int $colourwayId)
    {
        if ($this->editingLineIndex === null) return;
        
        $colourway = Colourways::with(['style_versions.styles.designs'])->find($colourwayId);
        if (!$colourway) return;
        
        $style = $colourway->style_versions?->styles;
        $design = $style?->designs;
        
        $this->lines[$this->editingLineIndex]['colourway_id'] = $colourway->id;
        $this->lines[$this->editingLineIndex]['rt_number'] = $design?->id;
        $this->lines[$this->editingLineIndex]['description'] = $design?->description ?? '';
        $this->lines[$this->editingLineIndex]['colourway_name'] = $colourway->name ?? '';
        $this->lines[$this->editingLineIndex]['customer_ref'] = $style?->customer_ref ?? '';
        $this->lines[$this->editingLineIndex]['match_status'] = 'confirmed';
        
        $this->isDirty = true;
        $this->closeColourwayPicker();
    }
    
    #[Computed]
    public function searchedStyles()
    {
        if (strlen($this->colourwaySearch) < 2) return collect();
        if (!$this->customer_id) return collect();
        if (!$this->season_id) return collect();
        
        $search = $this->colourwaySearch;
        $customerId = $this->customer_id;
        $seasonId = $this->season_id;
        
        return Styles::with(['designs', 'style_versions.colourways', 'style_versions.factories'])
            ->where('customers_id', $customerId)
            ->where('seasons_id', $seasonId)
            ->where(function ($query) use ($search) {
                $query->whereHas('designs', function ($q) use ($search) {
                    $q->where('id', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                })
                ->orWhere('customer_ref', 'like', "%{$search}%")
                ->orWhereHas('style_versions.colourways', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                })
                ->orWhereHas('style_versions', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                })
                ->orWhereHas('style_versions.factories', function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%");
                });
            })
            ->limit(15)
            ->get();
    }
    
    #[Computed]
    public function stylesForNewColourway()
    {
        if (!$this->customer_id) return collect();
        if (!$this->season_id) return collect();
        
        return Styles::with(['designs', 'style_versions'])
            ->where('customers_id', $this->customer_id)
            ->where('seasons_id', $this->season_id)
            ->orderBy('customer_ref')
            ->limit(100)
            ->get();
    }
    
    // === Create New Colourway ===
    
    public function openNewColourwayModal(?int $styleId = null)
    {
        if ($this->editingLineIndex !== null && !$styleId) {
            $line = $this->lines[$this->editingLineIndex];
            if (!empty($line['imported_style_ref']) && $this->customer_id) {
                $style = Styles::where('customer_ref', $line['imported_style_ref'])
                    ->where('customers_id', $this->customer_id)
                    ->first();
                $styleId = $style?->id;
            }
        }
        
        $this->newColourwayStyleId = $styleId;
        $this->newColourwayName = $this->editingLineIndex !== null 
            ? ($this->lines[$this->editingLineIndex]['imported_colour'] ?? '') 
            : '';
        $this->newColourwayCode = '';
        $this->showNewColourwayModal = true;
    }
    
    public function closeNewColourwayModal()
    {
        $this->showNewColourwayModal = false;
        $this->newColourwayStyleId = null;
        $this->newColourwayName = '';
        $this->newColourwayCode = '';
    }
    
    public function createColourway()
    {
        $this->validate([
            'newColourwayStyleId' => 'required|exists:styles,id',
            'newColourwayName' => 'required|string|max:255',
        ]);
        
        $style = Styles::with('style_versions')->find($this->newColourwayStyleId);
        $styleVersion = $style->style_versions->first();
        
        if (!$styleVersion) {
            $this->addError('newColourwayName', 'Style has no version');
            return;
        }
        
        // Split colourway names by comma and create each one
        $colourwayNames = array_map('trim', explode(',', $this->newColourwayName));
        $colourwayNames = array_filter($colourwayNames);
        
        $firstColourway = null;
        foreach ($colourwayNames as $colourwayName) {
            $colourway = Colourways::create([
                'style_versions_id' => $styleVersion->id,
                'name' => $colourwayName,
                'code' => $this->newColourwayCode ?: null,
            ]);
            if (!$firstColourway) {
                $firstColourway = $colourway;
            }
        }
        
        if ($this->editingLineIndex !== null && $firstColourway) {
            $this->changeLineColourway($firstColourway->id);
        }
        
        $this->closeNewColourwayModal();
        $this->closeColourwayPicker();
    }
    
    // === Create New Style ===
    
    public function openNewStyleModal()
    {
        if ($this->editingLineIndex !== null) {
            $line = $this->lines[$this->editingLineIndex];
            $this->newStyleCustomerRef = $line['imported_style_ref'] ?? '';
            $this->newStyleDescription = $line['imported_style_description'] ?? '';
            $this->newStyleYarn = '';
            $this->newStyleFactoryId = 5; // Default to Erdos
            $this->newStyleColourName = $line['imported_colour'] ?? '';
        } else {
            $this->newStyleCustomerRef = '';
            $this->newStyleDescription = '';
            $this->newStyleYarn = '';
            $this->newStyleFactoryId = 5; // Default to Erdos
            $this->newStyleColourName = '';
        }
        $this->showNewStyleModal = true;
    }

    public function closeNewStyleModal()
    {
        $this->showNewStyleModal = false;
        $this->newStyleCustomerRef = '';
        $this->newStyleDescription = '';
        $this->newStyleYarn = '';
        $this->newStyleFactoryId = 5;
        $this->newStyleColourName = '';
    }
    
    public function createStyle()
    {
        $this->validate([
            'newStyleCustomerRef' => 'required|string|max:100',
            'newStyleDescription' => 'required|string|max:255',
            'newStyleYarn' => 'required|string|max:255',
            'newStyleFactoryId' => 'required|exists:suppliers,id',
            'newStyleColourName' => 'required|string|max:255',
        ]);
        
        $design = \App\Models\Design::create([
            'description' => $this->newStyleDescription,
            'customers_id' => $this->customer_id,
            'yarn' => $this->newStyleYarn,
            'factory_id' => $this->newStyleFactoryId,
        ]);
        
        $style = Styles::create([
            'designs_id' => $design->id,
            'customer_ref' => $this->newStyleCustomerRef,
            'customers_id' => $this->customer_id,
            'seasons_id' => $this->season_id,
        ]);
        
        $styleVersion = \App\Models\StyleVersions::create([
            'styles_id' => $style->id,
            'factory_id' => $this->newStyleFactoryId,
            'name' => 'A',
        ]);
        
        // Split colourway names by comma and create each one
        $colourwayNames = array_map('trim', explode(',', $this->newStyleColourName));
        $colourwayNames = array_filter($colourwayNames); // Remove empty strings
        
        $firstColourway = null;
        foreach ($colourwayNames as $colourwayName) {
            $colourway = Colourways::create([
                'style_versions_id' => $styleVersion->id,
                'name' => $colourwayName,
            ]);
            if (!$firstColourway) {
                $firstColourway = $colourway;
            }
        }
        
        if ($this->editingLineIndex !== null && $firstColourway) {
            $this->changeLineColourway($firstColourway->id);
        }
        
        $this->closeNewStyleModal();
        $this->closeColourwayPicker();
    }

    public function addColourway(int $colourwayId)
    {
        $colourway = Colourways::with(['style_versions.styles.designs'])->find($colourwayId);
        if (!$colourway) return;

        $style = $colourway->style_versions?->styles;
        $design = $style?->designs;

        $this->lines[] = [
            'id' => null,
            'colourway_id' => $colourway->id,
            'rt_number' => $design?->id,
            'description' => $design?->description ?? '',
            'colourway_name' => $colourway->name ?? '',
            'customer_ref' => $style?->customer_ref ?? '',
            'imported_style_ref' => null,
            'imported_colour' => null,
            'match_status' => 'confirmed',
            'commission_percent' => $this->default_commission_percent,
            'discount_percent' => $this->default_discount_percent,
            'cancelled' => false,
            'sizes' => [],
            'drops' => [],
        ];

        $this->isDirty = true;
        $this->closeStylePicker();
    }

    public function removeLine(int $index)
    {
        if (isset($this->lines[$index])) {
            unset($this->lines[$index]);
            $this->lines = array_values($this->lines);
            $this->isDirty = true;
        }
    }
    
    public function updateLineQty(int $lineIndex, int $sizeId, $qty)
    {
        if (!isset($this->lines[$lineIndex])) return;
        
        $qty = (int) $qty;
        $sizes = $this->lines[$lineIndex]['sizes'] ?? [];
        
        // Find existing size entry
        $found = false;
        foreach ($sizes as $i => $size) {
            if ($size['size_id'] == $sizeId) {
                if ($qty > 0) {
                    $this->lines[$lineIndex]['sizes'][$i]['qty'] = $qty;
                } else {
                    // Remove size if qty is 0
                    unset($this->lines[$lineIndex]['sizes'][$i]);
                    $this->lines[$lineIndex]['sizes'] = array_values($this->lines[$lineIndex]['sizes']);
                }
                $found = true;
                break;
            }
        }
        
        // Add new size entry if not found and qty > 0
        if (!$found && $qty > 0) {
            $sizeName = Sizes::find($sizeId)?->name ?? '';
            $this->lines[$lineIndex]['sizes'][] = [
                'id' => null,
                'size_id' => $sizeId,
                'size_name' => $sizeName,
                'qty' => $qty,
                'price' => 0,
            ];
        }
        
        $this->isDirty = true;
    }
    
    public function updateLinePrice(int $lineIndex, int $sizeId, $price)
    {
        if (!isset($this->lines[$lineIndex])) return;
        
        $price = (float) $price;
        $sizes = $this->lines[$lineIndex]['sizes'] ?? [];
        
        // Find existing size entry
        $found = false;
        foreach ($sizes as $i => $size) {
            if ($size['size_id'] == $sizeId) {
                $this->lines[$lineIndex]['sizes'][$i]['price'] = $price;
                $found = true;
                break;
            }
        }
        
        // Add new size entry if not found and price > 0
        // This allows setting price before quantity
        if (!$found && $price > 0) {
            $sizeName = Sizes::find($sizeId)?->name ?? '';
            $this->lines[$lineIndex]['sizes'][] = [
                'id' => null,
                'size_id' => $sizeId,
                'size_name' => $sizeName,
                'qty' => 0,
                'price' => $price,
            ];
        }
        
        $this->isDirty = true;
    }
    
    #[Renderless]
    public function getLineData(int $index): ?array
    {
        return $this->lines[$index] ?? null;
    }
    
    public function addSize(int $lineIndex)
    {
        if (isset($this->lines[$lineIndex])) {
            $this->lines[$lineIndex]['sizes'][] = [
                'id' => null,
                'size_id' => null,
                'size_name' => '',
                'qty' => 0,
                'price' => 0,
            ];
            $this->isDirty = true;
        }
    }
    
    public function removeSize(int $lineIndex, int $sizeIndex)
    {
        if (isset($this->lines[$lineIndex]['sizes'][$sizeIndex])) {
            unset($this->lines[$lineIndex]['sizes'][$sizeIndex]);
            $this->lines[$lineIndex]['sizes'] = array_values($this->lines[$lineIndex]['sizes']);
            $this->isDirty = true;
        }
    }
    
    // === Quick Add Size (inline) ===
    
    public function addQuickSizeInput()
    {
        if (!$this->canEditQuantities) {
            return;
        }
        
        // Initialize values for each line
        $values = [];
        foreach ($this->lines as $lineIndex => $line) {
            $values[$lineIndex] = ['qty' => 0, 'price' => 0];
        }
        
        // Add a new input with name and per-line values
        $this->quickAddSizeInputs[] = [
            'name' => '',
            'values' => $values,
        ];
    }
    
    public function addQuickSize(int $index)
    {
        if (!isset($this->quickAddSizeInputs[$index])) {
            return;
        }
        
        $sizeName = trim($this->quickAddSizeInputs[$index]['name'] ?? '');
        $pendingValues = $this->quickAddSizeInputs[$index]['values'] ?? [];
        
        if (empty($sizeName)) {
            // Remove empty input
            unset($this->quickAddSizeInputs[$index]);
            $this->quickAddSizeInputs = array_values($this->quickAddSizeInputs);
            return;
        }
        
        // Find or create the size
        $size = Sizes::firstOrCreate(
            ['name' => $sizeName],
            ['order' => 999]
        );
        
        $sizeId = $size->id;
        
        // Add the size to all lines that don't already have it, with pending values
        $addedCount = 0;
        
        foreach ($this->lines as $lineIndex => $line) {
            // Check if size already exists in this line
            $existsInLine = false;
            foreach ($line['sizes'] ?? [] as $sizeData) {
                if ($sizeData['size_id'] == $sizeId) {
                    $existsInLine = true;
                    break;
                }
            }
            
            if (!$existsInLine) {
                $lineValues = $pendingValues[$lineIndex] ?? ['qty' => 0, 'price' => 0];
                $this->lines[$lineIndex]['sizes'][] = [
                    'id' => null,
                    'size_id' => $sizeId,
                    'size_name' => $sizeName,
                    'qty' => (int) ($lineValues['qty'] ?? 0),
                    'price' => (float) ($lineValues['price'] ?? 0),
                ];
                $addedCount++;
            }
        }
        
        // Remove this input after adding
        unset($this->quickAddSizeInputs[$index]);
        $this->quickAddSizeInputs = array_values($this->quickAddSizeInputs);
        
        $this->isDirty = true;
        
        if ($addedCount > 0) {
            $this->dispatch('notify', type: 'success', message: "Size '{$sizeName}' added.");
        }
    }
    
    public function removeQuickSizeInput(int $index)
    {
        if (isset($this->quickAddSizeInputs[$index])) {
            unset($this->quickAddSizeInputs[$index]);
            $this->quickAddSizeInputs = array_values($this->quickAddSizeInputs);
        }
    }

    public function addDrop(int $lineIndex)
    {
        if (isset($this->lines[$lineIndex])) {
            $dropSizes = [];
            foreach ($this->lines[$lineIndex]['sizes'] ?? [] as $size) {
                if ($size['size_id']) {
                    $dropSizes[] = [
                        'id' => null,
                        'size_id' => $size['size_id'],
                        'size_name' => $size['size_name'],
                        'qty' => 0,
                    ];
                }
            }
            
            $this->lines[$lineIndex]['drops'][] = [
                'id' => null,
                'due_date' => '',
                'sizes' => $dropSizes,
            ];
            $this->isDirty = true;
        }
    }

    public function removeDrop(int $lineIndex, int $dropIndex)
    {
        if (isset($this->lines[$lineIndex]['drops'][$dropIndex])) {
            unset($this->lines[$lineIndex]['drops'][$dropIndex]);
            $this->lines[$lineIndex]['drops'] = array_values($this->lines[$lineIndex]['drops']);
            $this->isDirty = true;
        }
    }

    /**
     * Apply default commission percent to all lines
     */
    public function applyCommissionToAllLines()
    {
        if ($this->default_commission_percent === null || $this->default_commission_percent <= 0) {
            return;
        }

        foreach ($this->lines as $lineIndex => $line) {
            $this->lines[$lineIndex]['commission_percent'] = $this->default_commission_percent;
        }

        $this->isDirty = true;
        $this->dispatch('notify', type: 'success', message: 'Commission % applied to all lines.');
    }

    /**
     * Apply default discount percent to all lines
     */
    public function applyDiscountToAllLines()
    {
        if ($this->default_discount_percent === null) {
            return;
        }

        foreach ($this->lines as $lineIndex => $line) {
            $this->lines[$lineIndex]['discount_percent'] = $this->default_discount_percent;
        }

        $this->isDirty = true;
        $this->dispatch('notify', type: 'success', message: 'Discount % applied to all lines.');
    }

    /**
     * Set ex-factory date for ALL lines
     */
    public function setAllLinesExfty(?string $date)
    {
        if (!$date) {
            return;
        }

        foreach ($this->lines as $lineIndex => $line) {
            $this->setLineExfty($lineIndex, $date);
        }

        $this->dispatch('notify', type: 'success', message: 'Ex-factory date set for all lines.');
    }

    /**
     * Clear ex-factory dates from all lines
     */
    public function clearAllLinesExfty()
    {
        foreach ($this->lines as $lineIndex => $line) {
            if (isset($this->lines[$lineIndex]['drops'][0])) {
                $this->lines[$lineIndex]['drops'][0]['due_date'] = '';
            }
        }
        $this->isDirty = true;
        $this->dispatch('notify', type: 'success', message: 'Ex-factory dates cleared.');
    }

    /**
     * Set the ex-factory date for a line (creates drop if needed)
     */
    public function setLineExfty(int $lineIndex, ?string $date)
    {
        if (!isset($this->lines[$lineIndex])) {
            return;
        }

        // Check if drops array exists, create if not
        if (!isset($this->lines[$lineIndex]['drops']) || empty($this->lines[$lineIndex]['drops'])) {
            // Create a new drop with sizes from the line
            $dropSizes = [];
            foreach ($this->lines[$lineIndex]['sizes'] ?? [] as $size) {
                if ($size['size_id']) {
                    $dropSizes[] = [
                        'id' => null,
                        'size_id' => $size['size_id'],
                        'size_name' => $size['size_name'] ?? '',
                        'qty' => $size['qty'] ?? 0,
                    ];
                }
            }
            
            $this->lines[$lineIndex]['drops'] = [[
                'id' => null,
                'due_date' => $date ?: '',
                'sizes' => $dropSizes,
            ]];
        } else {
            // Update existing drop
            $this->lines[$lineIndex]['drops'][0]['due_date'] = $date ?: '';
        }

        $this->isDirty = true;
    }

    /**
     * Create drops for order lines that don't have any drops yet
     * This is called when an order is confirmed to ensure it appears on shipment schedule
     */
    protected function createDropsForOrder(CommissionOrder $order)
    {
        $order->load('lines.quantities.size', 'lines.drops');
        
        foreach ($order->lines as $line) {
            // Skip cancelled lines
            if ($line->cancelled) {
                continue;
            }
            
            // Skip lines that already have drops
            if ($line->drops->count() > 0) {
                continue;
            }
            
            // Create a drop for this line
            $drop = CommissionDrop::create([
                'commission_order_lines_id' => $line->id,
                'exfty' => null, // Ex-factory date to be set later
            ]);
            
            // Create drop sizes based on line quantities
            foreach ($line->quantities as $qty) {
                if ($qty->size_id && $qty->qty > 0) {
                    CommissionDropSize::create([
                        'commission_drops_id' => $drop->id,
                        'sizes_id' => $qty->size_id,
                        'qty' => $qty->qty,
                    ]);
                }
            }
        }
    }

    public function save()
    {
        $this->validate();

        $wasNewOrder = $this->isNewOrder;

        try {
            DB::beginTransaction();

            if ($this->isNewOrder) {
                $order = CommissionOrder::create([
                    'seasons_id' => $this->season_id,
                    'customers_id' => $this->customer_id,
                    'customer_po' => $this->customer_po,
                    'order_date' => $this->order_date,
                    'status' => 'confirmed',
                    'confirmed_by' => auth()->id(),
                    'confirmed_at' => now(),
                ]);
                
                $this->orderId = $order->id;
                $this->isNewOrder = false;
            } else {
                CommissionOrder::where('id', $this->orderId)->update([
                    'seasons_id' => $this->season_id,
                    'customers_id' => $this->customer_id,
                    'customer_po' => $this->customer_po,
                    'order_date' => $this->order_date,
                ]);
            }

            // Track existing line IDs for deletion
            $existingLineIds = [];
            
            foreach ($this->lines as $lineData) {
                if ($lineData['id']) {
                    // Update existing line
                    $lineId = $lineData['id'];
                    $existingLineIds[] = $lineId;
                    
                    // Update the line's colourway, match status, and commission/discount
                    $lineModel = CommissionOrderLine::find($lineId);
                    if ($lineModel) {
                        $lineModel->update([
                            'colourways_id' => $lineData['colourway_id'] ?: null,
                            'match_status' => $lineData['match_status'] ?? 'confirmed',
                            'commission_percent' => $lineData['commission_percent'] ?? null,
                            'discount_percent' => $lineData['discount_percent'] ?? null,
                        ]);
                    }
                    
                    // Update quantities
                    $existingQtyIds = [];
                    foreach ($lineData['sizes'] ?? [] as $sizeData) {
                        if ($sizeData['id']) {
                            $qtyModel = CommissionOrderLineQuantity::find($sizeData['id']);
                            if ($qtyModel) {
                                $qtyModel->update([
                                    'sizes_id' => $sizeData['size_id'],
                                    'qty' => (int) ($sizeData['qty'] ?? 0),
                                    'price' => $sizeData['price'] ?? 0,
                                ]);
                            }
                            $existingQtyIds[] = $sizeData['id'];
                        } elseif ($sizeData['size_id']) {
                            $newQty = CommissionOrderLineQuantity::create([
                                'commission_order_lines_id' => $lineId,
                                'sizes_id' => $sizeData['size_id'],
                                'qty' => (int) ($sizeData['qty'] ?? 0),
                                'price' => $sizeData['price'] ?? 0,
                            ]);
                            $existingQtyIds[] = $newQty->id;
                        }
                    }
                    
                    // Delete removed quantities (use model for auditing)
                    CommissionOrderLineQuantity::where('commission_order_lines_id', $lineId)
                        ->whereNotIn('id', $existingQtyIds)
                        ->get()
                        ->each(fn($qty) => $qty->delete());
                    
                    // Update drops
                    $existingDropIds = [];
                    foreach ($lineData['drops'] ?? [] as $dropData) {
                        if ($dropData['id']) {
                            CommissionDrop::where('id', $dropData['id'])->update([
                                'exfty' => $dropData['due_date'] ?: null,
                            ]);
                            $dropId = $dropData['id'];
                            $existingDropIds[] = $dropId;
                            
                            // Update drop sizes
                            $existingDropSizeIds = [];
                            foreach ($dropData['sizes'] ?? [] as $dropSizeData) {
                                if ($dropSizeData['id']) {
                                    CommissionDropSize::where('id', $dropSizeData['id'])->update([
                                        'qty' => $dropSizeData['qty'] ?? 0,
                                    ]);
                                    $existingDropSizeIds[] = $dropSizeData['id'];
                                } elseif ($dropSizeData['size_id']) {
                                    $newDropSize = CommissionDropSize::create([
                                        'commission_drops_id' => $dropId,
                                        'sizes_id' => $dropSizeData['size_id'],
                                        'qty' => $dropSizeData['qty'] ?? 0,
                                    ]);
                                    $existingDropSizeIds[] = $newDropSize->id;
                                }
                            }
                            
                            CommissionDropSize::where('commission_drops_id', $dropId)
                                ->whereNotIn('id', $existingDropSizeIds)
                                ->delete();
                        } else {
                            // Create new drop
                            $newDrop = CommissionDrop::create([
                                'commission_order_lines_id' => $lineId,
                                'exfty' => $dropData['due_date'] ?: null,
                            ]);
                            $existingDropIds[] = $newDrop->id;
                            
                            foreach ($dropData['sizes'] ?? [] as $dropSizeData) {
                                if ($dropSizeData['size_id']) {
                                    CommissionDropSize::create([
                                        'commission_drops_id' => $newDrop->id,
                                        'sizes_id' => $dropSizeData['size_id'],
                                        'qty' => $dropSizeData['qty'] ?? 0,
                                    ]);
                                }
                            }
                        }
                    }
                    
                    // Delete removed drops
                    CommissionDrop::where('commission_order_lines_id', $lineId)
                        ->whereNotIn('id', $existingDropIds)
                        ->delete();
                        
                } else {
                    // Create new line
                    $line = CommissionOrderLine::create([
                        'commission_orders_id' => $this->orderId,
                        'colourways_id' => $lineData['colourway_id'],
                        'match_status' => 'confirmed',
                        'commission_percent' => $lineData['commission_percent'] ?? $this->default_commission_percent,
                        'discount_percent' => $lineData['discount_percent'] ?? $this->default_discount_percent,
                    ]);
                    $existingLineIds[] = $line->id;
                    
                    // Create quantities
                    foreach ($lineData['sizes'] ?? [] as $sizeData) {
                        if ($sizeData['size_id']) {
                            CommissionOrderLineQuantity::create([
                                'commission_order_lines_id' => $line->id,
                                'sizes_id' => $sizeData['size_id'],
                                'qty' => (int) ($sizeData['qty'] ?? 0),
                                'price' => $sizeData['price'] ?? 0,
                            ]);
                        }
                    }
                    
                    // Create drops
                    foreach ($lineData['drops'] ?? [] as $dropData) {
                        $drop = CommissionDrop::create([
                            'commission_order_lines_id' => $line->id,
                            'exfty' => $dropData['due_date'] ?: null,
                        ]);
                        
                        foreach ($dropData['sizes'] ?? [] as $dropSizeData) {
                            if ($dropSizeData['size_id']) {
                                CommissionDropSize::create([
                                    'commission_drops_id' => $drop->id,
                                    'sizes_id' => $dropSizeData['size_id'],
                                    'qty' => $dropSizeData['qty'] ?? 0,
                                ]);
                            }
                        }
                    }
                }
            }

            // Delete removed lines
            if (!$this->isNewOrder && !empty($existingLineIds)) {
                CommissionOrderLine::where('commission_orders_id', $this->orderId)
                    ->whereNotIn('id', $existingLineIds)
                    ->delete();
            }

            DB::commit();
            
            // Send notification email for new orders with a customer
            if ($wasNewOrder && $this->customer_id) {
                try {
                    $order = CommissionOrder::with('customer')->find($this->orderId);
                    Mail::to('sue@roberttodds.com')
                        ->cc('oliver@roberttodds.com')
                        ->queue(new OrderCreated($order));
                } catch (\Exception $e) {
                    \Log::error('Failed to send OrderCreated email: ' . $e->getMessage());
                }
            }
            
            $this->isDirty = false;
            $this->dispatch('order-saved');
            session()->flash('co-message', 'Order saved successfully!');
            
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Commission Order save error: ' . $e->getMessage());
            session()->flash('co-error', 'Error saving order: ' . $e->getMessage());
        }
    }

    #[Computed(persist: false)]
    public function auditHistory()
    {
        if (!$this->orderId) {
            return collect();
        }

        $order = CommissionOrder::find($this->orderId);
        if (!$order) {
            return collect();
        }

        // Get related IDs
        $lineIds = CommissionOrderLine::where('commission_orders_id', $this->orderId)->pluck('id')->toArray();
        $qtyIds = CommissionOrderLineQuantity::whereIn('commission_order_lines_id', $lineIds)->pluck('id')->toArray();

        // Build query for all models
        $audits = DB::table('audits')
            ->where(function ($query) use ($order, $lineIds, $qtyIds) {
                $query->where(function ($q) use ($order) {
                    $q->where('auditable_type', 'App\\Models\\CommissionOrder')
                      ->where('auditable_id', $order->id);
                });
                
                if (!empty($lineIds)) {
                    $query->orWhere(function ($q) use ($lineIds) {
                        $q->where('auditable_type', 'App\\Models\\CommissionOrderLine')
                          ->whereIn('auditable_id', $lineIds);
                    });
                }
                
                if (!empty($qtyIds)) {
                    $query->orWhere(function ($q) use ($qtyIds) {
                        $q->where('auditable_type', 'App\\Models\\CommissionOrderLineQuantity')
                          ->whereIn('auditable_id', $qtyIds);
                    });
                }
            })
            ->orderBy('created_at', 'desc')
            ->orderBy('id', 'desc')
            ->limit(100)
            ->get();

        // Get user names
        $userIds = $audits->pluck('user_id')->unique()->filter()->toArray();
        $users = User::whereIn('id', $userIds)->pluck('name', 'id');

        // Get line info for context
        $lineInfo = CommissionOrderLine::whereIn('id', $lineIds)
            ->with('colourway.style_versions.styles.designs')
            ->get()
            ->keyBy('id');
        
        $sizes = Sizes::pluck('name', 'id')->toArray();

        // Process audits
        $result = collect($audits)->map(function ($audit) use ($users, $lineInfo) {
            $oldValues = json_decode($audit->old_values, true) ?? [];
            $newValues = json_decode($audit->new_values, true) ?? [];
            $modelType = class_basename($audit->auditable_type);
            
            $itemName = null;
            if ($modelType === 'CommissionOrder') {
                $modelType = 'Order';
            } elseif ($modelType === 'CommissionOrderLine') {
                $modelType = 'Line';
                $line = $lineInfo[$audit->auditable_id] ?? null;
                if ($line) {
                    $colourway = $line->colourway;
                    $design = $colourway?->style_versions?->styles?->designs;
                    $itemName = $design ? $design->id . ' ' . ($colourway->name ?? '') : $colourway?->name;
                }
            } elseif ($modelType === 'CommissionOrderLineQuantity') {
                $modelType = 'Quantity';
            }
            
            $changes = [];
            $skipFields = ['id', 'commission_orders_id', 'commission_order_lines_id', 'colourways_id', 'created_at', 'updated_at', 'deleted_at'];
            
            foreach ($newValues as $field => $newValue) {
                if (in_array($field, $skipFields)) continue;
                $oldValue = $oldValues[$field] ?? null;
                if ($oldValue === $newValue) continue;
                
                $changes[] = [
                    'field' => ucwords(str_replace('_', ' ', $field)),
                    'old' => $oldValue ?? '—',
                    'new' => $newValue ?? '—',
                ];
            }
            
            return (object) [
                'id' => $audit->id,
                'event' => ucfirst($audit->event),
                'model_type' => $modelType,
                'item_name' => $itemName,
                'user_name' => $users[$audit->user_id] ?? 'System',
                'created_at' => \Carbon\Carbon::parse($audit->created_at),
                'changes' => $changes,
            ];
        })->filter(function ($audit) {
            if ($audit->event === 'Updated' && empty($audit->changes)) return false;
            return true;
        });

        return $result->sortByDesc('created_at')->values();
    }

    public function render()
    {
        return view('livewire.commission.order-edit');
    }
}
