<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use App\Models\CommissionInvoiceGroup;
use App\Models\Customer;
use Livewire\Attributes\Url;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\CommissionInvoiceReady;

class Invoices extends Component
{
    #[Url]
    public string $search = '';
    
    #[Url]
    public string $customer = '';
    
    #[Url]
    public string $status = '';
    
    #[Url]
    public string $dateFrom = '';
    
    #[Url]
    public string $dateTo = '';
    
    #[Url]
    public bool $showCompleted = false;
    
    #[Url]
    public string $sortBy = 'created_at';
    
    #[Url]
    public string $sortDir = 'desc';
    
    // Infinite scroll
    public int $perPage = 25;
    public array $loadedInvoices = [];
    public bool $hasMore = true;
    public bool $isLoading = false;

    #[On('invoice-saved')]
    #[On('invoice-imported')]
    public function refreshInvoices()
    {
        unset($this->stats);
        $this->loadInitialInvoices();
    }

    public function mount()
    {
        $this->loadInitialInvoices();
    }

    public function loadInitialInvoices()
    {
        $this->loadedInvoices = [];
        $this->hasMore = true;
        $this->loadMore();
    }

    public function updatedSearch()
    {
        $this->loadInitialInvoices();
    }

    public function updatedCustomer()
    {
        $this->loadInitialInvoices();
    }

    public function updatedStatus()
    {
        $this->loadInitialInvoices();
    }

    public function updatedDateFrom()
    {
        $this->loadInitialInvoices();
    }

    public function updatedDateTo()
    {
        $this->loadInitialInvoices();
    }

    public function clearFilters()
    {
        $this->reset(['search', 'customer', 'status', 'dateFrom', 'dateTo', 'showCompleted']);
        $this->loadInitialInvoices();
    }

    public function toggleShowCompleted()
    {
        $this->showCompleted = !$this->showCompleted;
        $this->loadInitialInvoices();
    }

    public function sort(string $column)
    {
        if ($this->sortBy === $column) {
            $this->sortDir = $this->sortDir === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $column;
            $this->sortDir = 'desc';
        }
        $this->loadInitialInvoices();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function statusOptions()
    {
        return [
            'draft' => ['label' => 'Draft', 'icon' => 'fa-file-alt', 'color' => 'secondary'],
            'ready' => ['label' => 'Sent', 'icon' => 'fa-check-circle', 'color' => 'success'],
            'sent_to_zoho' => ['label' => 'Sent', 'icon' => 'fa-check-circle', 'color' => 'success'],
            'complete' => ['label' => 'Sent', 'icon' => 'fa-check-circle', 'color' => 'success'],
        ];
    }

    protected function buildQuery()
    {
        return CommissionInvoiceGroup::query()
            ->with([
                'customer:id,name,currency',
                'uploadedBy:id,name',
            ])
            ->withCount([
                'drops',
                'drops as unshipped_drops_count' => fn($q) => $q->where('shipped', false),
            ])
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('reference', 'like', "%{$this->search}%")
                      ->orWhere('id', 'like', "%{$this->search}%")
                      ->orWhere('supplier_invoice_ref', 'like', "%{$this->search}%")
                      ->orWhereHas('customer', fn($cq) => $cq->where('name', 'like', "%{$this->search}%"));
                });
            })
            ->when($this->customer, fn($q) => $q->where('customers_id', $this->customer))
            ->when($this->status === 'notSentToZoho', fn($q) => $q->whereIn('status', ['draft', 'ready']))
            ->when($this->status && $this->status !== 'notSentToZoho', fn($q) => $q->where('status', $this->status))
            ->when($this->dateFrom, fn($q) => $q->whereDate('invoice_date', '>=', $this->dateFrom))
            ->when($this->dateTo, fn($q) => $q->whereDate('invoice_date', '<=', $this->dateTo))
            ->when(!$this->showCompleted && !$this->status, fn($q) => $q->where('status', 'draft'))
            ->orderBy($this->sortBy, $this->sortDir);
    }

    public function loadMore()
    {
        if (!$this->hasMore || $this->isLoading) {
            return;
        }

        $this->isLoading = true;

        $loadedIds = collect($this->loadedInvoices)->pluck('id')->toArray();
        
        $query = $this->buildQuery();
        
        if (!empty($loadedIds)) {
            $query->whereNotIn('commission_invoice_groups.id', $loadedIds);
        }

        $newInvoices = $query->limit($this->perPage)->get();

        if ($newInvoices->count() < $this->perPage) {
            $this->hasMore = false;
        }

        foreach ($newInvoices as $invoice) {
            $this->loadedInvoices[] = [
                'id' => $invoice->id,
                'reference' => $invoice->reference,
                'supplier_invoice_ref' => $invoice->supplier_invoice_ref,
                'customer_name' => $invoice->customer?->name,
                'customer_currency' => $invoice->customer?->currency ?? '$',
                'invoice_date' => $invoice->invoice_date?->format('d M Y'),
                'status' => $invoice->status,
                'drops_count' => $invoice->drops_count,
                'unshipped_drops_count' => $invoice->unshipped_drops_count,
                'total_qty' => $invoice->total_qty ?? 0,
                'total_value' => $invoice->total_value ?? 0,
                'invoice_file_path' => $invoice->invoice_file_path,
            ];
        }

        $this->isLoading = false;
    }
    
    #[Computed]
    public function stats()
    {
        return [
            'total' => CommissionInvoiceGroup::count(),
            'draft' => CommissionInvoiceGroup::where('status', 'draft')->count(),
            'ready' => CommissionInvoiceGroup::where('status', 'ready')->count(),
            'sent_to_zoho' => CommissionInvoiceGroup::where('status', 'sent_to_zoho')->count(),
            'complete' => CommissionInvoiceGroup::where('status', 'complete')->count(),
        ];
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if ($this->search) $count++;
        if ($this->customer) $count++;
        if ($this->status) $count++;
        if ($this->dateFrom || $this->dateTo) $count++;
        if ($this->showCompleted) $count++;
        return $count;
    }

    /**
     * Download invoice file
     */
    public function downloadFile(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if (!$invoice->invoice_file_path) {
            $this->dispatch('notify', type: 'error', message: 'No file attached to this invoice.');
            return;
        }
        
        if (!Storage::disk('local')->exists($invoice->invoice_file_path)) {
            $this->dispatch('notify', type: 'error', message: 'File not found on disk.');
            return;
        }
        
        return Storage::disk('local')->download(
            $invoice->invoice_file_path,
            $invoice->invoice_file_name ?? basename($invoice->invoice_file_path)
        );
    }

    /**
     * Send to Zoho (mark as sent)
     */
    public function sendToZoho(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if ($invoice->status !== 'ready') {
            $this->dispatch('notify', type: 'error', message: 'Invoice must be in "ready" status to send to Zoho.');
            return;
        }
        
        $invoice->update([
            'status' => 'sent_to_zoho',
            'sent_to_zoho_at' => now(),
        ]);
        
        $this->dispatch('notify', type: 'success', message: 'Invoice marked as sent to Zoho.');
        $this->refreshInvoices();
    }

    /**
     * Mark as ready (requires all drops to be shipped)
     */
    public function markReady(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if ($invoice->status !== 'draft') {
            $this->dispatch('notify', type: 'error', message: 'Only draft invoices can be marked as ready.');
            return;
        }
        
        // Check if invoice has drops attached
        if ($invoice->drops()->count() === 0) {
            $this->dispatch('notify', type: 'error', message: 'Invoice must have at least one drop attached.');
            return;
        }
        
        // Check if all drops are shipped
        $unshippedCount = $invoice->drops()->where('shipped', false)->count();
        if ($unshippedCount > 0) {
            $this->dispatch('notify', type: 'error', message: "Cannot send: {$unshippedCount} drop(s) not marked as shipped.");
            return;
        }
        
        $invoice->update(['status' => 'ready']);
        $invoice->recalculateTotals();
        
        // Send email to finance
        $this->sendFinanceEmail($invoice);
        
        $this->dispatch('notify', type: 'success', message: 'Invoice sent to finance.');
        $this->refreshInvoices();
    }

    /**
     * Mark all drops as shipped and send to finance
     */
    public function markShippedAndReady(int $invoiceId)
    {
        $invoice = CommissionInvoiceGroup::findOrFail($invoiceId);
        
        if ($invoice->status !== 'draft') {
            $this->dispatch('notify', type: 'error', message: 'Only draft invoices can be processed.');
            return;
        }
        
        // Check if invoice has drops attached
        if ($invoice->drops()->count() === 0) {
            $this->dispatch('notify', type: 'error', message: 'Invoice must have at least one drop attached.');
            return;
        }
        
        // Mark all drops as shipped
        $invoice->drops()->where('shipped', false)->update([
            'shipped' => true,
            'shipped_date' => now(),
        ]);
        
        // Also update shipped_qty for drop sizes if not set
        foreach ($invoice->drops as $drop) {
            $drop->sizes()->whereNull('shipped_qty')->update([
                'shipped_qty' => \DB::raw('qty'),
            ]);
        }
        
        $invoice->update(['status' => 'ready']);
        $invoice->recalculateTotals();
        
        // Send email to finance
        $this->sendFinanceEmail($invoice);
        
        $this->dispatch('notify', type: 'success', message: 'Drops marked as shipped and invoice sent to finance.');
        $this->refreshInvoices();
    }

    /**
     * Send email notification to finance
     */
    protected function sendFinanceEmail(CommissionInvoiceGroup $invoice)
    {
        $financeEmail = config('services.commission.finance_email');
        
        if (!$financeEmail) {
            return;
        }
        
        try {
            $invoice->load('customer');
            Mail::to($financeEmail)->queue(new CommissionInvoiceReady(
                $invoice,
                auth()->user()->name ?? 'System'
            ));
        } catch (\Exception $e) {
            \Log::error('Failed to send commission invoice email: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.commission.invoices')->layout('layouts.commission');
    }
}

