<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\CommissionInvoiceGroup;
use App\Models\CommissionDrop;
use App\Models\Customer;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use App\Mail\CommissionInvoiceReady;

class InvoiceEdit extends Component
{
    use WithFileUploads;

    public bool $isOpen = false;
    public ?int $invoiceId = null;
    public bool $isDirty = false;
    public bool $isNewInvoice = false;
    public string $activeTab = 'details';
    
    // Invoice fields
    public ?int $customer_id = null;
    public ?string $reference = '';
    public ?string $supplierInvoiceRef = '';
    public ?string $invoiceDate = null;
    public ?string $notes = '';
    public $invoiceFile = null;
    public ?string $existingFileName = null;
    public ?string $status = 'draft';
    
    // Drops selection
    public array $selectedDrops = [];
    public string $dropSearch = '';

    protected function rules()
    {
        return [
            'customer_id' => 'required|exists:customers,id',
            'reference' => 'nullable|string|max:255',
            'supplierInvoiceRef' => 'nullable|string|max:255',
            'invoiceDate' => 'nullable|date',
            'invoiceFile' => 'nullable|file|mimes:pdf,jpg,jpeg,png,gif|max:10240',
        ];
    }

    #[On('open-invoice-edit')]
    public function openInvoice(int $invoiceId)
    {
        $this->resetState();
        $this->invoiceId = $invoiceId;
        $this->isNewInvoice = false;
        $this->loadInvoice();
        $this->isOpen = true;
    }

    #[On('create-new-invoice')]
    public function createNew()
    {
        $this->resetState();
        $this->invoiceId = null;
        $this->isNewInvoice = true;
        $this->invoiceDate = now()->format('Y-m-d');
        $this->status = 'draft';
        $this->selectedDrops = [];
        $this->isOpen = true;
    }

    protected function resetState()
    {
        $this->reset([
            'invoiceId', 'isDirty', 'isNewInvoice',
            'customer_id', 'reference', 'supplierInvoiceRef',
            'invoiceDate', 'notes', 'invoiceFile', 'existingFileName',
            'status', 'selectedDrops', 'dropSearch', 'activeTab'
        ]);
        $this->activeTab = 'details';
    }

    protected function loadInvoice()
    {
        $invoice = CommissionInvoiceGroup::with('drops')->find($this->invoiceId);

        if (!$invoice) return;

        $this->customer_id = $invoice->customers_id;
        $this->reference = $invoice->reference;
        $this->supplierInvoiceRef = $invoice->supplier_invoice_ref;
        $this->invoiceDate = $invoice->invoice_date?->format('Y-m-d');
        $this->notes = $invoice->notes;
        $this->existingFileName = $invoice->invoice_file_name;
        $this->status = $invoice->status;
        $this->selectedDrops = $invoice->drops->pluck('id')->toArray();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function currencySymbol()
    {
        if (!$this->customer_id) {
            return '$';
        }
        return Customer::find($this->customer_id)?->currency ?? '$';
    }

    #[Computed]
    public function currentInvoice()
    {
        if (!$this->invoiceId) {
            return null;
        }
        return CommissionInvoiceGroup::with(['drops.line.order', 'drops.sizes.size'])->find($this->invoiceId);
    }

    #[Computed]
    public function unshippedDropsCount()
    {
        if (!$this->invoiceId) {
            return 0;
        }
        return CommissionDrop::where('commission_invoice_groups_id', $this->invoiceId)
            ->where('shipped', false)
            ->count();
    }

    #[Computed]
    public function totalQty()
    {
        if (empty($this->selectedDrops)) {
            return 0;
        }
        
        return CommissionDrop::whereIn('id', $this->selectedDrops)
            ->with('sizes')
            ->get()
            ->sum(fn($drop) => $drop->sizes->sum('qty'));
    }

    #[Computed]
    public function totalShippedPcs()
    {
        if (empty($this->selectedDrops)) {
            return 0;
        }
        
        return CommissionDrop::whereIn('id', $this->selectedDrops)
            ->with('sizes')
            ->get()
            ->sum(fn($drop) => $drop->sizes->sum('shipped_qty'));
    }

    #[Computed]
    public function totalValue()
    {
        if (!$this->invoiceId) {
            return 0;
        }
        return $this->currentInvoice?->total_value ?? 0;
    }

    #[Computed]
    public function availableDrops()
    {
        if (!$this->customer_id) {
            return collect();
        }

        return CommissionDrop::query()
            ->with([
                'line.order.customer',
                'line.colourway.style_versions.styles.designs',
                'sizes.size',
            ])
            ->whereHas('line.order', function ($q) {
                $q->where('customers_id', $this->customer_id)
                  ->where('cancelled', false);
            })
            ->where(function ($q) {
                // Unassigned drops OR drops already assigned to this invoice group
                $q->whereNull('commission_invoice_groups_id');
                if ($this->invoiceId) {
                    $q->orWhere('commission_invoice_groups_id', $this->invoiceId);
                }
            })
            ->where('shipped', true) // Only shipped drops can be invoiced
            ->when($this->dropSearch, function ($q) {
                $search = $this->dropSearch;
                $q->where(function ($query) use ($search) {
                    $query->whereHas('line.order', function ($oq) use ($search) {
                        $oq->where('customer_po', 'like', "%{$search}%");
                    })->orWhereHas('line', function ($lq) use ($search) {
                        $lq->where('imported_style_ref', 'like', "%{$search}%")
                           ->orWhere('imported_colour', 'like', "%{$search}%");
                    })->orWhereHas('line.colourway.style_versions.styles.designs', function ($dq) use ($search) {
                        $dq->where('id', 'like', "%{$search}%");
                    });
                });
            })
            ->orderBy('shipped_date', 'desc')
            ->limit(100)
            ->get();
    }

    #[Computed]
    public function attachedDrops()
    {
        if (empty($this->selectedDrops)) {
            return collect();
        }
        
        return CommissionDrop::query()
            ->with([
                'line.order.customer',
                'line.colourway.style_versions.styles.designs',
                'sizes.size',
            ])
            ->whereIn('id', $this->selectedDrops)
            ->orderBy('shipped_date', 'desc')
            ->get();
    }

    public function updatedCustomerId()
    {
        // Reset drops when customer changes (only for new invoices)
        if (!$this->invoiceId) {
            $this->selectedDrops = [];
        }
        $this->isDirty = true;
    }

    public function updated($property)
    {
        $nonDirtyProperties = ['activeTab', 'dropSearch', 'isOpen'];
        
        if (!in_array($property, $nonDirtyProperties)) {
            $this->isDirty = true;
        }
    }

    public function toggleDrop(int $dropId)
    {
        if (in_array($dropId, $this->selectedDrops)) {
            $this->selectedDrops = array_values(array_diff($this->selectedDrops, [$dropId]));
        } else {
            $this->selectedDrops[] = $dropId;
        }
        $this->isDirty = true;
    }

    public function selectAllDrops()
    {
        $this->selectedDrops = $this->availableDrops->pluck('id')->toArray();
        $this->isDirty = true;
    }

    public function deselectAllDrops()
    {
        $this->selectedDrops = [];
        $this->isDirty = true;
    }

    public function close()
    {
        if ($this->isDirty) {
            $this->dispatch('confirm-close');
        } else {
            $this->forceClose();
        }
    }

    public function forceClose()
    {
        $this->isOpen = false;
        $this->resetState();
    }

    public function save()
    {
        $this->validate([
            'customer_id' => 'required|exists:customers,id',
            'reference' => 'nullable|string|max:255',
            'supplierInvoiceRef' => 'nullable|string|max:255',
            'invoiceDate' => 'nullable|date',
        ]);

        // Handle file upload
        $filePath = null;
        $fileName = null;
        
        if ($this->invoiceFile) {
            $fileName = $this->invoiceFile->getClientOriginalName();
            $filePath = $this->invoiceFile->store('commission-invoices', 'local');
        }

        if ($this->invoiceId) {
            // Update existing
            $invoice = CommissionInvoiceGroup::findOrFail($this->invoiceId);
            
            $updateData = [
                'customers_id' => $this->customer_id,
                'reference' => $this->reference,
                'supplier_invoice_ref' => $this->supplierInvoiceRef,
                'invoice_date' => $this->invoiceDate,
                'notes' => $this->notes,
            ];
            
            if ($filePath) {
                $updateData['invoice_file_path'] = $filePath;
                $updateData['invoice_file_name'] = $fileName;
                $updateData['invoice_uploaded_at'] = now();
                $updateData['invoice_uploaded_by'] = auth()->id();
            }
            
            $invoice->update($updateData);
        } else {
            // Create new
            $invoice = CommissionInvoiceGroup::create([
                'customers_id' => $this->customer_id,
                'reference' => $this->reference,
                'supplier_invoice_ref' => $this->supplierInvoiceRef,
                'invoice_date' => $this->invoiceDate,
                'notes' => $this->notes,
                'status' => 'draft',
                'invoice_file_path' => $filePath,
                'invoice_file_name' => $fileName,
                'invoice_uploaded_at' => $filePath ? now() : null,
                'invoice_uploaded_by' => $filePath ? auth()->id() : null,
            ]);
            
            $this->invoiceId = $invoice->id;
            $this->isNewInvoice = false;
        }

        // Update drops assignment
        $this->updateDropsAssignment($invoice);

        // Recalculate totals
        $invoice->recalculateTotals();

        $this->isDirty = false;
        $this->invoiceFile = null;
        $this->existingFileName = $invoice->invoice_file_name;
        
        session()->flash('inv-message', 'Invoice saved successfully!');
        $this->dispatch('invoice-saved');
    }

    protected function updateDropsAssignment(CommissionInvoiceGroup $invoice)
    {
        // Remove assignment from drops no longer selected
        CommissionDrop::where('commission_invoice_groups_id', $invoice->id)
            ->whereNotIn('id', $this->selectedDrops)
            ->update(['commission_invoice_groups_id' => null, 'invoiced' => false]);
        
        // Assign selected drops to this invoice group
        if (!empty($this->selectedDrops)) {
            CommissionDrop::whereIn('id', $this->selectedDrops)
                ->update([
                    'commission_invoice_groups_id' => $invoice->id,
                    'invoiced' => true,
                ]);
        }
    }

    public function markReady()
    {
        if (!$this->invoiceId) {
            session()->flash('inv-error', 'Please save the invoice first.');
            return;
        }

        $invoice = CommissionInvoiceGroup::findOrFail($this->invoiceId);
        
        if ($invoice->drops()->count() === 0) {
            session()->flash('inv-error', 'Please attach at least one drop before sending to finance.');
            return;
        }

        // Check if all drops are shipped
        $unshippedCount = $invoice->drops()->where('shipped', false)->count();
        if ($unshippedCount > 0) {
            session()->flash('inv-error', "Cannot send: {$unshippedCount} drop(s) not marked as shipped.");
            return;
        }

        $invoice->update(['status' => 'ready']);
        $invoice->recalculateTotals();
        $this->status = 'ready';

        // Send email to finance
        $this->sendFinanceEmail($invoice);

        $this->dispatch('invoice-saved');
        $this->forceClose();
    }

    public function markShippedAndReady()
    {
        if (!$this->invoiceId) {
            session()->flash('inv-error', 'Please save the invoice first.');
            return;
        }

        $invoice = CommissionInvoiceGroup::findOrFail($this->invoiceId);
        
        if ($invoice->drops()->count() === 0) {
            session()->flash('inv-error', 'Please attach at least one drop.');
            return;
        }

        // Mark all drops as shipped
        $invoice->drops()->where('shipped', false)->update([
            'shipped' => true,
            'shipped_date' => now(),
        ]);
        
        // Also update shipped_qty for drop sizes if not set
        foreach ($invoice->drops as $drop) {
            $drop->sizes()->whereNull('shipped_qty')->update([
                'shipped_qty' => DB::raw('qty'),
            ]);
        }

        $invoice->update(['status' => 'ready']);
        $invoice->recalculateTotals();
        $this->status = 'ready';

        // Send email to finance
        $this->sendFinanceEmail($invoice);

        $this->dispatch('invoice-saved');
        $this->forceClose();
    }

    /**
     * Send email notification to finance
     */
    protected function sendFinanceEmail(CommissionInvoiceGroup $invoice)
    {
        $financeEmail = config('services.commission.finance_email');
        
        if (!$financeEmail) {
            return;
        }
        
        try {
            $invoice->load('customer');
            Mail::to($financeEmail)->queue(new CommissionInvoiceReady(
                $invoice,
                auth()->user()->name ?? 'System'
            ));
        } catch (\Exception $e) {
            \Log::error('Failed to send commission invoice email: ' . $e->getMessage());
        }
    }

    public function downloadFile()
    {
        if (!$this->invoiceId) return;
        
        $invoice = CommissionInvoiceGroup::findOrFail($this->invoiceId);
        
        if (!$invoice->invoice_file_path) {
            session()->flash('inv-error', 'No file attached to this invoice.');
            return;
        }
        
        if (!Storage::disk('local')->exists($invoice->invoice_file_path)) {
            session()->flash('inv-error', 'File not found on disk.');
            return;
        }
        
        return Storage::disk('local')->download(
            $invoice->invoice_file_path,
            $invoice->invoice_file_name ?? basename($invoice->invoice_file_path)
        );
    }

    public function render()
    {
        return view('livewire.commission.invoice-edit');
    }
}

