<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Models\CommissionInvoiceGroup;
use App\Models\CommissionDrop;
use App\Models\Customer;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Url;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class ImportInvoice extends Component
{
    use WithFileUploads;

    #[Url]
    public ?int $edit = null;
    
    #[Url]
    public ?string $mode = null;

    public ?int $selectedCustomer = null;
    public $invoiceFile = null;
    public ?string $reference = '';
    public ?string $supplierInvoiceRef = '';
    public ?string $invoiceDate = null;
    public ?string $notes = '';
    
    public ?int $invoiceGroupId = null;
    public array $selectedDrops = [];
    
    public string $dropSearch = '';

    public function mount()
    {
        if ($this->edit) {
            $this->loadInvoiceGroup($this->edit);
        }
        
        // Set default invoice date for new invoices
        if (!$this->edit && !$this->invoiceDate) {
            $this->invoiceDate = now()->format('Y-m-d');
        }
    }
    
    public function isCreateMode(): bool
    {
        return $this->mode === 'create' || (!$this->edit && !$this->invoiceGroupId);
    }

    protected function loadInvoiceGroup(int $id)
    {
        $invoiceGroup = CommissionInvoiceGroup::with('drops')->findOrFail($id);
        
        $this->invoiceGroupId = $invoiceGroup->id;
        $this->selectedCustomer = $invoiceGroup->customers_id;
        $this->reference = $invoiceGroup->reference;
        $this->supplierInvoiceRef = $invoiceGroup->supplier_invoice_ref;
        $this->invoiceDate = $invoiceGroup->invoice_date?->format('Y-m-d');
        $this->notes = $invoiceGroup->notes;
        
        // Pre-select attached drops
        $this->selectedDrops = $invoiceGroup->drops->pluck('id')->toArray();
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function currentInvoiceGroup()
    {
        if (!$this->invoiceGroupId) {
            return null;
        }
        return CommissionInvoiceGroup::with(['drops.line.order', 'drops.sizes'])->find($this->invoiceGroupId);
    }

    #[Computed]
    public function unshippedDropsCount()
    {
        if (!$this->invoiceGroupId) {
            return 0;
        }
        return CommissionDrop::where('commission_invoice_groups_id', $this->invoiceGroupId)
            ->where('shipped', false)
            ->count();
    }

    #[Computed]
    public function availableDrops()
    {
        if (!$this->selectedCustomer) {
            return collect();
        }

        return CommissionDrop::query()
            ->with([
                'line.order.customer',
                'line.colourway.style_versions.styles',
                'sizes.size',
            ])
            ->whereHas('line.order', function ($q) {
                $q->where('customers_id', $this->selectedCustomer)
                  ->where('cancelled', false);
            })
            ->where(function ($q) {
                // Unassigned drops OR drops already assigned to this invoice group
                $q->whereNull('commission_invoice_groups_id');
                if ($this->invoiceGroupId) {
                    $q->orWhere('commission_invoice_groups_id', $this->invoiceGroupId);
                }
            })
            ->where('shipped', true) // Only shipped drops can be invoiced
            ->when($this->dropSearch, function ($q) {
                $q->whereHas('line.order', function ($oq) {
                    $oq->where('customer_po', 'like', "%{$this->dropSearch}%");
                })->orWhereHas('line', function ($lq) {
                    $lq->where('imported_style_ref', 'like', "%{$this->dropSearch}%")
                       ->orWhere('imported_colour', 'like', "%{$this->dropSearch}%");
                });
            })
            ->orderBy('shipped_date', 'desc')
            ->limit(100)
            ->get();
    }

    public function updatedSelectedCustomer()
    {
        // Reset drops when customer changes
        if (!$this->invoiceGroupId) {
            $this->selectedDrops = [];
        }
    }

    public function updatedInvoiceFile()
    {
        $this->validate([
            'invoiceFile' => 'required|file|mimes:pdf,jpg,jpeg,png,gif|max:10240', // 10MB max
        ]);
    }

    public function toggleDrop(int $dropId)
    {
        if (in_array($dropId, $this->selectedDrops)) {
            $this->selectedDrops = array_values(array_diff($this->selectedDrops, [$dropId]));
        } else {
            $this->selectedDrops[] = $dropId;
        }
    }

    public function selectAllDrops()
    {
        $this->selectedDrops = $this->availableDrops->pluck('id')->toArray();
    }

    public function deselectAllDrops()
    {
        $this->selectedDrops = [];
    }

    /**
     * Create or update the invoice group
     */
    public function saveInvoiceGroup()
    {
        $this->validate([
            'selectedCustomer' => 'required|exists:customers,id',
            'reference' => 'nullable|string|max:255',
            'supplierInvoiceRef' => 'nullable|string|max:255',
            'invoiceDate' => 'nullable|date',
        ]);

        // Handle file upload
        $filePath = null;
        $fileName = null;
        
        if ($this->invoiceFile) {
            $fileName = $this->invoiceFile->getClientOriginalName();
            $filePath = $this->invoiceFile->store('commission-invoices', 'local');
        }

        if ($this->invoiceGroupId) {
            // Update existing
            $invoiceGroup = CommissionInvoiceGroup::findOrFail($this->invoiceGroupId);
            
            $updateData = [
                'customers_id' => $this->selectedCustomer,
                'reference' => $this->reference,
                'supplier_invoice_ref' => $this->supplierInvoiceRef,
                'invoice_date' => $this->invoiceDate,
                'notes' => $this->notes,
            ];
            
            if ($filePath) {
                $updateData['invoice_file_path'] = $filePath;
                $updateData['invoice_file_name'] = $fileName;
                $updateData['invoice_uploaded_at'] = now();
                $updateData['invoice_uploaded_by'] = auth()->id();
            }
            
            $invoiceGroup->update($updateData);
        } else {
            // Create new
            $invoiceGroup = CommissionInvoiceGroup::create([
                'customers_id' => $this->selectedCustomer,
                'reference' => $this->reference,
                'supplier_invoice_ref' => $this->supplierInvoiceRef,
                'invoice_date' => $this->invoiceDate,
                'notes' => $this->notes,
                'status' => 'draft',
                'invoice_file_path' => $filePath,
                'invoice_file_name' => $fileName,
                'invoice_uploaded_at' => $filePath ? now() : null,
                'invoice_uploaded_by' => $filePath ? auth()->id() : null,
            ]);
            
            $this->invoiceGroupId = $invoiceGroup->id;
        }

        // Update drops assignment
        $this->updateDropsAssignment($invoiceGroup);

        // Recalculate totals
        $invoiceGroup->recalculateTotals();

        $this->dispatch('notify', type: 'success', message: 'Invoice group saved successfully.');
        $this->dispatch('invoice-saved');
        
        // Clear the file input
        $this->invoiceFile = null;
    }

    protected function updateDropsAssignment(CommissionInvoiceGroup $invoiceGroup)
    {
        // Remove assignment from drops no longer selected
        CommissionDrop::where('commission_invoice_groups_id', $invoiceGroup->id)
            ->whereNotIn('id', $this->selectedDrops)
            ->update(['commission_invoice_groups_id' => null, 'invoiced' => false]);
        
        // Assign selected drops to this invoice group
        if (!empty($this->selectedDrops)) {
            CommissionDrop::whereIn('id', $this->selectedDrops)
                ->update([
                    'commission_invoice_groups_id' => $invoiceGroup->id,
                    'invoiced' => true,
                ]);
        }
    }

    /**
     * Mark invoice as ready (requires all drops to be shipped)
     */
    public function markReady()
    {
        if (!$this->invoiceGroupId) {
            $this->dispatch('notify', type: 'error', message: 'Please save the invoice first.');
            return;
        }

        $invoiceGroup = CommissionInvoiceGroup::findOrFail($this->invoiceGroupId);
        
        if ($invoiceGroup->drops()->count() === 0) {
            $this->dispatch('notify', type: 'error', message: 'Please attach at least one drop before sending to finance.');
            return;
        }

        // Check if all drops are shipped
        $unshippedCount = $invoiceGroup->drops()->where('shipped', false)->count();
        if ($unshippedCount > 0) {
            $this->dispatch('notify', type: 'error', message: "Cannot send: {$unshippedCount} drop(s) not marked as shipped. Use 'Mark Shipped & Send' instead.");
            return;
        }

        $invoiceGroup->update(['status' => 'ready']);
        $invoiceGroup->recalculateTotals();

        $this->dispatch('notify', type: 'success', message: 'Invoice sent to finance.');
        $this->dispatch('invoice-saved');
        
        return redirect()->route('commission.invoices');
    }

    /**
     * Mark all drops as shipped and send to finance
     */
    public function markShippedAndReady()
    {
        if (!$this->invoiceGroupId) {
            $this->dispatch('notify', type: 'error', message: 'Please save the invoice first.');
            return;
        }

        $invoiceGroup = CommissionInvoiceGroup::findOrFail($this->invoiceGroupId);
        
        if ($invoiceGroup->drops()->count() === 0) {
            $this->dispatch('notify', type: 'error', message: 'Please attach at least one drop.');
            return;
        }

        // Mark all drops as shipped
        $invoiceGroup->drops()->where('shipped', false)->update([
            'shipped' => true,
            'shipped_date' => now(),
        ]);
        
        // Also update shipped_qty for drop sizes if not set
        foreach ($invoiceGroup->drops as $drop) {
            $drop->sizes()->whereNull('shipped_qty')->update([
                'shipped_qty' => DB::raw('qty'),
            ]);
        }

        $invoiceGroup->update(['status' => 'ready']);
        $invoiceGroup->recalculateTotals();

        $this->dispatch('notify', type: 'success', message: 'Drops marked as shipped and invoice sent to finance.');
        $this->dispatch('invoice-saved');
        
        return redirect()->route('commission.invoices');
    }

    /**
     * Cancel and return to list
     */
    public function cancel()
    {
        return redirect()->route('commission.invoices');
    }

    public function render()
    {
        return view('livewire.commission.import-invoice')->layout('layouts.commission');
    }
}

