<?php

namespace App\Http\Livewire\Commission;

use App\Models\CommissionOrder;
use App\Models\CommissionOrderLine;
use App\Models\CommissionDrop;
use App\Models\Customer;
use App\Models\Seasons;
use Livewire\Component;
use Livewire\Attributes\Computed;
use Livewire\Attributes\Url;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class CommissionDashboard extends Component
{
    public $selectedPeriod = '30';
    
    #[Url(as: 'season')]
    public $selectedSeason = null;

    #[Computed]
    public function seasons()
    {
        return Seasons::orderByDesc('id')->get();
    }

    #[Computed]
    public function selectedSeasonModel()
    {
        if ($this->selectedSeason) {
            return Seasons::find($this->selectedSeason);
        }
        return null;
    }

    #[Computed]
    public function orderStats()
    {
        $baseQuery = CommissionOrder::query()->whereNull('deleted_at');
        
        // Apply season filter
        if ($this->selectedSeason) {
            $baseQuery->where('seasons_id', $this->selectedSeason);
        }
        
        $activeQuery = $baseQuery->clone()->where('cancelled', false);
        
        // Get total units ordered (from confirmed orders)
        $unitsQuery = DB::table('commission_order_line_quantities as colq')
            ->join('commission_order_lines as col', 'col.id', '=', 'colq.commission_order_lines_id')
            ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
            ->whereNull('co.deleted_at')
            ->whereNull('col.deleted_at')
            ->where('co.cancelled', false)
            ->where('co.status', 'confirmed');
        
        if ($this->selectedSeason) {
            $unitsQuery->where('co.seasons_id', $this->selectedSeason);
        }
        
        $totalUnitsOrdered = $unitsQuery->sum('colq.qty');
        
        return [
            'total' => $baseQuery->clone()->count(),
            'active' => $activeQuery->clone()->count(),
            'skeleton' => $activeQuery->clone()->where('status', 'skeleton')->count(),
            'pending_pd' => $activeQuery->clone()->where('status', 'pending_pd')->count(),
            'pending_merch' => $activeQuery->clone()->where('status', 'pending_merch')->count(),
            'confirmed' => $activeQuery->clone()->where('status', 'confirmed')->count(),
            'cancelled' => $baseQuery->clone()->where('cancelled', true)->count(),
            'total_units_ordered' => (int) $totalUnitsOrdered,
        ];
    }

    #[Computed]
    public function shipmentStats()
    {
        // Base query for drops with season filter via order relationship
        $baseQuery = CommissionDrop::query()
            ->whereNull('commission_drops.deleted_at');
        
        if ($this->selectedSeason) {
            $baseQuery->whereHas('line.order', function ($q) {
                $q->where('seasons_id', $this->selectedSeason);
            });
        }
        
        $totalDrops = $baseQuery->clone()->count();
        $shippedDrops = $baseQuery->clone()->where('shipped', true)->count();
        
        // Invoiced drops = shipped AND invoiced (we only care about invoiced once shipped)
        $shippedAndInvoiced = $baseQuery->clone()
            ->where('shipped', true)
            ->where('invoiced', true)
            ->count();
        
        // Get total quantities - use COALESCE for shipped_qty since some are null
        $totalQtyQuery = DB::table('commission_drop_sizes as cds')
            ->join('commission_drops as cd', 'cd.id', '=', 'cds.commission_drops_id')
            ->join('commission_order_lines as col', 'col.id', '=', 'cd.commission_order_lines_id')
            ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
            ->whereNull('cd.deleted_at');
        
        if ($this->selectedSeason) {
            $totalQtyQuery->where('co.seasons_id', $this->selectedSeason);
        }
        
        $totalQty = $totalQtyQuery->sum('cds.qty');
            
        $shippedQtyQuery = DB::table('commission_drop_sizes as cds')
            ->join('commission_drops as cd', 'cd.id', '=', 'cds.commission_drops_id')
            ->join('commission_order_lines as col', 'col.id', '=', 'cd.commission_order_lines_id')
            ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
            ->whereNull('cd.deleted_at')
            ->where('cd.shipped', true);
        
        if ($this->selectedSeason) {
            $shippedQtyQuery->where('co.seasons_id', $this->selectedSeason);
        }
        
        $shippedQty = $shippedQtyQuery->sum(DB::raw('COALESCE(cds.shipped_qty, cds.qty)'));
        
        // Get total ordered qty (for fulfillment rate)
        $totalOrderedQtyQuery = DB::table('commission_order_line_quantities as colq')
            ->join('commission_order_lines as col', 'col.id', '=', 'colq.commission_order_lines_id')
            ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
            ->whereNull('co.deleted_at')
            ->whereNull('col.deleted_at')
            ->where('co.cancelled', false)
            ->where('co.status', 'confirmed');
        
        if ($this->selectedSeason) {
            $totalOrderedQtyQuery->where('co.seasons_id', $this->selectedSeason);
        }
        
        $totalOrderedQty = $totalOrderedQtyQuery->sum('colq.qty');
        
        $fulfillmentRate = $totalOrderedQty > 0 
            ? round(($shippedQty / $totalOrderedQty) * 100, 1) 
            : 0;
        
        return [
            'total_drops' => $totalDrops,
            'shipped_drops' => $shippedDrops,
            'pending_drops' => $totalDrops - $shippedDrops,
            'invoiced_drops' => $shippedAndInvoiced,
            'awaiting_invoice' => $shippedDrops - $shippedAndInvoiced,
            'total_qty' => (int) $totalQty,
            'shipped_qty' => (int) $shippedQty,
            'total_ordered_qty' => (int) $totalOrderedQty,
            'remaining_qty' => (int) ($totalOrderedQty - $shippedQty),
            'fulfillment_rate' => $fulfillmentRate,
        ];
    }

    #[Computed]
    public function pipelineStats()
    {
        $today = now()->toDateString();
        $seasonId = $this->selectedSeason;
        
        // Base query builder for drops with optional season filter
        $baseDropQuery = function () use ($seasonId) {
            $q = CommissionDrop::whereNull('commission_drops.deleted_at');
            if ($seasonId) {
                $q->whereHas('line.order', function ($sq) use ($seasonId) {
                    $sq->where('seasons_id', $seasonId);
                });
            }
            return $q;
        };
        
        // Waiting to ship (unshipped with future or no exfty date)
        $waitingToShip = $baseDropQuery()
            ->where('shipped', false)
            ->where(function ($q) use ($today) {
                $q->whereNull('exfty')
                  ->orWhere('exfty', '>=', $today);
            })
            ->count();
        
        // Past ship date (overdue - unshipped and exfty has passed)
        $pastShipDate = $baseDropQuery()
            ->where('shipped', false)
            ->whereNotNull('exfty')
            ->where('exfty', '<', $today)
            ->count();
        
        // Shipped but not invoiced yet (using the invoiced boolean flag)
        $shippedNoInvoice = $baseDropQuery()
            ->where('shipped', true)
            ->where('invoiced', false)
            ->count();
        
        // In invoice group but not sent to Zoho
        $inInvoiceNotSentQuery = DB::table('commission_drops as cd')
            ->join('commission_invoice_groups as cig', 'cig.id', '=', 'cd.commission_invoice_groups_id')
            ->join('commission_order_lines as col', 'col.id', '=', 'cd.commission_order_lines_id')
            ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
            ->whereNull('cd.deleted_at')
            ->whereNull('cig.deleted_at')
            ->whereNull('cig.sent_to_zoho_at');
        
        if ($seasonId) {
            $inInvoiceNotSentQuery->where('co.seasons_id', $seasonId);
        }
        
        $inInvoiceNotSent = $inInvoiceNotSentQuery->count();
        
        return [
            'waiting_to_ship' => $waitingToShip,
            'past_ship_date' => $pastShipDate,
            'shipped_no_invoice' => $shippedNoInvoice,
            'in_invoice_not_sent' => $inInvoiceNotSent,
        ];
    }

    #[Computed]
    public function pendingActions()
    {
        $actions = [];
        $seasonId = $this->selectedSeason;
        $seasonParam = $seasonId ? ['season' => $seasonId] : [];
        
        // Orders waiting for PD review
        $pdQuery = CommissionOrder::where('status', 'pending_pd')
            ->where('cancelled', false)
            ->whereNull('deleted_at');
        if ($seasonId) {
            $pdQuery->where('seasons_id', $seasonId);
        }
        $pdPending = $pdQuery->count();
        
        if ($pdPending > 0) {
            $actions[] = [
                'type' => 'warning',
                'icon' => 'fa-pencil-ruler',
                'title' => "{$pdPending} orders awaiting PD review",
                'link' => route('commission.orders', array_merge(['status' => ['pending_pd']], $seasonParam)),
            ];
        }
        
        // Orders waiting for Merch approval
        $merchQuery = CommissionOrder::where('status', 'pending_merch')
            ->where('cancelled', false)
            ->whereNull('deleted_at');
        if ($seasonId) {
            $merchQuery->where('seasons_id', $seasonId);
        }
        $merchPending = $merchQuery->count();
        
        if ($merchPending > 0) {
            $actions[] = [
                'type' => 'info',
                'icon' => 'fa-check-circle',
                'title' => "{$merchPending} orders awaiting Merch approval",
                'link' => route('commission.orders', array_merge(['status' => ['pending_merch']], $seasonParam)),
            ];
        }
        
        // Unmatched lines
        $unmatchedQuery = CommissionOrderLine::where('match_status', 'unmatched')
            ->whereNull('deleted_at');
        if ($seasonId) {
            $unmatchedQuery->whereHas('order', function ($q) use ($seasonId) {
                $q->where('seasons_id', $seasonId);
            });
        }
        $unmatchedLines = $unmatchedQuery->count();
        
        if ($unmatchedLines > 0) {
            $actions[] = [
                'type' => 'danger',
                'icon' => 'fa-unlink',
                'title' => "{$unmatchedLines} order lines need style matching",
                'link' => route('commission.orders', array_merge(['status' => ['skeleton', 'pending_pd']], $seasonParam)),
            ];
        }
        
        // Drops ready but not shipped (overdue) - this is an alert condition
        $overdueQuery = CommissionDrop::where('shipped', false)
            ->whereNotNull('exfty')
            ->where('exfty', '<', now()->toDateString())
            ->whereNull('deleted_at');
        if ($seasonId) {
            $overdueQuery->whereHas('line.order', function ($q) use ($seasonId) {
                $q->where('seasons_id', $seasonId);
            });
        }
        $readyToShip = $overdueQuery->count();
        
        if ($readyToShip > 0) {
            $actions[] = [
                'type' => 'danger',
                'icon' => 'fa-truck',
                'title' => "{$readyToShip} drops past ex-factory date",
                'link' => route('commission.shipment-schedule', array_merge(['view' => 'overdue'], $seasonParam)),
            ];
        }
        
        // Invoices not sent to Zoho (draft or ready status) - not season-filtered as invoice groups span seasons
        $notSentToZoho = DB::table('commission_invoice_groups')
            ->whereNull('deleted_at')
            ->whereIn('status', ['draft', 'ready'])
            ->count();
        if ($notSentToZoho > 0) {
            $actions[] = [
                'type' => 'primary',
                'icon' => 'fa-file-invoice-dollar',
                'title' => "{$notSentToZoho} invoices not sent to Zoho",
                'link' => route('commission.invoices', ['status' => 'notSentToZoho']),
            ];
        }
        
        return $actions;
    }

    #[Computed]
    public function upcomingShipments()
    {
        $query = CommissionDrop::with(['line.order.customer', 'line.colourway.style_versions.styles'])
            ->whereNull('deleted_at')
            ->where('shipped', false)
            ->whereNotNull('exfty')
            ->where('exfty', '>=', now()->subDays(7));
        
        if ($this->selectedSeason) {
            $query->whereHas('line.order', function ($q) {
                $q->where('seasons_id', $this->selectedSeason);
            });
        }
        
        return $query->orderBy('exfty')
            ->limit(8)
            ->get();
    }

    #[Computed]
    public function topCustomers()
    {
        $query = DB::table('commission_orders as co')
            ->select([
                'c.id',
                'c.name',
                DB::raw('COUNT(DISTINCT co.id) as order_count'),
                DB::raw('SUM(colq.qty) as total_qty'),
                DB::raw('SUM(colq.qty * COALESCE(colq.price, 0)) as total_value'),
            ])
            ->join('customers as c', 'c.id', '=', 'co.customers_id')
            ->join('commission_order_lines as col', 'col.commission_orders_id', '=', 'co.id')
            ->join('commission_order_line_quantities as colq', 'colq.commission_order_lines_id', '=', 'col.id')
            ->whereNull('co.deleted_at')
            ->whereNull('col.deleted_at')
            ->where('co.status', 'confirmed');
        
        if ($this->selectedSeason) {
            $query->where('co.seasons_id', $this->selectedSeason);
        }
        
        return $query->groupBy('c.id', 'c.name')
            ->orderByDesc('total_value')
            ->limit(5)
            ->get();
    }

    #[Computed]
    public function ordersByMonth()
    {
        $data = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $start = $date->copy()->startOfMonth()->toDateString();
            $end = $date->copy()->endOfMonth()->toDateString();
            
            $query = DB::table('commission_order_line_quantities as colq')
                ->join('commission_order_lines as col', 'col.id', '=', 'colq.commission_order_lines_id')
                ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
                ->whereNull('co.deleted_at')
                ->whereNull('col.deleted_at')
                ->where('co.status', 'confirmed')
                ->whereBetween('co.order_date', [$start, $end]);
            
            if ($this->selectedSeason) {
                $query->where('co.seasons_id', $this->selectedSeason);
            }
            
            $qty = $query->sum('colq.qty');
                
            $data[] = [
                'label' => $date->format('M'),
                'qty' => (int) $qty,
            ];
        }
        
        return $data;
    }

    #[Computed]
    public function shipmentsByMonth()
    {
        $data = [];
        for ($i = 5; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $start = $date->copy()->startOfMonth()->toDateString();
            $end = $date->copy()->endOfMonth()->toDateString();
            
            // Use exfty date for shipped items since shipped_date may not be set
            $query = DB::table('commission_drop_sizes as cds')
                ->join('commission_drops as cd', 'cd.id', '=', 'cds.commission_drops_id')
                ->join('commission_order_lines as col', 'col.id', '=', 'cd.commission_order_lines_id')
                ->join('commission_orders as co', 'co.id', '=', 'col.commission_orders_id')
                ->whereNull('cd.deleted_at')
                ->where('cd.shipped', true)
                ->whereBetween('cd.exfty', [$start, $end]);
            
            if ($this->selectedSeason) {
                $query->where('co.seasons_id', $this->selectedSeason);
            }
            
            $qty = $query->sum(DB::raw('COALESCE(cds.shipped_qty, cds.qty)'));
                
            $data[] = [
                'label' => $date->format('M'),
                'qty' => (int) $qty,
            ];
        }
        
        return $data;
    }

    public function render()
    {
        return view('livewire.commission.commission-dashboard')
            ->layout('layouts.commission');
    }
}
