<?php

namespace App\Http\Livewire\Commission;

use App\Models\CommissionDrop;
use App\Models\Customer;
use App\Models\Seasons;
use Livewire\Component;
use Livewire\Attributes\Url;
use Livewire\Attributes\Computed;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Gate;

class Calendar extends Component
{
    // ==================
    // Date Navigation
    // ==================

    #[Url]
    public int $year;

    #[Url]
    public int $month;

    // ==================
    // Filters
    // ==================

    #[Url]
    public array $customers = [];

    #[Url]
    public array $seasons = [];

    #[Url]
    public string $shipped = ''; // 'shipped', 'unshipped', ''

    // ==================
    // Modal State
    // ==================

    public bool $showDayModal = false;
    public ?string $selectedDate = null;

    // ==================
    // Lifecycle
    // ==================

    public function mount()
    {
        if (!isset($this->year) || !$this->year) {
            $this->year = now()->year;
        }
        if (!isset($this->month) || !$this->month) {
            $this->month = now()->month;
        }
    }

    // ==================
    // Navigation
    // ==================

    public function previousMonth()
    {
        $date = Carbon::create($this->year, $this->month, 1)->subMonth();
        $this->year = $date->year;
        $this->month = $date->month;
    }

    public function nextMonth()
    {
        $date = Carbon::create($this->year, $this->month, 1)->addMonth();
        $this->year = $date->year;
        $this->month = $date->month;
    }

    public function goToToday()
    {
        $this->year = now()->year;
        $this->month = now()->month;
    }

    public function goToMonth(int $year, int $month)
    {
        $this->year = $year;
        $this->month = $month;
    }

    // ==================
    // Modal Actions
    // ==================

    public function openDayModal(string $date)
    {
        $this->selectedDate = $date;
        $this->showDayModal = true;
    }

    public function closeDayModal()
    {
        $this->showDayModal = false;
        $this->selectedDate = null;
    }

    // ==================
    // Filter Actions
    // ==================

    public function toggleCustomer(int $customerId)
    {
        if (in_array($customerId, $this->customers)) {
            $this->customers = array_values(array_diff($this->customers, [$customerId]));
        } else {
            $this->customers[] = $customerId;
        }
    }

    public function toggleSeason(int $seasonId)
    {
        if (in_array($seasonId, $this->seasons)) {
            $this->seasons = array_values(array_diff($this->seasons, [$seasonId]));
        } else {
            $this->seasons[] = $seasonId;
        }
    }

    public function clearFilters()
    {
        $this->reset(['customers', 'seasons', 'shipped']);
    }

    // ==================
    // Computed Properties
    // ==================

    #[Computed]
    public function currentMonth()
    {
        return Carbon::create($this->year, $this->month, 1);
    }

    #[Computed]
    public function calendarDays()
    {
        $start = $this->currentMonth->copy()->startOfMonth();
        $end = $this->currentMonth->copy()->endOfMonth();

        // Adjust to include days from previous month to fill the first week
        $startOfCalendar = $start->copy()->startOfWeek(Carbon::MONDAY);
        // Adjust to include days from next month to fill the last week
        $endOfCalendar = $end->copy()->endOfWeek(Carbon::SUNDAY);

        $days = [];
        $current = $startOfCalendar->copy();

        while ($current <= $endOfCalendar) {
            $days[] = [
                'date' => $current->format('Y-m-d'),
                'day' => $current->day,
                'isCurrentMonth' => $current->month === $this->month,
                'isToday' => $current->isToday(),
                'isWeekend' => $current->isWeekend(),
                'carbon' => $current->copy(),
            ];
            $current->addDay();
        }

        return $days;
    }

    #[Computed]
    public function drops()
    {
        $startOfCalendar = $this->currentMonth->copy()->startOfMonth()->startOfWeek(Carbon::MONDAY);
        $endOfCalendar = $this->currentMonth->copy()->endOfMonth()->endOfWeek(Carbon::SUNDAY);

        $query = CommissionDrop::query()
            ->with([
                'line.order.customer:id,name',
                'line.order.season:id,description',
                'line.colourway:id,name,img_thumb,style_versions_id',
                'line.colourway.style_versions:id,styles_id',
                'line.colourway.style_versions.styles:id,designs_id,customer_ref',
                'line.colourway.style_versions.styles.designs:id,description',
                'sizes.size:id,name,order',
            ])
            ->whereNotNull('exfty')
            ->whereBetween('exfty', [$startOfCalendar->format('Y-m-d'), $endOfCalendar->format('Y-m-d')])
            ->whereHas('line.order', function ($q) {
                $q->where('cancelled', false)
                  ->where('status', 'confirmed');
            })
            ->whereHas('line', function ($q) {
                $q->where('cancelled', false);
            });

        // Customer filter
        if (!empty($this->customers)) {
            $query->whereHas('line.order', function ($q) {
                $q->whereIn('customers_id', $this->customers);
            });
        }

        // Season filter
        if (!empty($this->seasons)) {
            $query->whereHas('line.order', function ($q) {
                $q->whereIn('seasons_id', $this->seasons);
            });
        }

        // Shipped filter
        if ($this->shipped === 'shipped') {
            $query->where('shipped', true);
        } elseif ($this->shipped === 'unshipped') {
            $query->where('shipped', false);
        }

        return $query->orderBy('exfty')->get();
    }

    #[Computed]
    public function dropsByDate()
    {
        return $this->drops->groupBy(function ($drop) {
            return $drop->exfty->format('Y-m-d');
        });
    }

    #[Computed]
    public function selectedDateDrops()
    {
        if (!$this->selectedDate) {
            return collect();
        }
        return $this->dropsByDate->get($this->selectedDate, collect());
    }

    #[Computed]
    public function monthStats()
    {
        $startOfMonth = $this->currentMonth->copy()->startOfMonth();
        $endOfMonth = $this->currentMonth->copy()->endOfMonth();

        $monthDrops = $this->drops->filter(function ($drop) use ($startOfMonth, $endOfMonth) {
            return $drop->exfty >= $startOfMonth && $drop->exfty <= $endOfMonth;
        });

        return [
            'total' => $monthDrops->count(),
            'shipped' => $monthDrops->where('shipped', true)->count(),
            'unshipped' => $monthDrops->where('shipped', false)->count(),
            'totalQty' => $monthDrops->sum(fn($d) => $d->sizes->sum('qty')),
        ];
    }

    #[Computed]
    public function customerList()
    {
        return Customer::whereHas('commissionOrders', function ($q) {
            $q->where('cancelled', false)->where('status', 'confirmed');
        })
        ->orderBy('name')
        ->get(['id', 'name']);
    }

    #[Computed]
    public function seasonList()
    {
        return Seasons::whereHas('commissionOrders', function ($q) {
            $q->where('cancelled', false)->where('status', 'confirmed');
        })
        ->orderBy('created_at', 'desc')
        ->get(['id', 'description']);
    }

    #[Computed]
    public function activeFiltersCount()
    {
        $count = 0;
        if (!empty($this->customers)) $count++;
        if (!empty($this->seasons)) $count++;
        if ($this->shipped) $count++;
        return $count;
    }

    // ==================
    // Helpers
    // ==================

    public function getDropStatusClass(CommissionDrop $drop): string
    {
        if ($drop->shipped) {
            return 'shipped';
        }
        
        if ($drop->exfty && Carbon::parse($drop->exfty)->lt(now()->startOfDay())) {
            return 'overdue';
        }
        
        return 'pending';
    }

    // ==================
    // Render
    // ==================

    public function render()
    {
        Gate::authorize('shipment:read');

        return view('livewire.commission.calendar')
            ->layout('layouts.commission');
    }
}

