<?php

namespace App\Http\Livewire\Commission;

use Livewire\Component;
use Livewire\WithFileUploads;
use Livewire\Attributes\Computed;
use App\Models\CommissionImportBatch;
use App\Models\CommissionImportBatchFile;
use App\Models\Customer;
use App\Models\Seasons;
use App\Jobs\ProcessCommissionBatchImport;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class BatchImport extends Component
{
    use WithFileUploads;

    public string $batchName = '';
    public ?int $customerId = null;
    public ?int $seasonId = null;
    public array $files = [];
    
    public bool $isUploading = false;
    public ?int $activeBatchId = null;

    // Detail modal
    public bool $showDetailModal = false;
    public ?int $detailBatchId = null;
    
    // Admin toggle to show all batches
    public bool $showAllBatches = false;

    protected $rules = [
        'batchName' => 'required|string|max:255',
        'customerId' => 'required|exists:customers,id',
        'seasonId' => 'required|exists:seasons,id',
        'files' => 'required|array|min:1',
        'files.*' => 'file|mimes:pdf,xlsx,xls,csv|max:20480',
    ];

    public function mount()
    {
        // Default to most recent season
        $this->seasonId = Seasons::orderBy('created_at', 'desc')->first()?->id;
    }

    #[Computed]
    public function customers()
    {
        return Customer::where('type', 'commission')
            ->orderBy('name')
            ->get();
    }

    #[Computed]
    public function seasons()
    {
        return Seasons::orderBy('created_at', 'desc')->get();
    }


    #[Computed]
    public function selectedCustomer()
    {
        return $this->customerId ? Customer::find($this->customerId) : null;
    }

    #[Computed]
    public function isAdmin()
    {
        $user = auth()->user();
        $rtTeam = \App\Models\Team::find(1);
        return $rtTeam && $user->hasTeamRole($rtTeam, 'admin');
    }

    #[Computed]
    public function recentBatches()
    {
        return CommissionImportBatch::with(['customer', 'user', 'commissionOrder', 'commissionOrders'])
            ->withCount('commissionOrders')
            ->when(!$this->showAllBatches || !$this->isAdmin, function ($query) {
                $query->where('user_id', auth()->id());
            })
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();
    }

    public function updatedFiles()
    {
        // Auto-generate batch name if empty
        if (empty($this->batchName) && count($this->files) > 0) {
            $firstFile = $this->files[0];
            $this->batchName = pathinfo($firstFile->getClientOriginalName(), PATHINFO_FILENAME);
        }
    }

    public function removeFile($index)
    {
        unset($this->files[$index]);
        $this->files = array_values($this->files);
    }

    public function startImport()
    {
        $this->validate();

        $this->isUploading = true;

        try {
            // Create the batch
            $batch = CommissionImportBatch::create([
                'batch_name' => $this->batchName,
                'customer_name' => $this->selectedCustomer->name,
                'customer_id' => $this->customerId,
                'season_id' => $this->seasonId,
                'user_id' => auth()->id(),
                'status' => 'pending',
                'workflow_status' => 'pending',
                'total_files' => count($this->files),
            ]);

            // Store files and create records
            $uploadDir = "commission-imports/{$batch->id}";
            Storage::disk('local')->makeDirectory($uploadDir);
            
            foreach ($this->files as $file) {
                $originalName = $file->getClientOriginalName();
                $storedName = Str::uuid() . '.' . $file->getClientOriginalExtension();
                $path = $file->storeAs($uploadDir, $storedName, 'local');

                CommissionImportBatchFile::create([
                    'batch_id' => $batch->id,
                    'original_filename' => $originalName,
                    'stored_filename' => $storedName,
                    'file_path' => $path,
                    'status' => 'pending',
                ]);
            }

            // Dispatch background job
            ProcessCommissionBatchImport::dispatch($batch);

            $this->activeBatchId = $batch->id;
            
            // Reset form
            $this->reset(['batchName', 'files']);
            
            $this->dispatch('notify', type: 'success', message: 'Import started! You will be emailed when complete.');

        } catch (\Exception $e) {
            $this->dispatch('notify', type: 'error', message: 'Import failed: ' . $e->getMessage());
        } finally {
            $this->isUploading = false;
        }
    }

    public function viewBatch(int $batchId)
    {
        return redirect()->route('commission.orders', ['batch' => $batchId]);
    }

    public function openDetails(int $batchId)
    {
        $this->detailBatchId = $batchId;
        $this->showDetailModal = true;
    }

    public function closeDetails()
    {
        $this->showDetailModal = false;
        $this->detailBatchId = null;
    }

    #[Computed]
    public function detailBatch()
    {
        if (!$this->detailBatchId) {
            return null;
        }

        return CommissionImportBatch::with(['customer', 'season', 'user', 'files', 'commissionOrder.lines', 'commissionOrders.lines'])
            ->find($this->detailBatchId);
    }

    public function retryBatch(int $batchId)
    {
        $batch = CommissionImportBatch::findOrFail($batchId);

        if (!in_array($batch->status, ['failed', 'completed', 'completed_with_warnings'])) {
            $this->dispatch('notify', type: 'error', message: 'Can only retry failed or completed batches.');
            return;
        }

        // Check if any orders have been approved (confirmed)
        $hasApprovedOrders = $batch->commissionOrders()
            ->where('status', 'confirmed')
            ->exists();

        if ($hasApprovedOrders) {
            $this->dispatch('notify', type: 'error', message: 'Cannot retry: this batch has approved orders.');
            return;
        }

        // Delete any skeleton orders created from previous attempt to avoid duplicates
        $batch->commissionOrders()
            ->where('status', 'skeleton')
            ->each(function ($order) {
                // Delete lines and their quantities first
                $order->lines()->each(function ($line) {
                    $line->quantities()->delete();
                    $line->delete();
                });
                $order->delete();
            });

        // Clear the commission_order_id link if set
        $batch->update(['commission_order_id' => null]);

        // Reset batch status
        $batch->update([
            'status' => 'pending',
            'extraction_result' => null,
            'matching_result' => null,
            'verification_result' => null,
            'processed_files' => 0,
            'failed_files' => 0,
            'successful_files' => 0,
            'started_at' => null,
            'completed_at' => null,
        ]);

        // Reset file statuses
        $batch->files()->update([
            'status' => 'pending',
            'error_message' => null,
            'extracted_data' => null,
            'matched_lines' => null,
            'verification_issues' => null,
        ]);

        // Re-dispatch job
        ProcessCommissionBatchImport::dispatch($batch);

        $this->dispatch('notify', type: 'success', message: 'Batch queued for reprocessing.');
        $this->closeDetails();
    }

    public function retryFile(int $fileId)
    {
        $file = CommissionImportBatchFile::findOrFail($fileId);
        $batch = $file->batch;

        // Check if any orders have been approved (confirmed)
        $hasApprovedOrders = $batch->commissionOrders()
            ->where('status', 'confirmed')
            ->exists();

        if ($hasApprovedOrders) {
            $this->dispatch('notify', type: 'error', message: 'Cannot retry: this batch has approved orders.');
            return;
        }

        // Delete skeleton orders that may have been created from this file
        // (We delete all skeleton orders from batch since files may contribute to same orders)
        $batch->commissionOrders()
            ->where('status', 'skeleton')
            ->each(function ($order) {
                $order->lines()->each(function ($line) {
                    $line->quantities()->delete();
                    $line->delete();
                });
                $order->delete();
            });

        // Reset file status
        $file->update([
            'status' => 'pending',
            'error_message' => null,
            'extracted_data' => null,
            'matched_lines' => null,
            'verification_issues' => null,
        ]);

        // Update batch status if needed
        if ($batch->status === 'completed' || $batch->status === 'failed') {
            $batch->update(['status' => 'pending']);
        }

        // Re-dispatch job
        ProcessCommissionBatchImport::dispatch($batch);

        $this->dispatch('notify', type: 'success', message: 'File queued for reprocessing.');
    }

    public function deleteBatch(int $batchId)
    {
        $batch = CommissionImportBatch::with(['files', 'commissionOrders'])->findOrFail($batchId);

        // Delete files from storage
        foreach ($batch->files as $file) {
            Storage::disk('local')->delete($file->file_path);
        }

        // Delete the batch import directory
        Storage::disk('local')->deleteDirectory("commission-imports/{$batchId}");

        // Delete all related skeleton orders
        foreach ($batch->commissionOrders as $order) {
            if ($order->status === 'skeleton') {
                // Delete order files
                Storage::disk('local')->deleteDirectory("commission-orders/{$order->id}");
                
                // Delete order line quantities and lines
                foreach ($order->lines as $line) {
                    $line->quantities()->delete();
                }
                $order->lines()->delete();
                $order->delete();
            }
        }

        $batch->files()->delete();
        $batch->delete();

        $this->dispatch('notify', type: 'success', message: 'Batch deleted.');
        $this->closeDetails();
    }

    public function render()
    {
        return view('livewire.commission.batch-import')->layout('layouts.commission');
    }
}

