<?php

namespace App\Http\Livewire;

use Livewire\Component;
use App\Http\Livewire\BaseComponent;
use Illuminate\Support\Facades\Cache;

class CacheFilterableComponent extends BaseComponent
{
    public $filters = [];
    public $defaultFilters = [];

    /**
     * Define a unique cache key for each implementation.
     */
    protected function filterKey(): string
    {
        return 'default_filters';
    }

    /**
     * Load filters from cache or set defaults.
     */
    public function loadFilters()
    {
        $userId = auth()->id() ?: session()->getId();
        $cacheKey = 'filters_' . $userId . '_' . $this->filterKey();

        $this->filters = Cache::get($cacheKey, $this->defaultFilters);
    }

    /**
     * Save filters to cache.
     */
    public function saveFilters()
    {
        $userId = auth()->id() ?: session()->getId();
        $cacheKey = 'filters_' . $userId . '_' . $this->filterKey();

        Cache::put($cacheKey, $this->filters, now()->addDay());
    }

    /**
     * Reset filters to their default values.
     */
    public function resetFilters()
    {
        $this->filters = $this->defaultFilters;
        $this->saveFilters();
    }

    /**
     * Livewire Hook: Auto-save filters on update.
     */
    public function updatedFilters()
    {
        $this->saveFilters();
    }
}
