<?php

namespace App\Http\Livewire\Admin\Sizes;

use App\Models\Sizes;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class SizeView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    #[Validate([
        'search' => ['nullable', 'string', 'max:30'],
    ])]
    public $search;

    protected function filterKey(): string
    {
        return 'sizeslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['search'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['search'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    #[On('close-modal')]
    public function render()
    {
        Gate::authorize('order:read');
        $sizes = Sizes::search($this->search)->orderBy('order')->paginate(200);

        return view('livewire.admin.sizes.size-view', [
            'sizes' => $sizes
        ]);
    }

    public function delete($sizeId)
    {
        $inUse =
            DB::table('samples')->where('sizes_id', $sizeId)->exists() ||
            DB::table('prices')->where('sizes_id', $sizeId)->exists() ||
            DB::table('customer_order_line_quantities')->where('sizes_id', $sizeId)->exists() ||
            DB::table('shipment_line_sizes')->where('sizes_id', $sizeId)->exists();

        if ($inUse) {
            $this->addError('delete', 'Cannot delete this size because it is used in other records.');
            return;
        }

        // Sizes::findOrFail($sizeId)->delete();

        session()->flash('message', 'Size deleted successfully.');
    }

    public function reorderSizes($order)
    {
        $orderedIds = array_values(array_filter(explode(',', $order)));

        if (empty($orderedIds)) {
            return;
        }

        // Get the existing order slots for these IDs (sorted ascending by current order)
        $existingSlots = Sizes::whereIn('id', $orderedIds)
            ->orderBy('order')
            ->pluck('order')
            ->values()
            ->all();

        // Reassign the same slots to the IDs in their new visual order
        DB::transaction(function () use ($orderedIds, $existingSlots) {
            foreach ($orderedIds as $index => $sizeId) {
                $slot = $existingSlots[$index] ?? null;
                if ($slot !== null) {
                    Sizes::where('id', $sizeId)->update(['order' => $slot]);
                }
            }
        });
    }
}
