<?php

namespace App\Http\Livewire\Admin\Departments;

use Livewire\Component;
use App\Models\Departments;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class DepartmentView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    #[Validate([
        'search' => ['nullable', 'string', 'max:30'],
    ])]
    public $search;

    protected function filterKey(): string
    {
        return 'departmentslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['search'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['search'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount()
    {
        $this->loadFilterSettings();
    }

    #[On('close-modal')]
    public function render()
    {
        Gate::authorize('department:read');
        $departments = Departments::search($this->search)->paginate(100);

        return view('livewire.admin.departments.department-view', [
            'departments' => $departments
        ]);
    }
}
