<?php

namespace App\Http\Livewire\Admin\Customers;

use Livewire\Component;
use App\Models\Customer;
use App\Models\Countries;
use App\Models\CustomerAddress;
use App\Models\CustomerPaymentTerm;
use App\Mail\NewCustomerCreated;
use Livewire\Attributes\On;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class NewCustomer extends Component
{
    // Customer fields
    public $name = '';
    public $countries_id = '';
    public $currency = '£';
    public $vat = '';
    public $finance_email = '';
    public $phone = '';
    public $factored = false;
    public $payment_terms = 30;

    // Address fields
    public $address_name = '';
    public $address_line1 = '';
    public $address_city = '';
    public $address_postcode = '';
    public $address_countries_id = '';

    public $isOpen = false;

    protected function rules()
    {
        return [
            'name' => 'required|string|max:35|unique:customers,name',
            'countries_id' => 'required|integer|exists:countries,id',
            'currency' => 'required|in:£,$,€',
            'vat' => 'required|string|max:11',
            'finance_email' => 'required|email|unique:customers,finance_email',
            'phone' => 'required|string|max:35',
            'payment_terms' => 'required|integer|min:0|max:365',
            'address_name' => 'required|string|max:35',
            'address_countries_id' => 'required|integer|exists:countries,id',
            'address_line1' => 'nullable|string|max:50',
            'address_city' => 'nullable|string|max:35',
            'address_postcode' => 'nullable|string|max:35',
        ];
    }

    protected $messages = [
        'name.required' => 'Customer name is required.',
        'name.unique' => 'A customer with this name already exists.',
        'countries_id.required' => 'Please select a country.',
        'vat.required' => 'VAT number is required.',
        'finance_email.required' => 'Finance email is required.',
        'finance_email.email' => 'Please enter a valid email address.',
        'finance_email.unique' => 'This email is already in use.',
        'phone.required' => 'Phone number is required.',
        'payment_terms.required' => 'Payment terms (days) is required.',
        'address_name.required' => 'Address name is required.',
        'address_countries_id.required' => 'Please select a country for the address.',
    ];

    /**
     * Check if user can create customers via this quick form.
     * Allowed for: style:update, order:update, or finance:update
     */
    public function canCreate(): bool
    {
        return Gate::allows('style:update') 
            || Gate::allows('order:update') 
            || Gate::allows('finance:update');
    }

    #[Computed]
    public function countries()
    {
        return Countries::orderBy('country')->get();
    }

    #[On('open-new-customer-quick')]
    public function open()
    {
        if (!$this->canCreate()) {
            session()->flash('message', 'You do not have permission to create customers.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }
        
        $this->reset([
            'name', 'countries_id', 'currency', 'vat', 'finance_email', 
            'phone', 'payment_terms', 'address_name', 
            'address_line1', 'address_city', 'address_postcode', 'address_countries_id'
        ]);
        $this->currency = '£';
        $this->payment_terms = 30;
        $this->factored = false; // Always default to false
        $this->resetValidation();
        $this->isOpen = true;
    }

    public function close()
    {
        $this->isOpen = false;
        $this->resetValidation();
    }

    public function setCurrency($currency)
    {
        $this->currency = $currency;
    }

    public function save()
    {
        if (!$this->canCreate()) {
            session()->flash('message', 'You do not have permission to create customers.');
            session()->flash('alert-class', 'alert-danger');
            return;
        }

        $this->validate();

        try {
            DB::beginTransaction();

            // Create the customer
            $customer = Customer::create([
                'name' => $this->name,
                'countries_id' => $this->countries_id,
                'currency' => $this->currency,
                'vat' => strtoupper($this->vat),
                'finance_email' => $this->finance_email,
                'phone' => $this->phone,
                'factored' => $this->factored,
            ]);

            // Create the payment term (100% at X days)
            CustomerPaymentTerm::create([
                'customer_id' => $customer->id,
                'percentage' => 100,
                'days' => $this->payment_terms,
            ]);

            // Create the address
            CustomerAddress::create([
                'customer_id' => $customer->id,
                'name' => $this->address_name,
                'line1' => $this->address_line1,
                'city' => $this->address_city,
                'postcode' => $this->address_postcode,
                'countries_id' => $this->address_countries_id,
            ]);

            DB::commit();

            // Send email notification to finance
            $this->sendNotificationEmail($customer);

            $this->isOpen = false;
            $this->dispatch('customer-created', customerId: $customer->id);
            $this->dispatch('close-modal', name: 'new-customer-quick');
            
            session()->flash('message', "Customer '{$customer->name}' created successfully!");
            session()->flash('alert-class', 'alert-success');

        } catch (\Exception $e) {
            DB::rollBack();
            session()->flash('message', 'Error creating customer: ' . $e->getMessage());
            session()->flash('alert-class', 'alert-danger');
        }
    }

    /**
     * Send notification email to finance about the new customer.
     */
    protected function sendNotificationEmail(Customer $customer): void
    {
        $notifyEmail = env('NEW_CUSTOMER_NOTIFY_EMAIL', 'finance@roberttodds.com');
        
        if (empty($notifyEmail)) {
            return;
        }

        try {
            // Load relationships for the email
            $customer->load(['countries', 'customer_payment_terms', 'customer_addresses.countries']);
            
            $createdBy = Auth::user()->name ?? 'Unknown User';
            
            // Build URL here (not in mailable) to avoid queue serialization issues
            $baseUrl = config('app.url', 'https://knit.roberttodds.com');
            $customerUrl = $baseUrl . '/customers/' . $customer->id;
            
            // Send synchronously (not queued) to avoid serialization issues
            Mail::send('mail.new-customer-created', [
                'customer' => $customer,
                'createdBy' => $createdBy,
                'customerUrl' => $customerUrl,
            ], function ($message) use ($notifyEmail, $customer) {
                $message->to($notifyEmail)
                        ->subject('New Customer Created: ' . $customer->name);
            });
            
            Log::info('New customer notification email sent', [
                'customer_id' => $customer->id,
                'customer_name' => $customer->name,
                'notify_email' => $notifyEmail,
                'created_by' => $createdBy,
            ]);
        } catch (\Exception $e) {
            // Don't fail the whole operation if email fails
            Log::error('Failed to send new customer notification email', [
                'customer_id' => $customer->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    public function render()
    {
        return view('livewire.admin.customers.new-customer');
    }
}

