<?php

namespace App\Http\Livewire\Admin\Customers;

use Livewire\Component;
use App\Models\Customer;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Validate;
use App\Models\CustomerPaymentTerm;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class CustomerView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;
    public $openCustomerId = null;

    #[Validate([
        'search' => ['nullable', 'string', 'max:30'],
    ])]
    public $search;

    protected function filterKey(): string
    {
        return 'customerslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['search'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }

    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);

        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['search'])) {
                $this->resetPage();
            }

            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }

    public function mount($customerId = null)
    {
        $this->loadFilterSettings();
        
        // If a customer ID is provided via route, store it to auto-open the edit modal
        if ($customerId) {
            $this->openCustomerId = $customerId;
        }
    }

    #[On('close-modal')]
    public function render()
    {
        // $this->reverse();
        Gate::authorize('customer:read');

        $query = Customer::with('countries', 'customer_payment_terms');

        // Only apply search scope if search value is not null
        if (!is_null($this->search) && $this->search !== '') {
            $query->search($this->search);
        }

        $customers = $query->paginate(100);

        return view('livewire.admin.customers.customer-view', [
            'customers' => $customers
        ]);
    }

    public function reverse()
    {
        // Fetch all customers with 100% terms in CustomerPaymentTerm
        $terms = CustomerPaymentTerm::where('percentage', 100)
            ->whereNotNull('days')
            ->get();

        foreach ($terms as $term) {
            // Restore the payment_terms field in the Customer model
            Customer::where('id', $term->customer_id)
                ->update(['payment_terms' => $term->days]);
        }
    }

    public function updateFactored($customerId)
    {
        // Check if user has finance:update permission
        Gate::authorize('finance:update');
        
        $customer = Customer::findOrFail($customerId);
        $customer->update(['factored' => !$customer->factored]);
        
        session()->flash('message', 'Customer factored status updated successfully.');
        session()->flash('alert-class', 'alert-success');
    }
}
