<?php

namespace App\Http\Livewire\Admin\Customers;

use App\Models\Sizes;
use App\Models\Customer;
use Livewire\Component;
use App\Models\Countries;
use App\Models\SampleTypes;
use Livewire\Attributes\On;
use App\Models\InvoiceLayout;
use Livewire\WithFileUploads;
use Livewire\Attributes\Computed;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use App\Livewire\Forms\AdminCustomerForm;
use App\Services\ZohoService;
use App\Http\Livewire\Traits\HandlesZohoOAuth;

class CustomerEdit extends Component
{
    use WithFileUploads;
    use HandlesZohoOAuth;

    public $customer, $new;
    public AdminCustomerForm $form;
    public $zohoConnected = false;
    public $zohoMessage = '';
    public $zohoMessageType = '';
    
    // Zoho organization selection
    public $showOrgSelection = false;
    public $zohoOrganizations = [];
    public $selectedOrgs = [];
    public $sendingToZoho = false;

    public function mount()
    {
        $this->checkZohoConnection();
    }

    #[Computed]
    public function countries(){
        return Countries::all();
    }

    #[Computed]
    public function sampleTypes(){
        return SampleTypes::all();
    }

    #[Computed]
    public function sizes(){
        return Sizes::all();
    }

    #[Computed]
    public function invoiceLayouts(){
        return InvoiceLayout::all();
    }

    public function render()
    {
        Gate::authorize('customer:read');
        return view('livewire.admin.customers.customer-edit');
    }

    #[On('edit-customer')]
	public function edit($customer){
        Log::info('CustomerEdit: edit() called', ['customer' => $customer]);
        $this->customer = $customer;
		$this->form->set($customer);
        $this->checkZohoConnection();
	}
    #[On('new-customer')]
	public function new(){
		$this->form->new();
        $this->new = TRUE;
	}

    public function newAddress(){
        $this->form->newAddress();
    }

    public function deleteAddress($i){
        $this->form->deleteAddress($i);
    }

    public function newSize(){
        $this->form->newSize();
    }

    public function deleteSize($i){
        $this->form->deleteSize($i);
    }

    public function updatedForm($val, $var){
        // dd($val, $var);
    }

    public function addPaymentTerm()
    {
        $this->form->addPaymentTerm();
    }

    public function removePaymentTerm($index)
    {
        $this->form->removePaymentTerm($index);
    }

    public function save(){
        Log::info('CustomerEdit: save() called');
        // dd($this->form->samples_required);
        // Check for create permission if new customer, update permission if editing
        if ($this->new) {
            Gate::authorize('customer:create');
        } else {
            Gate::authorize('customer:update');
        }
        $this->form->update();
        // $this->reset('form');
        $this->dispatch('close-modal');
    }

    /**
     * Check if connected to Zoho
     */
    public function checkZohoConnection(): void
    {
        $zohoService = app(ZohoService::class);
        $this->zohoConnected = $zohoService->isConnected();
    }

    /**
     * Open organization selection popup for Zoho
     */
    public function saveAndSendToZoho()
    {
        // Show immediate feedback
        $this->zohoMessage = 'Processing... Please wait.';
        $this->zohoMessageType = 'info';

        try {
            // Check Zoho connection first
            $zohoService = app(ZohoService::class);
            $this->zohoConnected = $zohoService->isConnected();
            
            if (!$this->zohoConnected) {
                $this->zohoMessage = 'Please connect to Zoho Books first. Click "Connect Zoho" button.';
                $this->zohoMessageType = 'warning';
                return;
            }

            // Save the customer first
            $this->form->update();
            session()->forget(['message', 'alert-class']);

            // Get the customer model
            $customer = Customer::find($this->form->id);
            if (!$customer) {
                $this->zohoMessage = 'Customer not found after save.';
                $this->zohoMessageType = 'danger';
                return;
            }

            // Check if already has a Zoho ID
            if (!empty($customer->zoho_customer)) {
                $this->zohoMessage = 'This customer already exists in Zoho: ' . $customer->zoho_customer;
                $this->zohoMessageType = 'warning';
                return;
            }

            // Fetch organizations from Zoho
            $orgs = $zohoService->getOrganisations(cacheTtl: 300);
            
            if ($orgs->isEmpty()) {
                $this->zohoMessage = 'No Zoho organizations found. Please check your connection.';
                $this->zohoMessageType = 'warning';
                return;
            }

            $this->zohoOrganizations = $orgs->toArray();
            $this->selectedOrgs = [];
            $this->showOrgSelection = true;
            $this->zohoMessage = 'Select organization(s) below:';
            $this->zohoMessageType = 'success';

        } catch (\Exception $e) {
            $this->zohoMessage = 'Error: ' . $e->getMessage();
            $this->zohoMessageType = 'danger';
        }
    }

    /**
     * Close the organization selection popup
     */
    public function closeOrgSelection()
    {
        $this->showOrgSelection = false;
        $this->selectedOrgs = [];
    }

    /**
     * Confirm and send customer to selected Zoho organizations
     */
    public function confirmSendToZoho()
    {
        if (empty($this->selectedOrgs)) {
            $this->zohoMessage = 'Please select at least one organization.';
            $this->zohoMessageType = 'warning';
            return;
        }

        $this->sendingToZoho = true;
        $zohoService = app(ZohoService::class);
        
        $customer = Customer::find($this->form->id);
        if (!$customer) {
            $this->zohoMessage = 'Customer not found.';
            $this->zohoMessageType = 'danger';
            $this->sendingToZoho = false;
            return;
        }

        $payload = $this->buildZohoContactPayload($customer);
        $successCount = 0;
        $errors = [];
        $createdOrgNames = [];

        foreach ($this->selectedOrgs as $orgId) {
            try {
                $zohoContact = $zohoService->createContact($payload, $orgId);

                if ($zohoContact && !empty($zohoContact['contact_id'])) {
                    $successCount++;
                    
                    // Find org name for logging
                    $orgName = collect($this->zohoOrganizations)
                        ->firstWhere('organization_id', $orgId)['name'] ?? $orgId;
                    $createdOrgNames[] = $orgName;
                    
                    Log::info('Customer created in Zoho Books', [
                        'customer_id' => $customer->id,
                        'org_id' => $orgId,
                        'org_name' => $orgName,
                        'zoho_contact_id' => $zohoContact['contact_id'],
                    ]);
                }
            } catch (\Exception $e) {
                $orgName = collect($this->zohoOrganizations)
                    ->firstWhere('organization_id', $orgId)['name'] ?? $orgId;
                $errors[] = "{$orgName}: {$e->getMessage()}";
                
                Log::error('Failed to create customer in Zoho organization', [
                    'customer_id' => $customer->id,
                    'org_id' => $orgId,
                    'error' => $e->getMessage(),
                ]);
            }
        }

        // Update customer record with Zoho info if at least one succeeded
        if ($successCount > 0) {
            $customer->zoho_customer = $customer->name;
            $customer->save();
            $this->form->zoho_customer = $customer->zoho_customer;
            
            // Clear Zoho cache
            $zohoService->clearCache();
        }

        // Build result message
        if ($successCount > 0 && empty($errors)) {
            $this->zohoMessage = "Customer created in {$successCount} organization(s): " . implode(', ', $createdOrgNames);
            $this->zohoMessageType = 'success';
        } elseif ($successCount > 0 && !empty($errors)) {
            $this->zohoMessage = "Created in {$successCount} org(s), but errors occurred: " . implode('; ', $errors);
            $this->zohoMessageType = 'warning';
        } else {
            $this->zohoMessage = 'Failed to create in any organization: ' . implode('; ', $errors);
            $this->zohoMessageType = 'danger';
        }

        $this->showOrgSelection = false;
        $this->selectedOrgs = [];
        $this->sendingToZoho = false;
    }

    /**
     * Build the Zoho Books contact payload from a Customer model
     */
    protected function buildZohoContactPayload(Customer $customer): array
    {
        // Map currency symbols to Zoho currency codes
        $currencyMap = [
            '£' => 'GBP',
            '€' => 'EUR',
            '$' => 'USD',
        ];

        $payload = [
            'contact_name' => $customer->name,
            'contact_type' => 'customer',
            'currency_code' => $currencyMap[$customer->currency] ?? 'GBP',
        ];

        // Add VAT number with country code if present
        if (!empty($customer->vat)) {
            $payload['vat_reg_no'] = $customer->vat;
            
            // Zoho requires country code with VAT - get from customer's country
            $country = Countries::find($customer->countries_id);
            if ($country) {
                // Map country names to ISO country codes for Zoho
                $countryCodeMap = [
                    'United Kingdom' => 'GB',
                    'UK' => 'GB',
                    'Ireland' => 'IE',
                    'France' => 'FR',
                    'Germany' => 'DE',
                    'Italy' => 'IT',
                    'Spain' => 'ES',
                    'Netherlands' => 'NL',
                    'Belgium' => 'BE',
                    'Portugal' => 'PT',
                    'Austria' => 'AT',
                    'Denmark' => 'DK',
                    'Sweden' => 'SE',
                    'Finland' => 'FI',
                    'Poland' => 'PL',
                    'Czech Republic' => 'CZ',
                    'Romania' => 'RO',
                    'Hungary' => 'HU',
                    'Greece' => 'GR',
                    'Bulgaria' => 'BG',
                    'Croatia' => 'HR',
                    'Slovakia' => 'SK',
                    'Slovenia' => 'SI',
                    'Lithuania' => 'LT',
                    'Latvia' => 'LV',
                    'Estonia' => 'EE',
                    'Luxembourg' => 'LU',
                    'Malta' => 'MT',
                    'Cyprus' => 'CY',
                    'USA' => 'US',
                    'United States' => 'US',
                    'China' => 'CN',
                ];
                $payload['country_code'] = $countryCodeMap[$country->country] ?? 'GB';
            }
        }

        // Add payment terms (use first payment term if available)
        $paymentTerm = $customer->customer_payment_terms()->first();
        if ($paymentTerm) {
            $payload['payment_terms'] = $paymentTerm->days;
            $payload['payment_terms_label'] = 'Net ' . $paymentTerm->days;
        }

        // Add primary contact person
        if (!empty($customer->contact) || !empty($customer->email) || !empty($customer->phone)) {
            $payload['contact_persons'] = [[
                'first_name' => $customer->contact ?? $customer->name,
                'email' => $customer->email ?? '',
                'phone' => $customer->phone ?? '',
            ]];
        }

        // Add finance contact if different
        if (!empty($customer->finance_email) && $customer->finance_email !== $customer->email) {
            $payload['contact_persons'][] = [
                'first_name' => $customer->finance_contact ?? 'Finance',
                'email' => $customer->finance_email,
                'phone' => $customer->finance_phone ?? '',
            ];
        }

        // Add billing address from first customer address
        $address = $customer->customer_addresses()->first();
        if ($address) {
            $country = Countries::find($address->countries_id);
            $payload['billing_address'] = [
                'attention' => $address->name,
                'address' => trim(implode("\n", array_filter([
                    $address->line1,
                    $address->line2,
                    $address->line3,
                ]))),
                'city' => $address->city,
                'state' => $address->region,
                'zip' => $address->postcode,
                'country' => $country->country ?? '',
            ];
        }

        // Add notes
        if (!empty($customer->notes)) {
            $payload['notes'] = $customer->notes;
        }

        return $payload;
    }
}
