<?php

namespace App\Http\Livewire\Admin\Counts;

use App\Models\Count;
use Livewire\Component;
use Livewire\Attributes\On;
use Livewire\WithPagination;
use Livewire\Attributes\Validate;
use Illuminate\Support\Facades\Gate;
use App\Http\Livewire\Traits\Filterable;
use App\Http\Livewire\FilterableComponent;

class CountView extends FilterableComponent
{
    use WithPagination;
    use Filterable;

    public $hideFilters = false;

    #[Validate([
        'search' => ['nullable', 'string', 'max:30'],
    ])]
    public $search;

    protected function filterKey(): string
    {
        return 'countslistfilters';
    }

    protected function columnsVersion(): float
    {
        return 1;
    }

    protected function filters(): array
    {
        return ['search'];
    }

    public function getFilterKeyString(): string
    {
        return $this->filterKey();
    }
    
    public function updated($propertyName)
    {
        // Validate individual field
        $this->validateOnly($propertyName);
        
        // Check for errors and reset if validation fails
        if ($this->getErrorBag()->has($propertyName)) {
            $this->reset($propertyName);
        } else {
            // Reset pagination only if a relevant property is changed
            if (in_array($propertyName, ['search'])) {
                $this->resetPage();
            }
            
            // Save filter settings since validation passed
            $this->saveFilterSettings();
        }
    }
    
    public function mount()
    {
        $this->loadFilterSettings();
    }

    #[On('close-modal')]
    public function render()
    {
        Gate::authorize('style:read');
        $counts = Count::search($this->search)->paginate(100);

        return view('livewire.admin.counts.count-view', [
            'counts' => $counts
        ]);
    }
}
