<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use App\Models\ShipmentLine;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ShipmentScheduleExportController extends Controller
{
    public function exportToGoogleSheets(Request $request)
    {
        Gate::authorize('shipment:read');
        
        try {
            // Get filters from request
            $filters = $request->all();
            
            // Build query with filters
            $query = $this->buildExportQuery($filters);
            $drops = $query->get();
            
            // Initialize Google Client with user's OAuth token
            $user = auth()->user();
            
            // Check if user has connected their Google account
            if (!$user->google_access_token) {
                return response()->json([
                    'success' => false,
                    'error' => 'Please connect your Google account first',
                    'needs_auth' => true
                ], 401);
            }
            
            // Check if token is expired and refresh if needed
            if ($user->google_token_expires_at && $user->google_token_expires_at->isPast()) {
                $client = new \Google_Client();
                $client->setClientId(config('services.google.client_id'));
                $client->setClientSecret(config('services.google.client_secret'));
                $client->setAccessType('offline');
                $client->fetchAccessTokenWithRefreshToken($user->google_refresh_token);
                
                $newToken = $client->getAccessToken();
                $user->google_access_token = $newToken['access_token'];
                $user->google_token_expires_at = now()->addSeconds($newToken['expires_in']);
                $user->save();
            }
            
            $client = new \Google_Client();
            $client->setApplicationName('Shipment Schedule Export');
            $client->setAccessToken($user->google_access_token);
            
            $service = new \Google_Service_Sheets($client);
            
            // Create spreadsheet
            $spreadsheet = new \Google_Service_Sheets_Spreadsheet([
                'properties' => [
                    'title' => 'Shipment Schedule - ' . date('Y-m-d H:i:s')
                ]
            ]);
            
            $spreadsheet = $service->spreadsheets->create($spreadsheet);
            $spreadsheetId = $spreadsheet->spreadsheetId;
            
            // Prepare data
            $values = $this->prepareSheetData($drops, $filters);
            
            // Update spreadsheet with data
            $body = new \Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);
            
            $params = [
                'valueInputOption' => 'RAW'
            ];
            
            $range = 'Sheet1!A1';
            $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
            
            // Format header row
            $requests = [
                new \Google_Service_Sheets_Request([
                    'repeatCell' => [
                        'range' => [
                            'sheetId' => 0,
                            'startRowIndex' => 0,
                            'endRowIndex' => 1,
                        ],
                        'cell' => [
                            'userEnteredFormat' => [
                                'backgroundColor' => [
                                    'red' => 0.2,
                                    'green' => 0.2,
                                    'blue' => 0.2
                                ],
                                'textFormat' => [
                                    'foregroundColor' => [
                                        'red' => 1.0,
                                        'green' => 1.0,
                                        'blue' => 1.0
                                    ],
                                    'bold' => true
                                ]
                            ]
                        ],
                        'fields' => 'userEnteredFormat(backgroundColor,textFormat)'
                    ]
                ])
            ];
            
            $batchUpdateRequest = new \Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
                'requests' => $requests
            ]);
            
            $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
            
            // Return URL - spreadsheet is already in user's Drive
            $url = "https://docs.google.com/spreadsheets/d/{$spreadsheetId}/edit";
            
            return response()->json([
                'success' => true,
                'url' => $url
            ]);
            
        } catch (\Exception $e) {
            Log::error('Google Sheets export failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    private function buildExportQuery($filters)
    {
        $query = ShipmentLine::query();
        
        $query = $this->applyFilters($query, $filters)
            ->select([
                'shipment_lines.id',
                'shipment_lines.exfty',
                'shipment_lines.shipment_id',
                'shipment_lines.notes',
                'shipment_lines.complete',
                'shipment_lines.rt_invoice',
                'shipment_lines.collection_date',
                'customer_orders.customer_po',
                'customer_orders.incoterms',
                'customers.name as customer_name',
                'seasons.description as season_description',
                'factory.name as factory_name',
                'styles.designs_id',
                'styles.customer_ref',
                'colourways.name as colourway_name',
                'style_versions.name as version_name',
                'designs.description as design_description',
                'customer_order_lines.factory_cust_date',
                'customer_order_lines.wh_cust_date',
                'shipments.customs_submitted_text',
                'shipments.organiser as truck_organiser',
                'shipments.truck_first_collection',
            ])
            ->selectRaw('
                (SELECT SUM(sls.qty) 
                 FROM shipment_line_sizes sls 
                 WHERE sls.shipment_line_id = shipment_lines.id) as total_pieces
            ')
            ->selectRaw('
                (SELECT SUM(sls.shipped_qty) 
                 FROM shipment_line_sizes sls 
                 WHERE sls.shipment_line_id = shipment_lines.id) as total_pieces_shipped
            ')
            ->selectRaw("
                COALESCE(
                    (SELECT JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                     FROM audits AS audit
                     WHERE audit.auditable_id = shipment_lines.id
                     AND audit.auditable_type = 'App\\\Models\\\ShipmentLine'
                     AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                     ORDER BY audit.created_at ASC
                     LIMIT 1),
                    shipment_lines.exfty
                ) AS first_exfty
            ")
            ->selectRaw("
                (SELECT status 
                 FROM samples 
                 WHERE samples.colourways_id = colourways.id 
                 AND samples.sample_types_id = 3 
                 AND samples.deleted_at IS NULL
                 ORDER BY samples.created_at DESC
                 LIMIT 1) as sealer_sample_status
            ")
            ->selectRaw("
                (SELECT status 
                 FROM samples 
                 WHERE samples.colourways_id = colourways.id 
                 AND samples.sample_types_id = 7 
                 AND samples.deleted_at IS NULL
                 ORDER BY samples.created_at DESC
                 LIMIT 1) as shipment_sample_status
            ")
            ->selectRaw('
                (SELECT JSON_ARRAYAGG(
                    JSON_OBJECT(
                        "name", sizes.name,
                        "qty", sls.qty,
                        "shipped_qty", sls.shipped_qty
                    )
                )
                FROM shipment_line_sizes sls
                JOIN sizes ON sizes.id = sls.sizes_id
                WHERE sls.shipment_line_id = shipment_lines.id
                ORDER BY sizes.order) as shipment_line_sizes
            ')
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->join('customer_orders', function($join) {
                $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                $join->on('customer_orders.cancelled', '=', DB::raw(0));
            })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
            ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
            ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
            ->join('designs', 'designs.id', '=', 'styles.designs_id')
            ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
            ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
            ->leftJoin('shipments', 'shipments.id', '=', 'shipment_lines.shipment_id')
            ->leftJoin('suppliers as transporters', 'transporters.id', '=', 'shipments.transporter_id')
            ->orderByRaw("
                -(shipment_lines.complete),
                -(shipments.truck_first_collection) desc,
                shipment_lines.shipment_id,
                -(COALESCE(shipments.truck_first_collection, shipment_lines.exfty)) desc,
                factory.countries_id,
                -(factory.name),
                -(customers.name),
                -(customer_orders.customer_po),
                -(styles.id),
                -(colourways.name),
                shipment_lines.id
            ");
        
        return $query;
    }
    
    private function applyFilters($query, $filters)
    {
        $query->where('customer_orders.cancelled', 0)
              ->where('customer_order_lines.cancelled', 0)
              ->where('customer_orders.order_type', 'wholesale');

        // Hide Mabli filter (temporary)
        if (!empty($filters['hideMabli'])) {
            $query->where('customer_orders.customers_id', '!=', 9);
        }

        if (!empty($filters['view'])) {
            if ($filters['view'] == "unshipped") {
                $query->where('shipment_lines.complete', false);
            } elseif ($filters['view'] == "financeNC") {
                $query->where('shipment_lines.invoiced', false);
            } elseif ($filters['view'] == "cashflow") {
                $query->whereNull('shipment_lines.rt_invoice');
            } elseif ($filters['view'] == "noRTInvoice") {
                $query->where(function($q) {
                    $q->whereNull('shipment_lines.rt_invoice')
                      ->orWhere('shipment_lines.rt_invoice', '');
                });
            } else {
                // Only apply fromDate when view is "all"
                if (!empty($filters['fromDate'])) {
                    $query->where('shipment_lines.exfty', '>=', $filters['fromDate']);
                }
            }
        } else {
            // If no view is set, treat it as "all" and apply fromDate
            if (!empty($filters['fromDate'])) {
                $query->where('shipment_lines.exfty', '>=', $filters['fromDate']);
            }
        }

        if (!empty($filters['toDate'])) {
            $query->where('shipment_lines.exfty', '<=', $filters['toDate']);
        }

        if (!empty($filters['season'])) {
            $query->whereIn('customer_orders.seasons_id', array_keys($filters['season']));
        }

        if (!empty($filters['customer'])) {
            $query->whereIn('customer_orders.customers_id', array_keys($filters['customer']));
        }

        if (!empty($filters['factory'])) {
            $query->whereIn('style_versions.factory_id', array_keys($filters['factory']));
        }

        // COO (Country of Origin) filter
        if (!empty($filters['coo'])) {
            $query->whereIn('factory.countries_id', array_keys($filters['coo']));
        }

        // Department filter
        if (!empty($filters['department'])) {
            $query->whereIn('styles.departments_id', array_keys($filters['department']));
        }

        // Category filter
        if (!empty($filters['category'])) {
            $query->whereIn('styles.category', array_keys($filters['category']));
        }

        // Truck filter
        if (!empty($filters['noTruck'])) {
            $query->whereNull('shipment_lines.shipment_id');
        } elseif (!empty($filters['truck'])) {
            $query->where('shipments.id', $filters['truck']);
        }

        if (!empty($filters['search'])) {
            $query->search($filters['search']);
        }

        return $query;
    }
    
    private function prepareSheetData($drops, $filters)
    {
        $columns = $filters['columns'] ?? [];
        $data = [];
        
        // First pass: collect all unique sizes in order
        $allSizeNames = [];
        if($columns['Sizes'] ?? true) {
            foreach($drops as $drop) {
                $sizes = json_decode($drop->shipment_line_sizes ?? '[]', true) ?: [];
                foreach($sizes as $size) {
                    $sizeName = $size['name'] ?? '';
                    if ($sizeName && !in_array($sizeName, $allSizeNames)) {
                        $allSizeNames[] = $sizeName;
                    }
                }
            }
        }
        
        // Build header row
        $headerRow = ['Drop#'];
        if($columns['Season'] ?? true) $headerRow[] = 'Season';
        if($columns['Factory'] ?? true) $headerRow[] = 'Factory';
        if($columns['Customer'] ?? true) $headerRow[] = 'Customer';
        if($columns['Order No'] ?? true) $headerRow[] = 'Order No';
        if($columns['Style'] ?? true) $headerRow[] = 'Style';
        $headerRow[] = 'Customer Ref';
        if($columns['Description'] ?? true) $headerRow[] = 'Description';
        if($columns['Colour'] ?? true) $headerRow[] = 'Colour';
        
        // Add size columns - ordered first, then shipped
        if($columns['Sizes'] ?? true) {
            foreach($allSizeNames as $sizeName) {
                $headerRow[] = $sizeName . ' (Ordered)';
            }
            foreach($allSizeNames as $sizeName) {
                $headerRow[] = $sizeName . ' (Shipped)';
            }
        }
        
        if($columns['Total Qty'] ?? true) $headerRow[] = 'Total Qty';
        if($columns['Total Shipped'] ?? true) $headerRow[] = 'Total Shipped';
        if($columns['Sealer Sample'] ?? true) $headerRow[] = 'Sealer Sample';
        if($columns['Shipment Sample'] ?? true) $headerRow[] = 'Shipment Sample';
        if($columns['ExFty'] ?? true) $headerRow[] = 'First ExFty';
        if($columns['Revised ExFty'] ?? true) $headerRow[] = 'Revised ExFty';
        if($columns['Customer exFty'] ?? true) $headerRow[] = 'Customer exFty';
        if($columns['Customer Into w/h'] ?? true) $headerRow[] = 'Customer Into w/h';
        if($columns['Truck'] ?? true) $headerRow[] = 'Truck';
        $headerRow[] = 'Truck Organiser';
        $headerRow[] = 'Truck Collection Date';
        if($columns['Customs ERN'] ?? true) $headerRow[] = 'Customs ERN';
        if($columns['Shipped'] ?? true) $headerRow[] = 'Shipped';
        if($columns['RT Invoice'] ?? false) $headerRow[] = 'RT Invoice';
        $headerRow[] = 'Notes';
        
        $data[] = $headerRow;
        
        // Add data rows
        foreach($drops as $drop) {
            $row = [$drop->id];
            
            if($columns['Season'] ?? true) $row[] = $drop->season_description ?? '';
            if($columns['Factory'] ?? true) $row[] = $drop->factory_name ?? '';
            if($columns['Customer'] ?? true) $row[] = $drop->customer_name ?? '';
            if($columns['Order No'] ?? true) $row[] = $drop->customer_po ?? '';
            if($columns['Style'] ?? true) $row[] = ($drop->designs_id ?? '') . ' - ' . ($drop->version_name ?? '');
            $row[] = $drop->customer_ref ?? '';
            if($columns['Description'] ?? true) $row[] = $drop->design_description ?? '';
            if($columns['Colour'] ?? true) $row[] = $drop->colourway_name ?? '';
            
            if($columns['Sizes'] ?? true) {
                $sizes = json_decode($drop->shipment_line_sizes ?? '[]', true) ?: [];
                
                // Create a map of size name to quantities
                $sizeMap = [];
                foreach($sizes as $size) {
                    $sizeMap[$size['name'] ?? ''] = [
                        'qty' => $size['qty'] ?? 0,
                        'shipped_qty' => $size['shipped_qty'] ?? 0
                    ];
                }
                
                // Add ordered quantities for each size
                foreach($allSizeNames as $sizeName) {
                    $row[] = $sizeMap[$sizeName]['qty'] ?? 0;
                }
                
                // Add shipped quantities for each size
                foreach($allSizeNames as $sizeName) {
                    $row[] = $sizeMap[$sizeName]['shipped_qty'] ?? 0;
                }
            }
            
            if($columns['Total Qty'] ?? true) $row[] = $drop->total_pieces ?? 0;
            if($columns['Total Shipped'] ?? true) $row[] = $drop->total_pieces_shipped ?? 0;
            if($columns['Sealer Sample'] ?? true) $row[] = $drop->sealer_sample_status ?? '';
            if($columns['Shipment Sample'] ?? true) $row[] = $drop->shipment_sample_status ?? '';
            if($columns['ExFty'] ?? true) $row[] = $drop->first_exfty ? date('Y-m-d', strtotime($drop->first_exfty)) : '';
            if($columns['Revised ExFty'] ?? true) $row[] = $drop->exfty ? date('Y-m-d', strtotime($drop->exfty)) : '';
            if($columns['Customer exFty'] ?? true) $row[] = $drop->factory_cust_date ? date('Y-m-d', strtotime($drop->factory_cust_date)) : '';
            if($columns['Customer Into w/h'] ?? true) $row[] = $drop->wh_cust_date ? date('Y-m-d', strtotime($drop->wh_cust_date)) : '';
            if($columns['Truck'] ?? true) $row[] = $drop->shipment_id ?? '';
            $row[] = $drop->truck_organiser ?? '';
            $row[] = $drop->truck_first_collection ? date('Y-m-d', strtotime($drop->truck_first_collection)) : '';
            if($columns['Customs ERN'] ?? true) $row[] = $drop->customs_submitted_text ?? '';
            if($columns['Shipped'] ?? true) $row[] = $drop->complete ? 'Yes' : 'No';
            if($columns['RT Invoice'] ?? false) $row[] = $drop->rt_invoice ?? '';
            $row[] = $drop->notes ?? '';
            
            $data[] = $row;
        }
        
        return $data;
    }
}

