<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use App\Models\ShipmentLine;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SalesOutlookExportController extends Controller
{
    public function exportToGoogleSheets(Request $request)
    {
        Gate::authorize('shipment:read');
        
        try {
            // Get filters from request
            $filters = $request->all();
            
            // Validate date range
            $this->validateFilters($filters);
            
            // Build query with filters
            $data = $this->getData($filters);
            $months = $this->getMonthsWithData($data, $filters['fromDate'], $filters['toDate']);
            
            // Initialize Google Client with user's OAuth token
            $user = auth()->user();
            
            // Check if user has connected their Google account
            if (!$user->google_access_token) {
                return response()->json([
                    'success' => false,
                    'error' => 'Please connect your Google account first',
                    'needs_auth' => true
                ], 401);
            }
            
            // Check if token is expired and refresh if needed
            if ($user->google_token_expires_at && $user->google_token_expires_at->isPast()) {
                $client = new \Google_Client();
                $client->setClientId(config('services.google.client_id'));
                $client->setClientSecret(config('services.google.client_secret'));
                $client->setAccessType('offline');
                $client->fetchAccessTokenWithRefreshToken($user->google_refresh_token);
                
                $newToken = $client->getAccessToken();
                $user->google_access_token = $newToken['access_token'];
                $user->google_token_expires_at = now()->addSeconds($newToken['expires_in']);
                $user->save();
            }
            
            $client = new \Google_Client();
            $client->setApplicationName('Sales Outlook Export');
            $client->setAccessToken($user->google_access_token);
            
            $service = new \Google_Service_Sheets($client);
            
            // Create spreadsheet
            $spreadsheet = new \Google_Service_Sheets_Spreadsheet([
                'properties' => [
                    'title' => 'Sales Outlook - ' . date('Y-m-d H:i:s')
                ]
            ]);
            
            $spreadsheet = $service->spreadsheets->create($spreadsheet);
            $spreadsheetId = $spreadsheet->spreadsheetId;
            
            // Prepare data
            $values = $this->prepareSheetData($data, $months, $filters);
            
            // Update spreadsheet with data
            $body = new \Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);
            
            $params = [
                'valueInputOption' => 'RAW'
            ];
            
            $range = 'Sheet1!A1';
            $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
            
            // Format header row
            $requests = [
                new \Google_Service_Sheets_Request([
                    'repeatCell' => [
                        'range' => [
                            'sheetId' => 0,
                            'startRowIndex' => 0,
                            'endRowIndex' => 1,
                        ],
                        'cell' => [
                            'userEnteredFormat' => [
                                'backgroundColor' => [
                                    'red' => 0.2,
                                    'green' => 0.2,
                                    'blue' => 0.2
                                ],
                                'textFormat' => [
                                    'foregroundColor' => [
                                        'red' => 1.0,
                                        'green' => 1.0,
                                        'blue' => 1.0
                                    ],
                                    'bold' => true
                                ]
                            ]
                        ],
                        'fields' => 'userEnteredFormat(backgroundColor,textFormat)'
                    ]
                ])
            ];
            
            $batchUpdateRequest = new \Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
                'requests' => $requests
            ]);
            
            $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
            
            // Return URL - spreadsheet is already in user's Drive
            $url = "https://docs.google.com/spreadsheets/d/{$spreadsheetId}/edit";
            
            return response()->json([
                'success' => true,
                'url' => $url
            ]);
            
        } catch (\Exception $e) {
            Log::error('Sales Outlook Google Sheets export failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    public function exportToCsv(Request $request)
    {
        Gate::authorize('shipment:read');
        
        try {
            $filters = $request->all();
            
            // Validate date range
            $this->validateFilters($filters);
            
            // Build query with filters
            $data = $this->getData($filters);
            $months = $this->getMonthsWithData($data, $filters['fromDate'], $filters['toDate']);
            
            // Prepare CSV data
            $values = $this->prepareSheetData($data, $months, $filters);
            
            // Generate CSV
            $filename = 'sales-outlook-' . date('Y-m-d-His') . '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename=\"$filename\"",
            ];
            
            $callback = function() use ($values) {
                $file = fopen('php://output', 'w');
                foreach ($values as $row) {
                    fputcsv($file, $row);
                }
                fclose($file);
            };
            
            return response()->stream($callback, 200, $headers);
            
        } catch (\Exception $e) {
            Log::error('Sales Outlook CSV Export Error: ' . $e->getMessage());
            return back()->with('error', 'Error exporting to CSV: ' . $e->getMessage());
        }
    }
    
    private function validateFilters($filters)
    {
        if (empty($filters['fromDate']) || empty($filters['toDate'])) {
            throw new \Exception('Date range is required');
        }
        
        $from = Carbon::parse($filters['fromDate']);
        $to = Carbon::parse($filters['toDate']);
        
        if ($to->diffInYears($from) > 2) {
            throw new \Exception('Date range cannot exceed 2 years');
        }
        
        if (!isset($filters['metric']) || !in_array($filters['metric'], ['sales_value', 'total_qty'])) {
            throw new \Exception('Invalid metric selected');
        }
    }
    
    private function getMonthsWithData($data, $fromDate, $toDate)
    {
        // First, get all possible months in the range
        $allMonths = [];
        $start = Carbon::parse($fromDate)->startOfMonth();
        $end = Carbon::parse($toDate)->endOfMonth();
        
        while ($start->lte($end)) {
            $allMonths[] = [
                'key' => $start->format('Y-m'),
                'label' => $start->format('M Y')
            ];
            $start->addMonth();
        }
        
        // Find months that actually have data (using base values for consistency)
        $monthsWithData = [];
        foreach ($allMonths as $month) {
            $hasData = false;
            foreach ($data as $customerData) {
                if (isset($customerData['months_base'][$month['key']]) && $customerData['months_base'][$month['key']] > 0) {
                    $hasData = true;
                    break;
                }
            }
            if ($hasData) {
                $monthsWithData[] = $month;
            }
        }
        
        return $monthsWithData;
    }
    
    private function getData($filters)
    {
        $fromDate = $filters['fromDate'];
        $toDate = $filters['toDate'];
        $includeAll = $filters['includeAll'] ?? false;
        
        $query = ShipmentLine::query()
            ->select([
                'customers.id as customer_id',
                'customers.name as customer_name',
                'customers.currency as customer_currency',
                DB::raw('DATE_FORMAT(shipment_lines.exfty, "%Y-%m") as month'),
                DB::raw('SUM(COALESCE(
                    CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote")) AS DECIMAL(18,2)), 
                    0
                )) as sales_value_customer'),
                DB::raw('SUM(COALESCE(
                    CAST(JSON_UNQUOTE(JSON_EXTRACT(total_cache.cached_data, "$.quote_base")) AS DECIMAL(18,2)), 
                    0
                )) as sales_value_base'),
                DB::raw('SUM(COALESCE((
                    SELECT SUM(colq.qty) 
                    FROM customer_order_line_quantities colq 
                    WHERE colq.customer_order_lines_id = customer_order_lines.id
                    AND colq.deleted_at IS NULL
                ), 0)) as total_qty'),
                DB::raw('SUM(COALESCE((
                    SELECT SUM(sls.qty) 
                    FROM shipment_line_sizes sls 
                    WHERE sls.shipment_line_id = shipment_lines.id
                    AND sls.deleted_at IS NULL
                ), 0)) as total_pieces')
            ])
            ->join('customer_order_lines', function ($join) {
                $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id')
                    ->where('customer_order_lines.cancelled', 0);
            })
            ->join('customer_orders', function($join) {
                $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
                $join->on('customer_orders.order_type', '=', DB::raw("'wholesale'"));
                $join->on('customer_orders.cancelled', '=', DB::raw(0));
            })
            ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
            ->leftJoin('total_cache', function($join) {
                $join->on('total_cache.entity_id', '=', 'shipment_lines.id')
                    ->where('total_cache.entity_type', '=', 'shipment_line')
                    ->where('total_cache.cache_key', '=', 'prices')
                    ->whereNotNull('total_cache.fresh_at');
            })
            ->whereNull('shipment_lines.deleted_at')
            ->whereNull('customer_order_lines.deleted_at')
            ->whereNull('customer_orders.deleted_at')
            ->whereNull('customers.deleted_at')
            ->whereBetween('shipment_lines.exfty', [$fromDate, $toDate]);
        
        // Filter for unshipped or all
        if (!$includeAll) {
            $query->where('shipment_lines.complete', false);
        }
        
        $query->groupBy('customers.id', 'customers.name', 'customers.currency', 'month')
            ->orderBy('customers.name')
            ->orderBy('month');
        
        $results = $query->get();
        
        // Transform data into customer => month => value structure
        $data = [];
        foreach ($results as $row) {
            $customerId = $row->customer_id;
            $customerName = $row->customer_name;
            $currency = $row->customer_currency;
            $month = $row->month;
            
            if (!isset($data[$customerId])) {
                $data[$customerId] = [
                    'customer_id' => $customerId,
                    'customer_name' => $customerName,
                    'currency' => $currency,
                    'months' => [],
                    'months_base' => [],
                    'total' => 0,
                    'total_base' => 0
                ];
            }
            
            // Get the metric values (customer currency and base)
            $value = $this->getMetricValue($row, $filters['metric'], false);
            $valueBase = $this->getMetricValue($row, $filters['metric'], true);
            
            $data[$customerId]['months'][$month] = $value;
            $data[$customerId]['months_base'][$month] = $valueBase;
            $data[$customerId]['total'] += $value;
            $data[$customerId]['total_base'] += $valueBase;
        }
        
        return array_values($data);
    }
    
    private function getMetricValue($row, $metric, $useBase = false)
    {
        switch ($metric) {
            case 'sales_value':
                if ($useBase) {
                    return round($row->sales_value_base ?? 0, 2);
                }
                return round($row->sales_value_customer ?? 0, 2);
            case 'total_qty':
                return $row->total_qty ?? 0;
            default:
                return 0;
        }
    }
    
    private function prepareSheetData($data, $months, $filters)
    {
        $metric = $filters['metric'];
        $showBaseCurrency = $filters['showBaseCurrency'] ?? false;
        $values = [];
        
        // Build header row
        $headerRow = ['Customer', 'Currency'];
        foreach ($months as $month) {
            $headerRow[] = $month['label'];
        }
        $headerRow[] = 'Total';
        $values[] = $headerRow;
        
        // Add data rows
        $columnTotals = [];
        $grandTotal = 0;
        
        foreach ($data as $row) {
            $dataRow = [
                $row['customer_name'],
                $showBaseCurrency ? 'GBP' : $row['currency']
            ];
            
            foreach ($months as $month) {
                // Display value (customer or base depending on toggle)
                $displayValue = $showBaseCurrency 
                    ? ($row['months_base'][$month['key']] ?? 0)
                    : ($row['months'][$month['key']] ?? 0);
                
                // Always use base for totals
                $baseValue = $row['months_base'][$month['key']] ?? 0;
                if (!isset($columnTotals[$month['key']])) {
                    $columnTotals[$month['key']] = 0;
                }
                $columnTotals[$month['key']] += $baseValue;
                
                if ($metric === 'sales_value') {
                    $dataRow[] = $displayValue > 0 ? number_format($displayValue, 2, '.', '') : '';
                } else {
                    $dataRow[] = $displayValue > 0 ? $displayValue : '';
                }
            }
            
            // Display total (customer or base depending on toggle)
            $displayTotal = $showBaseCurrency ? $row['total_base'] : $row['total'];
            // Always use base for grand total
            $grandTotal += $row['total_base'];
            
            if ($metric === 'sales_value') {
                $dataRow[] = number_format($displayTotal, 2, '.', '');
            } else {
                $dataRow[] = $displayTotal;
            }
            
            $values[] = $dataRow;
        }
        
        // Add grand totals row (always in base currency)
        $totalsRow = ['GRAND TOTAL', 'GBP'];
        foreach ($months as $month) {
            $value = $columnTotals[$month['key']] ?? 0;
            if ($metric === 'sales_value') {
                $totalsRow[] = number_format($value, 2, '.', '');
            } else {
                $totalsRow[] = $value;
            }
        }
        
        if ($metric === 'sales_value') {
            $totalsRow[] = number_format($grandTotal, 2, '.', '');
        } else {
            $totalsRow[] = $grandTotal;
        }
        
        $values[] = $totalsRow;
        
        return $values;
    }
}

