<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use App\Models\ShipmentLine;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CommissionShipmentScheduleExportController extends Controller
{
    public function exportToGoogleSheets(Request $request)
    {
        Gate::authorize('shipment:read');
        
        try {
            // Get filters from request
            $filters = $request->all();
            
            // Build query with filters
            $query = $this->buildExportQuery($filters);
            $drops = $query->get();
            
            // Initialize Google Client with user's OAuth token
            $user = auth()->user();
            
            // Check if user has connected their Google account
            if (!$user->google_access_token) {
                return response()->json([
                    'success' => false,
                    'error' => 'Please connect your Google account first',
                    'needs_auth' => true
                ], 401);
            }
            
            // Check if token is expired and refresh if needed
            if ($user->google_token_expires_at && $user->google_token_expires_at->isPast()) {
                $client = new \Google_Client();
                $client->setClientId(config('services.google.client_id'));
                $client->setClientSecret(config('services.google.client_secret'));
                $client->setAccessType('offline');
                $client->fetchAccessTokenWithRefreshToken($user->google_refresh_token);
                
                $newToken = $client->getAccessToken();
                $user->google_access_token = $newToken['access_token'];
                $user->google_token_expires_at = now()->addSeconds($newToken['expires_in']);
                $user->save();
            }
            
            $client = new \Google_Client();
            $client->setApplicationName('Commission Shipment Schedule Export');
            $client->setAccessToken($user->google_access_token);
            
            $service = new \Google_Service_Sheets($client);
            
            // Create spreadsheet
            $spreadsheet = new \Google_Service_Sheets_Spreadsheet([
                'properties' => [
                    'title' => 'Commission Shipment Schedule - ' . date('Y-m-d H:i:s')
                ]
            ]);
            
            $spreadsheet = $service->spreadsheets->create($spreadsheet);
            $spreadsheetId = $spreadsheet->spreadsheetId;
            
            // Prepare data
            $values = $this->prepareSheetData($drops);
            
            // Update spreadsheet with data
            $body = new \Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);
            
            $params = [
                'valueInputOption' => 'RAW'
            ];
            
            $range = 'Sheet1!A1';
            $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
            
            // Format header row
            $requests = [
                new \Google_Service_Sheets_Request([
                    'repeatCell' => [
                        'range' => [
                            'sheetId' => 0,
                            'startRowIndex' => 0,
                            'endRowIndex' => 1,
                        ],
                        'cell' => [
                            'userEnteredFormat' => [
                                'backgroundColor' => [
                                    'red' => 0.2,
                                    'green' => 0.2,
                                    'blue' => 0.2
                                ],
                                'textFormat' => [
                                    'foregroundColor' => [
                                        'red' => 1.0,
                                        'green' => 1.0,
                                        'blue' => 1.0
                                    ],
                                    'bold' => true
                                ]
                            ]
                        ],
                        'fields' => 'userEnteredFormat(backgroundColor,textFormat)'
                    ]
                ])
            ];
            
            $batchUpdateRequest = new \Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
                'requests' => $requests
            ]);
            
            $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
            
            // Return URL - spreadsheet is already in user's Drive
            $url = "https://docs.google.com/spreadsheets/d/{$spreadsheetId}/edit";
            
            return response()->json([
                'success' => true,
                'url' => $url
            ]);
            
        } catch (\Exception $e) {
            Log::error('Commission Google Sheets export failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    private function buildExportQuery($filters)
    {
        $query = ShipmentLine::select(
            "shipment_lines.id AS shipment_lines_id",
            "seasons.description AS season",
            "factory.name AS factory",
            "customers.name AS customer",
            "customer_orders.customer_po AS customer_po",
            "designs.description AS description",
            "colourways.name AS colour",
            "shipment_lines.complete AS shipped",
            "shipment_lines.rt_invoice AS rt_invoice",
            "shipment_lines.exfty AS revised_exfty",
            "styles.designs_id",
            "customers.currency AS cust_currency",
        )
        ->selectRaw("
            COALESCE(
                (SELECT JSON_UNQUOTE(JSON_EXTRACT(audit.new_values, '$.exfty'))
                 FROM audits AS audit
                 WHERE audit.auditable_id = shipment_lines.id
                 AND audit.auditable_type = 'App\\\Models\\\ShipmentLine'
                 AND JSON_EXTRACT(audit.new_values, '$.exfty') IS NOT NULL
                 ORDER BY audit.created_at ASC
                 LIMIT 1),
                shipment_lines.exfty
            ) AS first_exfty
        ")
        ->selectRaw("
            (SELECT
                concat('[', group_concat(
                JSON_OBJECT(
                    'name', sizes.name,
                    'qty', sls.qty,
                    'shipped_qty', sls.shipped_qty,
                    'price', colq.price,
                    'commission', colq.commission
                ) ORDER BY sizes.`order` SEPARATOR ','), ']')
            FROM shipment_line_sizes sls
            JOIN sizes on sizes.id = sls.sizes_id and sls.shipment_line_id = shipment_lines.id
            JOIN customer_order_line_quantities colq on colq.customer_order_lines_id = customer_order_lines.id and colq.sizes_id = sls.sizes_id
            GROUP BY sls.shipment_line_id
            ) AS sizes
        ")
        ->join('customer_order_lines', function($join) {
            $join->on('customer_order_lines.id', '=', 'shipment_lines.customer_order_lines_id');
            $join->on('customer_order_lines.cancelled', '=', DB::raw(0));
        })
        ->join('customer_orders', function($join) {
            $join->on('customer_orders.id', '=', 'customer_order_lines.customer_orders_id');
            $join->on('customer_orders.order_type', '=', DB::raw("'commission'"));
            $join->on('customer_orders.cancelled', '=', DB::raw(0));
        })
        ->join('customers', 'customers.id', '=', 'customer_orders.customers_id')
        ->join('colourways', 'colourways.id', '=', 'customer_order_lines.colourways_id')
        ->join('style_versions', 'style_versions.id', '=', 'colourways.style_versions_id')
        ->join('styles', 'styles.id', '=', 'style_versions.styles_id')
        ->join('seasons', 'seasons.id', '=', 'customer_orders.seasons_id')
        ->join('suppliers as factory', 'factory.id', '=', 'style_versions.factory_id')
        ->join('designs', 'designs.id', '=', 'styles.designs_id')
        ->whereNull('shipment_lines.deleted_at');
        
        // Apply filters
        $query = $this->applyFilters($query, $filters);
        
        // Apply the same ordering as the main table
        $query->orderByRaw("
            -(shipment_lines.complete),
            shipment_lines.shipment_id,
            -(shipment_lines.exfty) desc,
            factory.countries_id,
            -(factory.name),
            -(customers.name),
            -(customer_orders.customer_po),
            -(styles.id),
            -(colourways.name)
        ");
        
        return $query;
    }
    
    private function applyFilters($query, $filters)
    {
        if (!empty($filters['view'])) {
            if ($filters['view'] == "unshipped") {
                $query->where('shipment_lines.complete', false);
            } elseif ($filters['view'] == "financeNC") {
                $query->where('shipment_lines.invoiced', false);
            }
        }
        
        if (!empty($filters['fromDate'])) {
            $query->where('shipment_lines.exfty', '>=', $filters['fromDate']);
        }
        
        if (!empty($filters['toDate'])) {
            $query->where('shipment_lines.exfty', '<=', $filters['toDate']);
        }
        
        if (!empty($filters['season'])) {
            $query->whereIn('customer_orders.seasons_id', array_keys($filters['season']));
        }
        
        if (!empty($filters['customer'])) {
            $query->whereIn('customer_orders.customers_id', array_keys($filters['customer']));
        }
        
        if (!empty($filters['factory'])) {
            $query->whereIn('style_versions.factory_id', array_keys($filters['factory']));
        }
        
        if (!empty($filters['coo'])) {
            $query->whereIn('factory.countries_id', array_keys($filters['coo']));
        }
        
        if (!empty($filters['search'])) {
            $query->where(function($q) use ($filters) {
                $q->where('customer_orders.customer_po', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('designs.description', 'like', '%' . $filters['search'] . '%')
                  ->orWhere('colourways.name', 'like', '%' . $filters['search'] . '%');
            });
        }
        
        return $query;
    }
    
    private function prepareSheetData($drops)
    {
        $data = [];
        
        // Header row - matching the exact order of the main table
        $headerRow = [
            'Drop#',
            'Season',
            'Factory',
            'Customer',
            'Order No',
            'Style',
            'Description',
            'Colour',
            'Total Qty',
            'Total Shipped',
            'Commission',
            'Value With Discount',
            'Value Without Discount',
            'First ExFty',
            'Revised ExFty',
            'Shipped',
            'RT Invoice'
        ];
        
        $data[] = $headerRow;
        
        // Add data rows
        foreach($drops as $drop) {
            $sizes = collect(json_decode($drop->sizes ?? '[]', true) ?: []);
            
            // Use the exact same calculations as the main table (lines 679-692 in CommissionShipmentSchedule.php)
            $totalQty = $sizes->sum('qty');
            $totalShipped = $sizes->sum('shipped_qty');
            
            // Commission calculation: (commission% / 100) * price * (shipped_qty if > 0, else qty)
            $commission = $sizes->map(function ($size) {
                $commission = $size['commission'] ?? 0;
                $price = $size['price'] ?? 0;
                $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                return $commission == 0 ? 0 : ($commission / 100) * $price * $qty;
            })->sum();
            
            // Value (this is "Value With Discount"): price * (shipped_qty if > 0, else qty)
            $value = $sizes->map(function ($size) {
                $price = $size['price'] ?? 0;
                $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                return $price * $qty;
            })->sum();
            
            // Value Discount (this is displayed as "Value Without Discount"): (discount% / 100) * price * (shipped_qty if > 0, else qty)
            $valueDiscount = $sizes->map(function ($size) {
                $discount = $size['discount'] ?? 0;
                $price = $size['price'] ?? 0;
                $qty = ($size['shipped_qty'] ?? 0) == 0 ? ($size['qty'] ?? 0) : ($size['shipped_qty'] ?? 0);
                return $discount == 0 ? 0 : ($discount / 100) * $price * $qty;
            })->sum();
            
            $currency = $drop->cust_currency ?? '';
            
            $row = [
                $drop->shipment_lines_id,
                $drop->season ?? '',
                $drop->factory ?? '',
                $drop->customer ?? '',
                $drop->customer_po ?? '',
                $drop->designs_id ?? '',
                $drop->description ?? '',
                $drop->colour ?? '',
                $totalQty,
                $totalShipped,
                $currency . number_format($commission, 2),
                $currency . number_format($value, 2),
                $currency . number_format($valueDiscount, 2),
                $drop->first_exfty ? date('Y-m-d', strtotime($drop->first_exfty)) : '',
                $drop->revised_exfty ? date('Y-m-d', strtotime($drop->revised_exfty)) : '',
                $drop->shipped ? 'Yes' : 'No',
                $drop->rt_invoice ?? ''
            ];
            
            $data[] = $row;
        }
        
        return $data;
    }
}

