<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
use App\Models\CommissionDrop;
use Illuminate\Support\Facades\Log;

class CommissionDropsExportController extends Controller
{
    public function exportToGoogleSheets(Request $request)
    {
        Gate::authorize('shipment:read');
        
        try {
            // Get filters from request
            $filters = $request->all();
            
            // Build query with filters
            $query = $this->buildExportQuery($filters);
            $drops = $query->get();
            
            // Initialize Google Client with user's OAuth token
            $user = auth()->user();
            
            // Check if user has connected their Google account
            if (!$user->google_access_token) {
                return response()->json([
                    'success' => false,
                    'error' => 'Please connect your Google account first',
                    'needs_auth' => true
                ], 401);
            }
            
            // Check if token is expired and refresh if needed
            if ($user->google_token_expires_at && $user->google_token_expires_at->isPast()) {
                $client = new \Google_Client();
                $client->setClientId(config('services.google.client_id'));
                $client->setClientSecret(config('services.google.client_secret'));
                $client->setAccessType('offline');
                $client->fetchAccessTokenWithRefreshToken($user->google_refresh_token);
                
                $newToken = $client->getAccessToken();
                $user->google_access_token = $newToken['access_token'];
                $user->google_token_expires_at = now()->addSeconds($newToken['expires_in']);
                $user->save();
            }
            
            $client = new \Google_Client();
            $client->setApplicationName('Commission Shipment Schedule Export');
            $client->setAccessToken($user->google_access_token);
            
            $service = new \Google_Service_Sheets($client);
            
            // Create spreadsheet
            $spreadsheet = new \Google_Service_Sheets_Spreadsheet([
                'properties' => [
                    'title' => 'Commission Shipment Schedule - ' . date('Y-m-d H:i:s')
                ]
            ]);
            
            $spreadsheet = $service->spreadsheets->create($spreadsheet);
            $spreadsheetId = $spreadsheet->spreadsheetId;
            
            // Prepare data
            $values = $this->prepareSheetData($drops);
            
            // Update spreadsheet with data
            $body = new \Google_Service_Sheets_ValueRange([
                'values' => $values
            ]);
            
            $params = [
                'valueInputOption' => 'RAW'
            ];
            
            $range = 'Sheet1!A1';
            $service->spreadsheets_values->update($spreadsheetId, $range, $body, $params);
            
            // Format header row
            $requests = [
                new \Google_Service_Sheets_Request([
                    'repeatCell' => [
                        'range' => [
                            'sheetId' => 0,
                            'startRowIndex' => 0,
                            'endRowIndex' => 1,
                        ],
                        'cell' => [
                            'userEnteredFormat' => [
                                'backgroundColor' => [
                                    'red' => 0.2,
                                    'green' => 0.2,
                                    'blue' => 0.2
                                ],
                                'textFormat' => [
                                    'foregroundColor' => [
                                        'red' => 1.0,
                                        'green' => 1.0,
                                        'blue' => 1.0
                                    ],
                                    'bold' => true
                                ]
                            ]
                        ],
                        'fields' => 'userEnteredFormat(backgroundColor,textFormat)'
                    ]
                ])
            ];
            
            $batchUpdateRequest = new \Google_Service_Sheets_BatchUpdateSpreadsheetRequest([
                'requests' => $requests
            ]);
            
            $service->spreadsheets->batchUpdate($spreadsheetId, $batchUpdateRequest);
            
            // Return URL - spreadsheet is already in user's Drive
            $url = "https://docs.google.com/spreadsheets/d/{$spreadsheetId}/edit";
            
            return response()->json([
                'success' => true,
                'url' => $url
            ]);
            
        } catch (\Exception $e) {
            Log::error('Commission Drops Google Sheets export failed: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    private function buildExportQuery($filters)
    {
        $query = CommissionDrop::query()
            ->with([
                'line.order.customer:id,name',
                'line.order.season:id,description',
                'line.colourway:id,name,img_thumb,style_versions_id',
                'line.colourway.style_versions:id,styles_id',
                'line.colourway.style_versions.styles:id,designs_id,customer_ref',
                'line.colourway.style_versions.styles.designs:id,description',
                'invoiceGroup:id,reference,status',
                'sizes.size:id,name,order',
            ])
            ->whereHas('line.order', function ($q) {
                $q->where('cancelled', false)
                  ->where('status', 'confirmed');
            })
            ->whereHas('line', function ($q) {
                $q->where('cancelled', false);
            });

        // Apply filters
        $query = $this->applyFilters($query, $filters);

        // Apply sorting
        $sortBy = $filters['sortBy'] ?? 'exfty';
        $sortDir = $filters['sortDir'] ?? 'asc';
        
        $query->orderBy($sortBy, $sortDir);
        
        return $query;
    }
    
    private function applyFilters($query, $filters)
    {
        // View filter
        $view = $filters['view'] ?? '';
        if ($view === 'unshipped') {
            $query->where('shipped', false);
        } elseif ($view === 'shipped') {
            $query->where('shipped', true);
        } elseif ($view === 'financeNC') {
            $query->where('invoiced', false);
        } elseif ($view === 'overdue') {
            $query->where('shipped', false)
                  ->whereNotNull('exfty')
                  ->where('exfty', '<', now()->toDateString());
        } elseif ($view === 'needsInvoice') {
            $query->where('shipped', true)
                  ->where('invoiced', false)
                  ->where(function ($q) {
                      $q->whereNull('rt_invoice')
                        ->orWhere('rt_invoice', '');
                  });
        }

        // Date range filter (skip for needsInvoice view)
        if ($view !== 'needsInvoice') {
            if (!empty($filters['fromDate'])) {
                $query->where('exfty', '>=', $filters['fromDate']);
            }
            if (!empty($filters['toDate'])) {
                $query->where('exfty', '<=', $filters['toDate']);
            }
        }

        // Customer filter
        if (!empty($filters['customers'])) {
            $query->whereHas('line.order', function ($q) use ($filters) {
                $q->whereIn('customers_id', $filters['customers']);
            });
        }

        // Season filter
        if (!empty($filters['seasons'])) {
            $query->whereHas('line.order', function ($q) use ($filters) {
                $q->whereIn('seasons_id', $filters['seasons']);
            });
        }

        // Invoice status filter
        if (!empty($filters['invoiceStatus'])) {
            if ($filters['invoiceStatus'] === 'has_invoice') {
                $query->whereNotNull('commission_invoice_groups_id');
            } elseif ($filters['invoiceStatus'] === 'no_invoice') {
                $query->whereNull('commission_invoice_groups_id');
            }
        }

        // Search
        if (!empty($filters['search'])) {
            $searchTerm = '%' . $filters['search'] . '%';
            $query->where(function ($q) use ($searchTerm) {
                $q->where('id', 'like', $searchTerm)
                  ->orWhereHas('line.order', function ($oq) use ($searchTerm) {
                      $oq->where('customer_po', 'like', $searchTerm);
                  })
                  ->orWhereHas('line.colourway.style_versions.styles', function ($sq) use ($searchTerm) {
                      $sq->where('customer_ref', 'like', $searchTerm);
                  })
                  ->orWhereHas('line.colourway.style_versions.styles.designs', function ($dq) use ($searchTerm) {
                      $dq->where('id', 'like', $searchTerm)
                         ->orWhere('description', 'like', $searchTerm);
                  })
                  ->orWhereHas('invoiceGroup', function ($iq) use ($searchTerm) {
                      $iq->where('reference', 'like', $searchTerm);
                  });
            });
        }
        
        return $query;
    }
    
    private function prepareSheetData($drops)
    {
        $data = [];
        
        // First pass: collect all unique sizes in order
        $allSizeNames = [];
        foreach ($drops as $drop) {
            foreach ($drop->sizes->sortBy('size.order') as $size) {
                $sizeName = $size->size?->name ?? '';
                if ($sizeName && !in_array($sizeName, $allSizeNames)) {
                    $allSizeNames[] = $sizeName;
                }
            }
        }
        
        // Header row
        $headerRow = [
            'Drop#',
            'Season',
            'Customer',
            'PO',
            'Style',
            'Customer Ref',
            'Description',
            'Colour',
        ];
        
        // Add size columns - ordered first, then shipped
        foreach ($allSizeNames as $sizeName) {
            $headerRow[] = $sizeName . ' (Ordered)';
        }
        foreach ($allSizeNames as $sizeName) {
            $headerRow[] = $sizeName . ' (Shipped)';
        }
        
        $headerRow = array_merge($headerRow, [
            'Total Qty',
            'Total Shipped',
            'Ex-Fty Date',
            'Shipped',
            'Shipped Date',
            'Invoice Group',
            'Finance Notes'
        ]);
        
        $data[] = $headerRow;
        
        // Add data rows
        foreach ($drops as $drop) {
            $order = $drop->line?->order;
            $colourway = $drop->line?->colourway;
            $style = $colourway?->style_versions?->styles;
            $design = $style?->designs;
            
            $totalQty = $drop->sizes->sum('qty');
            $totalShipped = $drop->sizes->sum('shipped_qty') ?: 0;
            
            $row = [
                $drop->id,
                $order?->season?->description ?? '',
                $order?->customer?->name ?? '',
                $order?->customer_po ?? '',
                $design?->id ? 'RT' . $design->id : '',
                $style?->customer_ref ?? '',
                $design?->description ?? '',
                $colourway?->name ?? '',
            ];
            
            // Create a map of size name to quantities
            $sizeMap = [];
            foreach ($drop->sizes as $size) {
                $sizeName = $size->size?->name ?? '';
                if ($sizeName) {
                    $sizeMap[$sizeName] = [
                        'qty' => $size->qty ?? 0,
                        'shipped_qty' => $size->shipped_qty ?? 0
                    ];
                }
            }
            
            // Add ordered quantities for each size
            foreach ($allSizeNames as $sizeName) {
                $row[] = $sizeMap[$sizeName]['qty'] ?? 0;
            }
            
            // Add shipped quantities for each size
            foreach ($allSizeNames as $sizeName) {
                $row[] = $sizeMap[$sizeName]['shipped_qty'] ?? 0;
            }
            
            $row = array_merge($row, [
                $totalQty,
                $totalShipped,
                $drop->exfty ? $drop->exfty->format('Y-m-d') : '',
                $drop->shipped ? 'Yes' : 'No',
                $drop->shipped_date ? $drop->shipped_date->format('Y-m-d') : '',
                $drop->invoiceGroup?->reference ?? '',
                $drop->fn_notes ?? ''
            ]);
            
            $data[] = $row;
        }
        
        return $data;
    }
}






