<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;
use PhpOffice\PhpSpreadsheet\IOFactory;
use PhpOffice\PhpSpreadsheet\Worksheet\Table as ExcelTable;

class CashflowExportController extends Controller
{
    private function colLetter(int $index): string { // 1-based
        $str = '';
        while ($index > 0) {
            $index--; $str = chr(65 + ($index % 26)) . $str; $index = intdiv($index, 26);
        }
        return $str;
    }
    public function exportPivot(Request $request): Response
    {
        @set_time_limit(300);
        @ini_set('memory_limit', '1024M');
        $rows = json_decode($request->get('rows', '[]'), true) ?: [];
        $measures = json_decode($request->get('measures', '[]'), true) ?: [];
        $factory = $request->get('factory');
        $customer = $request->get('customer');
        // Accept data via POST body (safer for size); fallback to GET param
        $payload = $request->input('data');
        if (is_string($payload)) {
            $data = json_decode($payload, true);
        } else if (is_array($payload)) {
            $data = $payload;
        } else {
            $data = json_decode($request->query('data', '[]'), true);
        }

        // If data not passed, try to reconstruct from session or failover
        if (!is_array($data) || empty($data)) {
            // As a fallback, we cannot re-query without context; require data param.
            return response()->streamDownload(function(){}, 'pivot.xlsx');
        }

        // Apply filters similar to client
        $data = array_values(array_filter($data, function($r) use ($factory, $customer){
            if ($factory && ($r['Factory'] ?? null) != $factory) return false;
            if ($customer && ($r['Customer'] ?? null) != $customer) return false;
            return true;
        }));

        // If a native Excel Pivot template exists, populate it
        $template = resource_path('excel_templates/cashflow_pivot_template.xlsx');
        if (is_file($template)) {
            // Direct OpenXML edit to preserve native PivotTable definitions
            $tmp = tempnam(sys_get_temp_dir(), 'pivot_tpl_');
            copy($template, $tmp);
            $zip = new \ZipArchive();
            if ($zip->open($tmp) !== true) {
                return response('Template open failed', 500);
            }

            // Build a case-insensitive index of entries to robustly locate parts
            $zipIndex = [];
            for ($i = 0; $i < $zip->numFiles; $i++) {
                $n = $zip->getNameIndex($i);
                if ($n !== false) { $zipIndex[strtolower(ltrim($n, '/'))] = $i; }
            }
            $get = function(string $name) use ($zip, $zipIndex) {
                $key = strtolower(ltrim($name, '/'));
                if (isset($zipIndex[$key])) {
                    return $zip->getFromIndex($zipIndex[$key]);
                }
                return null;
            };
            $put = function(string $name, string $contents) use ($zip) {
                $zip->addFromString($name, $contents);
            };

            $workbookXml = $get('xl/workbook.xml');
            $relsXml = $get('xl/_rels/workbook.xml.rels');
            if (!$workbookXml || !$relsXml) {
                $zip->close();
                $missing = [];
                if (!$workbookXml) { $missing[] = 'xl/workbook.xml'; }
                if (!$relsXml) { $missing[] = 'xl/_rels/workbook.xml.rels'; }
                return response('Invalid template: missing '.implode(', ', $missing), 422);
            }

            $domWb = new \DOMDocument();
            $domWb->preserveWhiteSpace = false; $domWb->loadXML($workbookXml);
            $xpWb = new \DOMXPath($domWb);
            $xpWb->registerNamespace('m', 'http://schemas.openxmlformats.org/spreadsheetml/2006/main');
            $xpWb->registerNamespace('r', 'http://schemas.openxmlformats.org/officeDocument/2006/relationships');
            $sheetNodes = $xpWb->query('//m:sheets/m:sheet');
            $dataRid = null;
            foreach ($sheetNodes as $s) {
                if ($s->getAttribute('name') === 'Data') { $dataRid = $s->getAttributeNS('http://schemas.openxmlformats.org/officeDocument/2006/relationships','id'); break; }
            }
            if (!$dataRid) { $zip->close(); return response('Data sheet not found in template', 422); }

            $domRel = new \DOMDocument();
            $domRel->preserveWhiteSpace = false; $domRel->loadXML($relsXml);
            $xpRel = new \DOMXPath($domRel);
            $xpRel->registerNamespace('r', 'http://schemas.openxmlformats.org/package/2006/relationships');
            $relNodes = $domRel->getElementsByTagName('Relationship');
            $target = null;
            foreach ($relNodes as $rel) { if ($rel->getAttribute('Id') === $dataRid) { $target = $rel->getAttribute('Target'); break; } }
            if (!$target) { $zip->close(); return response('Data sheet rel missing', 422); }
            $sheetPath = 'xl/' . ltrim($target, './');

            $sheetXml = $get($sheetPath);
            if (!$sheetXml) { $zip->close(); return response('Data sheet xml missing', 422); }
            $domSheet = new \DOMDocument();
            $domSheet->preserveWhiteSpace = false; $domSheet->loadXML($sheetXml);
            $xpSheet = new \DOMXPath($domSheet);
            $xpSheet->registerNamespace('m', 'http://schemas.openxmlformats.org/spreadsheetml/2006/main');
            $sheetDataNode = $xpSheet->query('//m:sheetData')->item(0);
            if (!$sheetDataNode) { $zip->close(); return response('sheetData missing', 422); }

            // Determine headers from first row if present else from data keys
            $headers = [];
            $row1 = $xpSheet->query('//m:sheetData/m:row[@r="1"]/m:c');
            if ($row1 && $row1->length) {
                foreach ($row1 as $c) {
                    $t = $c->getAttribute('t');
                    $text = '';
                    if ($t === 'inlineStr') {
                        $tn = $xpSheet->query('m:is/m:t', $c)->item(0);
                        $text = $tn ? $tn->nodeValue : '';
                    } else {
                        $vn = $xpSheet->query('m:v', $c)->item(0);
                        $text = $vn ? $vn->nodeValue : '';
                    }
                    if ($text === '') break; $headers[] = $text;
                }
            }
            if (empty($headers)) { $headers = array_keys($data[0] ?? ['Money In Week' => null]); }
            else {
                // If headers are placeholder numbers (1..N) or don't match data keys, use data keys
                $firstRowKeys = array_keys($data[0] ?? []);
                $numericSeq = true;
                for ($i = 0; $i < count($headers); $i++) {
                    if ((string)($i + 1) !== (string)$headers[$i]) { $numericSeq = false; break; }
                }
                $mismatch = false;
                foreach ($headers as $h) { if (!in_array($h, $firstRowKeys, true)) { $mismatch = true; break; } }
                if ($numericSeq || $mismatch) { $headers = $firstRowKeys; }
            }

            // Fast path: build sheetData XML string and inject into worksheet XML
            $sheetXml = $get($sheetPath);
            $colLetters = [];
            for ($i = 0; $i < max(1, count($headers)); $i++) { $colLetters[$i] = $this->colLetter($i + 1); }
            $esc = function ($s) { return htmlspecialchars((string)$s, ENT_XML1 | ENT_COMPAT, 'UTF-8'); };
            $sheetDataXml = '<sheetData>';
            $sheetDataXml .= '<row r="1">';
            for ($i = 0; $i < max(1, count($headers)); $i++) {
                $h = $headers[$i] ?? ('Column'.($i+1));
                $sheetDataXml .= '<c r="'.$colLetters[$i].'1" t="inlineStr"><is><t>'.$esc($h).'</t></is></c>';
            }
            $sheetDataXml .= '</row>';
            $rowIndex = 2;
            foreach ($data as $row) {
                $sheetDataXml .= '<row r="'.$rowIndex.'">';
                for ($i = 0; $i < max(1, count($headers)); $i++) {
                    $key = $headers[$i] ?? ('Column'.($i+1));
                    $val = $row[$key] ?? '';
                    if ($val === '' || $val === null) { continue; }
                    $refCell = $colLetters[$i].$rowIndex;
                    if (is_numeric($val)) {
                        $sheetDataXml .= '<c r="'.$refCell.'"><v>'.((string)$val).'</v></c>';
                    } else {
                        $sheetDataXml .= '<c r="'.$refCell.'" t="inlineStr"><is><t>'.$esc($val).'</t></is></c>';
                    }
                }
                $sheetDataXml .= '</row>';
                $rowIndex++;
            }
            $sheetDataXml .= '</sheetData>';
            $lastCol = $this->colLetter(max(1, count($headers))); $lastRow = max(1, count($data) + 1);
            $ref = 'A1:' . $lastCol . $lastRow;
            // Replace dimension
            if (preg_match('/<dimension[^>]*ref="[^"]*"[^\/>]*\/>/i', $sheetXml)) {
                $sheetXml = preg_replace('/(<dimension[^>]*ref=")[^"]*("[^\/>]*\/>)/i', '$1'.$ref.'$2', $sheetXml, 1);
            } else {
                // Insert dimension right after <worksheet ...>
                $sheetXml = preg_replace('/(<worksheet[^>]*>)/i', '$1' . '<dimension ref="'.$ref.'"/>', $sheetXml, 1);
            }
            // Replace sheetData block
            $sheetXml = preg_replace('/<sheetData[\s\S]*?<\/sheetData>/i', $sheetDataXml, $sheetXml, 1);
            $put($sheetPath, $sheetXml);

            // Update DataTbl table ref to cover new range
            $sheetRelsPath = preg_replace('/worksheets\//', 'worksheets/_rels/', $sheetPath) . '.rels';
            $sheetRels = $get($sheetRelsPath);
            if ($sheetRels) {
                $domSR = new \DOMDocument(); $domSR->loadXML($sheetRels);
                $rels = $domSR->getElementsByTagName('Relationship');
                // Prefer a table whose displayName is DataTbl; fallback to first table
                $tableTargets = [];
                foreach ($rels as $rel) {
                    if ($rel->getAttribute('Type') === 'http://schemas.openxmlformats.org/officeDocument/2006/relationships/table') {
                        $tableTargets[] = $rel->getAttribute('Target');
                    }
                }
                $chosenPath = null;
                foreach ($tableTargets as $tgt) {
                    $p = 'xl/' . ltrim($tgt, './');
                    $tx = $get($p);
                    if ($tx) {
                        $d = new \DOMDocument(); $d->loadXML($tx);
                        $disp = $d->documentElement->getAttribute('displayName');
                        if (strcasecmp($disp, 'DataTbl') === 0) { $chosenPath = $p; break; }
                        if ($chosenPath === null) { $chosenPath = $p; }
                    }
                }
                if ($chosenPath) {
                    $txml = $get($chosenPath);
                    if ($txml) {
                        $domT = new \DOMDocument(); $domT->loadXML($txml);
                        $lastCol = $this->colLetter(max(1, count($headers))); $lastRow = max(1, count($data) + 1);
                        $ref = 'A1:' . $lastCol . $lastRow;
                        $tableEl = $domT->documentElement;
                        $tableEl->setAttribute('ref', $ref);
                        // Ensure headerRowCount and displayName are set
                        $tableEl->setAttribute('headerRowCount', '1');
                        if (!$tableEl->hasAttribute('displayName') || strcasecmp($tableEl->getAttribute('displayName'), 'DataTbl') !== 0) {
                            $tableEl->setAttribute('displayName', 'DataTbl');
                            $tableEl->setAttribute('name', 'DataTbl');
                        }
                        // Synchronize tableColumns with headers
                        $xpT = new \DOMXPath($domT);
                        $xpT->registerNamespace('m', 'http://schemas.openxmlformats.org/spreadsheetml/2006/main');
                        $colsNode = $xpT->query('//m:tableColumns')->item(0);
                        if (!$colsNode) {
                            $colsNode = $domT->createElementNS('http://schemas.openxmlformats.org/spreadsheetml/2006/main', 'tableColumns');
                            $tableEl->appendChild($colsNode);
                        }
                        while ($colsNode->firstChild) { $colsNode->removeChild($colsNode->firstChild); }
                        $colsNode->setAttribute('count', (string)max(1, count($headers)));
                        $debugColNames = [];
                        for ($i = 0; $i < max(1, count($headers)); $i++) {
                            $colEl = $domT->createElementNS('http://schemas.openxmlformats.org/spreadsheetml/2006/main', 'tableColumn');
                            $colEl->setAttribute('id', (string)($i + 1));
                            $colEl->setAttribute('name', (string)($headers[$i] ?? 'Column'.($i+1)));
                            $colsNode->appendChild($colEl);
                            if ($i < 5) { $debugColNames[] = (string)($headers[$i] ?? ('Column'.($i+1))); }
                        }
                        $autoFilters = $domT->getElementsByTagName('autoFilter');
                        if ($autoFilters->length) { $autoFilters->item(0)->setAttribute('ref', $ref); }
                        $put($chosenPath, $domT->saveXML());
                    }
                }
            }

            $zip->close();
            return response()->download($tmp, 'pivot.xlsx', [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            ])->deleteFileAfterSend(true);
        }

        // Otherwise, generate a computed summary workbook (non-pivot fallback)
        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Data');

        // Build header from union of keys
        $headers = [];
        foreach ($data as $row) {
            foreach ($row as $k => $v) {
                if (!in_array($k, $headers, true)) $headers[] = $k;
            }
        }
        if (empty($headers)) $headers = ['Money In Week'];

        // Write headers
        foreach ($headers as $i => $h) {
            $sheet->setCellValueByColumnAndRow($i + 1, 1, $h);
        }
        // Write data
        $r = 2;
        foreach ($data as $row) {
            foreach ($headers as $i => $h) {
                $sheet->setCellValueByColumnAndRow($i + 1, $r, $row[$h] ?? null);
            }
            $r++;
        }

        // Build a computed Summary sheet (grouped totals) instead of a native Pivot
        $summarySheet = new Worksheet($spreadsheet, 'Summary');
        $spreadsheet->addSheet($summarySheet, 1);

        // Group by row dimensions and sum measures
        $groupTotals = [];
        $measureKeys = array_values(array_map(function($m){ return $m['uniqueName'] ?? ''; }, $measures));
        foreach ($data as $row) {
            $keyParts = [];
            foreach ($rows as $dim) { $keyParts[] = (string)($row[$dim] ?? ''); }
            $key = implode('\u0001', $keyParts);
            if (!isset($groupTotals[$key])) {
                $groupTotals[$key] = ['__dims' => $keyParts];
                foreach ($measureKeys as $mk) { $groupTotals[$key][$mk] = 0; }
            }
            foreach ($measureKeys as $mk) { $groupTotals[$key][$mk] += (float)($row[$mk] ?? 0); }
        }

        // Write header
        $c = 1; $rIdx = 1;
        foreach ($rows as $dim) { $summarySheet->setCellValueByColumnAndRow($c++, $rIdx, $dim); }
        foreach ($measureKeys as $mk) { $summarySheet->setCellValueByColumnAndRow($c++, $rIdx, $mk); }

        // Write rows
        $rowNum = 2;
        foreach ($groupTotals as $g) {
            $c = 1;
            foreach ($g['__dims'] as $dimVal) { $summarySheet->setCellValueByColumnAndRow($c++, $rowNum, $dimVal); }
            foreach ($measureKeys as $mk) { $summarySheet->setCellValueByColumnAndRow($c++, $rowNum, $g[$mk]); }
            $rowNum++;
        }

        // Auto-size columns
        $totalCols = count($rows) + count($measureKeys);
        for ($ci = 1; $ci <= $totalCols; $ci++) { $summarySheet->getColumnDimensionByColumn($ci)->setAutoSize(true); }
        // Freeze header and add autofilter
        $summarySheet->freezePane('A2');
        $summarySheet->setAutoFilterByColumnAndRow(1, 1, $totalCols, max(1, $rowNum-1));

        // Basic number formatting for measures (detect currency symbol by header name)
        for ($i = 0; $i < count($measureKeys); $i++) {
            $colIndex = count($rows) + 1 + $i; // 1-based
            $colLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($colIndex);
            $title = $measureKeys[$i];
            $range = $colLetter . '2:' . $colLetter . ($rowNum-1);
            $fmt = '#,##0.00';
            if (stripos($title, 'gbp') !== false || stripos($title, 'base') !== false) { $fmt = '"£"#,##0.00'; }
            if (stripos($title, 'usd') !== false) { $fmt = '"$"#,##0.00'; }
            if (stripos($title, 'euro') !== false || stripos($title, 'eur') !== false) { $fmt = '"€"#,##0.00'; }
            $summarySheet->getStyle($range)->getNumberFormat()->setFormatCode($fmt);
        }

        return response()->streamDownload(function () use ($spreadsheet) {
            $writer = new Xlsx($spreadsheet);
            $writer->save('php://output');
        }, 'pivot.xlsx', [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'Cache-Control' => 'max-age=0',
        ]);
    }
}


