<?php

namespace App\Console\Commands;

use App\Models\ShipmentLine;
use App\Services\TotalCacheService;
use Illuminate\Console\Command;

class WarmupAllTotalsCache extends Command
{
    protected $signature = 'cache:warmup-totals 
                            {--chunk-size=100 : Number of records to process per chunk}
                            {--only-zero : Only update lines with zero or missing cache}';

    protected $description = 'Warm up totals cache for all shipment lines';

    public function handle(TotalCacheService $totalCacheService)
    {
        $this->info('Starting totals cache warmup...');
        
        $chunkSize = (int) $this->option('chunk-size');
        $onlyZero = $this->option('only-zero');
        
        // Build query
        $query = ShipmentLine::query()->with([
            'customer_order_lines.customer_order_line_quantities.sizes',
            'shipment_line_sizes'
        ]);
        
        if ($onlyZero) {
            $this->info('Mode: Only updating lines with zero or missing cache');
            // Get IDs of lines with zero or missing cache
            $zeroIds = \DB::table('shipment_lines')
                ->leftJoin('total_cache', function ($join) {
                    $join->on('total_cache.entity_id', '=', 'shipment_lines.id')
                         ->where('total_cache.entity_type', '=', 'shipment_line')
                         ->where('total_cache.cache_key', '=', 'prices');
                })
                ->whereNull('shipment_lines.deleted_at')
                ->where(function ($q) {
                    $q->whereNull('total_cache.id')
                      ->orWhereRaw("JSON_EXTRACT(total_cache.cached_data, '$.subtotal_base') = 0")
                      ->orWhereRaw("JSON_EXTRACT(total_cache.cached_data, '$.subtotal_base') IS NULL");
                })
                ->pluck('shipment_lines.id');
            
            $query->whereIn('id', $zeroIds);
        } else {
            $this->info('Mode: Updating all shipment lines');
        }
        
        $total = $query->count();
        $this->info("Total shipment lines to process: {$total}");
        $this->newLine();
        
        if (!$this->confirm('Continue?', true)) {
            return 0;
        }
        
        $progressBar = $this->output->createProgressBar($total);
        $progressBar->start();
        
        $processed = 0;
        $updated = 0;
        $errors = 0;
        
        $query->chunk($chunkSize, function ($shipmentLines) use ($totalCacheService, &$processed, &$updated, &$errors, $progressBar) {
            foreach ($shipmentLines as $shipmentLine) {
                try {
                    $totalCacheService->warmupShipmentLine($shipmentLine);
                    $updated++;
                } catch (\Exception $e) {
                    $errors++;
                    // Silently continue - don't spam with log errors
                }
                
                $processed++;
                $progressBar->advance();
            }
        });
        
        $progressBar->finish();
        $this->newLine(2);
        
        $this->info("Warmup completed!");
        $this->table(
            ['Metric', 'Count'],
            [
                ['Processed', $processed],
                ['Successfully Updated', $updated],
                ['Errors', $errors],
            ]
        );
        
        return 0;
    }
}

