<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Jobs\WarmupTotalCacheJob;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;

class WarmupAllTotalCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'total-cache:warmup 
                            {--entity=shipment_line : Entity type to warm up (shipment_line, customer_order_line, customer_order, shipment, all)}
                            {--limit= : Limit number of entities to process}
                            {--chunk=500 : Number of entities to process per batch}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Dispatch warmup jobs for total cache to populate all totals in the background';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $entityType = $this->option('entity');
        $limit = $this->option('limit');
        $chunkSize = $this->option('chunk');
        
        $this->info("Starting total cache warmup for: {$entityType}");
        $this->newLine();

        if ($entityType === 'all') {
            $this->warmupAll($limit, $chunkSize);
        } else {
            $this->warmupEntity($entityType, $limit, $chunkSize);
        }

        $this->newLine();
        $this->info('✅ All warmup jobs dispatched!');
        $this->info('Monitor queue: php artisan queue:work');
        
        return 0;
    }

    /**
     * Warm up all entity types.
     */
    private function warmupAll(?int $limit, int $chunkSize): void
    {
        $this->warmupEntity('shipment_line', $limit, $chunkSize);
        $this->newLine();
        $this->warmupEntity('customer_order_line', $limit, $chunkSize);
        $this->newLine();
        $this->warmupEntity('customer_order', $limit, $chunkSize);
        $this->newLine();
        $this->warmupEntity('shipment', $limit, $chunkSize);
    }

    /**
     * Warm up a specific entity type.
     */
    private function warmupEntity(string $entityType, ?int $limit, int $chunkSize): void
    {
        $query = match($entityType) {
            'shipment_line' => ShipmentLine::query(),
            'customer_order_line' => CustomerOrderLines::query(),
            'customer_order' => CustomerOrders::query(),
            'shipment' => Shipment::query(),
            default => null,
        };

        if (!$query) {
            $this->error("Unknown entity type: {$entityType}");
            return;
        }

        if ($limit) {
            $query->limit($limit);
        }

        $total = $query->count();
        $this->info("Processing {$total} {$entityType} records...");

        $bar = $this->output->createProgressBar($total);
        $bar->start();

        $dispatched = 0;

        $query->chunk($chunkSize, function($entities) use ($entityType, &$dispatched, $bar) {
            foreach ($entities as $entity) {
                WarmupTotalCacheJob::dispatch($entityType, $entity->id);
                $dispatched++;
                $bar->advance();
            }
        });

        $bar->finish();
        $this->newLine();
        $this->info("✓ Dispatched {$dispatched} jobs for {$entityType}");
    }
}



















