<?php

namespace App\Console\Commands;

use App\Models\ShipmentLine;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class RevertExftyChanges extends Command
{
    protected $signature = 'shipment:revert-exfty 
                            {--user-id= : User ID who made the changes}
                            {--from= : Start datetime (Y-m-d H:i:s)}
                            {--to= : End datetime (Y-m-d H:i:s)}
                            {--new-exfty= : Only revert changes where exfty was changed TO this value}
                            {--dry-run : Preview changes without applying them}';

    protected $description = 'Revert accidental exfty date changes using audit logs';

    public function handle()
    {
        $userId = $this->option('user-id');
        $from = $this->option('from');
        $to = $this->option('to');
        $newExfty = $this->option('new-exfty');
        $dryRun = $this->option('dry-run');

        if (!$userId || !$from || !$to) {
            $this->error('Please provide --user-id, --from, and --to options');
            return 1;
        }

        $this->info("Fetching exfty changes from audit logs...");
        $this->info("User ID: {$userId}");
        $this->info("Time range: {$from} to {$to}");
        if ($newExfty) {
            $this->info("Filter: Only reverting changes TO {$newExfty}");
        }
        $this->info("Mode: " . ($dryRun ? 'DRY RUN (preview only)' : 'LIVE'));
        $this->newLine();

        // Fetch the audit records for exfty changes
        $query = DB::table('audits')
            ->where('user_id', $userId)
            ->where('auditable_type', 'App\Models\ShipmentLine')
            ->where('event', 'updated')
            ->whereRaw("JSON_CONTAINS_PATH(old_values, 'one', '$.exfty')")
            ->whereBetween('created_at', [$from, $to]);

        if ($newExfty) {
            $query->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(new_values, '$.exfty')) = ?", [$newExfty]);
        }

        $audits = $query->orderBy('created_at', 'asc')
            ->get(['id', 'auditable_id', 'old_values', 'new_values', 'created_at']);

        if ($audits->isEmpty()) {
            $this->warn('No audit records found matching the criteria.');
            return 0;
        }

        $this->info("Found {$audits->count()} records to revert.");
        $this->newLine();

        // Build a table preview
        $previewData = $audits->take(20)->map(function ($audit) {
            $old = json_decode($audit->old_values, true);
            $new = json_decode($audit->new_values, true);
            return [
                'Shipment Line ID' => $audit->auditable_id,
                'Current (Wrong)' => $new['exfty'] ?? 'null',
                'Restore To' => $old['exfty'] ?? 'null',
                'Changed At' => $audit->created_at,
            ];
        })->toArray();

        $this->table(['Shipment Line ID', 'Current (Wrong)', 'Restore To', 'Changed At'], $previewData);

        if ($audits->count() > 20) {
            $this->info("... and " . ($audits->count() - 20) . " more records");
        }
        $this->newLine();

        if ($dryRun) {
            $this->warn('DRY RUN complete. No changes were made.');
            $this->info('Run without --dry-run to apply the changes.');
            return 0;
        }

        if (!$this->confirm('Do you want to proceed with reverting these changes?')) {
            $this->info('Operation cancelled.');
            return 0;
        }

        $this->info('Reverting changes...');

        $reverted = 0;
        $failed = 0;

        DB::beginTransaction();

        try {
            $bar = $this->output->createProgressBar($audits->count());
            $bar->start();

            foreach ($audits as $audit) {
                $old = json_decode($audit->old_values, true);
                $oldExfty = $old['exfty'] ?? null;

                if ($oldExfty === null) {
                    $failed++;
                    $bar->advance();
                    continue;
                }

                // Use direct DB update to avoid triggering model events that would create more audit logs
                // However, we still need the cache invalidation, so we'll use the model but skip auditing
                $shipmentLine = ShipmentLine::withoutEvents(function () use ($audit, $oldExfty) {
                    $line = ShipmentLine::find($audit->auditable_id);
                    if ($line) {
                        // Temporarily disable auditing for this update
                        $line->disableAuditing();
                        $line->exfty = $oldExfty;
                        $line->save();
                        return $line;
                    }
                    return null;
                });

                if ($shipmentLine) {
                    $reverted++;
                } else {
                    $failed++;
                }

                $bar->advance();
            }

            $bar->finish();
            $this->newLine(2);

            DB::commit();

            $this->info("Successfully reverted {$reverted} records.");
            if ($failed > 0) {
                $this->warn("Failed to revert {$failed} records (shipment lines may have been deleted).");
            }

            // Clear relevant caches
            $this->info('Clearing caches...');
            foreach ($audits as $audit) {
                \Illuminate\Support\Facades\Cache::forget("shipment_line:{$audit->auditable_id}:first_exfty");
            }
            $this->info('Caches cleared.');

            return 0;
        } catch (\Exception $e) {
            DB::rollBack();
            $this->error('Error during revert: ' . $e->getMessage());
            $this->error('All changes have been rolled back.');
            return 1;
        }
    }
}

