<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\CommissionImportBatchFile;
use App\Jobs\ProcessCommissionFile;
use Illuminate\Support\Facades\Log;

class RequeueProcessingCommissionFiles extends Command
{
    protected $signature = 'commission:requeue-processing {--batch=}';
    protected $description = 'Reset files stuck in processing back to pending and re-dispatch jobs. If --batch is provided, limits to that batch.';

    public function handle(): int
    {
        $batchId = $this->option('batch');

        $query = CommissionImportBatchFile::query()->where('status', 'processing');
        if (!empty($batchId)) {
            $query->where('batch_id', (int)$batchId);
        }

        $files = $query->get();
        $total = $files->count();
        $requeued = 0;
        $fixedToExtracted = 0;

        $this->info("Found {$total} file(s) in processing state" . ($batchId ? " for batch {$batchId}" : '') . '.');

        foreach ($files as $file) {
            $file->refresh();
            // If data exists and totals present, mark extracted
            $hasData = !empty($file->extracted_data);
            $hasLines = (int)($file->total_lines ?? 0) > 0;

            if ($hasData && $hasLines) {
                $file->update(['status' => 'extracted']);
                $fixedToExtracted++;
                $this->line(" - File {$file->id} marked as extracted (data present)");
                continue;
            }

            // Otherwise, reset to pending and re-dispatch
            $metadata = $file->metadata ?? [];
            $metadata['is_retry'] = true;

            $file->update([
                'status' => 'pending',
                'metadata' => $metadata,
            ]);

            ProcessCommissionFile::dispatch($file);
            $requeued++;
            $this->line(" - File {$file->id} requeued");
        }

        Log::info('commission:requeue-processing completed', [
            'batch_id' => $batchId,
            'found' => $total,
            'requeued' => $requeued,
            'fixed_to_extracted' => $fixedToExtracted,
        ]);

        $this->info("Requeued: {$requeued}, Marked extracted: {$fixedToExtracted}");

        return Command::SUCCESS;
    }
}











































