<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\TotalCache;
use App\Services\TotalCacheService;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;
use App\Models\Shipment;
use Illuminate\Support\Facades\Log;

class RefreshTotalCache extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cache:refresh-totals {--dry-run : Show what would be done without actually doing it}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Clear all total caches and rebuild them fresh from price resolutions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        // Increase execution time limit for this command
        set_time_limit(0); // 0 = no time limit
        ini_set('memory_limit', '2G'); // Also increase memory limit if needed
        
        $dryRun = $this->option('dry-run');
        $totalCacheService = app(TotalCacheService::class);
        
        if ($dryRun) {
            $this->info('DRY RUN MODE - No changes will be made');
        }
        
        // Step 1: Clear all existing total caches
        $this->info('Step 1: Clearing all existing total caches...');
        $existingCount = TotalCache::count();
        $this->line("Found {$existingCount} existing total cache records");
        
        if (!$dryRun) {
            TotalCache::truncate();
            $this->info('✓ All total cache records cleared');
        } else {
            $this->line('Would clear all total cache records');
        }
        
        // Step 2: Rebuild customer order line caches
        $this->info('Step 2: Rebuilding customer order line caches...');
        $orderLineCount = CustomerOrderLines::count();
        $this->line("Found {$orderLineCount} customer order lines to process");
        
        $processedOrderLines = 0;
        $errorCount = 0;
        
        CustomerOrderLines::chunk(100, function ($orderLines) use ($totalCacheService, $dryRun, &$processedOrderLines, &$errorCount) {
            foreach ($orderLines as $orderLine) {
                try {
                    if (!$dryRun) {
                        $totalCacheService->warmupCustomerOrderLine($orderLine);
                    }
                    $processedOrderLines++;
                    
                    if ($processedOrderLines % 100 == 0) {
                        $this->line("Processed {$processedOrderLines} customer order lines...");
                    }
                } catch (\Exception $e) {
                    $errorCount++;
                    $this->error("Failed to process customer order line {$orderLine->id}: " . $e->getMessage());
                }
            }
        });
        
        $this->info("✓ Processed {$processedOrderLines} customer order lines, {$errorCount} errors");
        
        // Step 3: Rebuild customer order caches
        $this->info('Step 3: Rebuilding customer order caches...');
        $orderCount = CustomerOrders::count();
        $this->line("Found {$orderCount} customer orders to process");
        
        $processedOrders = 0;
        $orderErrorCount = 0;
        
        CustomerOrders::chunk(100, function ($orders) use ($totalCacheService, $dryRun, &$processedOrders, &$orderErrorCount) {
            foreach ($orders as $order) {
                try {
                    if (!$dryRun) {
                        $totalCacheService->warmupCustomerOrder($order);
                    }
                    $processedOrders++;
                    
                    if ($processedOrders % 100 == 0) {
                        $this->line("Processed {$processedOrders} customer orders...");
                    }
                } catch (\Exception $e) {
                    $orderErrorCount++;
                    $this->error("Failed to process customer order {$order->id}: " . $e->getMessage());
                }
            }
        });
        
        $this->info("✓ Processed {$processedOrders} customer orders, {$orderErrorCount} errors");
        
        // Step 4: Rebuild shipment line caches
        $this->info('Step 4: Rebuilding shipment line caches...');
        $shipmentLineCount = ShipmentLine::count();
        $this->line("Found {$shipmentLineCount} shipment lines to process");
        
        $processedShipmentLines = 0;
        $shipmentLineErrorCount = 0;
        
        ShipmentLine::chunk(100, function ($shipmentLines) use ($totalCacheService, $dryRun, &$processedShipmentLines, &$shipmentLineErrorCount) {
            foreach ($shipmentLines as $shipmentLine) {
                try {
                    if (!$dryRun) {
                        $totalCacheService->warmupShipmentLine($shipmentLine);
                    }
                    $processedShipmentLines++;
                    
                    if ($processedShipmentLines % 100 == 0) {
                        $this->line("Processed {$processedShipmentLines} shipment lines...");
                    }
                } catch (\Exception $e) {
                    $shipmentLineErrorCount++;
                    $this->error("Failed to process shipment line {$shipmentLine->id}: " . $e->getMessage());
                }
            }
        });
        
        $this->info("✓ Processed {$processedShipmentLines} shipment lines, {$shipmentLineErrorCount} errors");
        
        // Step 5: Rebuild shipment caches
        $this->info('Step 5: Rebuilding shipment caches...');
        $shipmentCount = Shipment::count();
        $this->line("Found {$shipmentCount} shipments to process");
        
        $processedShipments = 0;
        $shipmentErrorCount = 0;
        
        Shipment::chunk(100, function ($shipments) use ($totalCacheService, $dryRun, &$processedShipments, &$shipmentErrorCount) {
            foreach ($shipments as $shipment) {
                try {
                    if (!$dryRun) {
                        $totalCacheService->warmupShipment($shipment);
                    }
                    $processedShipments++;
                    
                    if ($processedShipments % 100 == 0) {
                        $this->line("Processed {$processedShipments} shipments...");
                    }
                } catch (\Exception $e) {
                    $shipmentErrorCount++;
                    $this->error("Failed to process shipment {$shipment->id}: " . $e->getMessage());
                }
            }
        });
        
        $this->info("✓ Processed {$processedShipments} shipments, {$shipmentErrorCount} errors");
        
        // Step 6: Summary
        $this->info('Step 6: Summary');
        
        if (!$dryRun) {
            $newCacheCount = TotalCache::count();
            $this->info("✓ Total cache refresh complete!");
            $this->line("Old cache records: {$existingCount}");
            $this->line("New cache records: {$newCacheCount}");
            
            // Show some sample records to verify cmt_base is included
            $this->info('Sample records with cmt_base:');
            $samples = TotalCache::whereIn('entity_type', ['customer_order_line', 'shipment_line'])
                ->limit(3)
                ->get(['id', 'entity_type', 'entity_id', 'cached_data']);
                
            foreach ($samples as $sample) {
                $data = $sample->cached_data;
                $cmtBase = $data['cmt_base'] ?? 'missing';
                $cmt = $data['cmt'] ?? 'missing';
                $this->line("Record {$sample->id} ({$sample->entity_type}): cmt={$cmt}, cmt_base={$cmtBase}");
            }
        } else {
            $this->info('DRY RUN COMPLETE - No changes were made');
        }
        
        $totalErrors = $errorCount + $orderErrorCount + $shipmentLineErrorCount + $shipmentErrorCount;
        
        if ($totalErrors > 0) {
            $this->warn("Total errors: {$totalErrors}");
            return Command::FAILURE;
        }
        
        return Command::SUCCESS;
    }
}
