<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\PriceResolutionService;
use App\Services\TotalCacheService;
use App\Models\Colourways;
use App\Models\ShipmentLine;
use App\Models\CustomerOrderLines;
use App\Models\CustomerOrders;

class RebuildAllCaches extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cache:rebuild-all 
                            {--force : Force rebuild without confirmation}
                            {--prices-only : Only rebuild price caches}
                            {--totals-only : Only rebuild total caches}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Rebuild all price and total caches from scratch. Scheduled to run every Sunday at 1am.';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $startTime = now();
        
        $this->info('═══════════════════════════════════════════════════════════');
        $this->info('  CACHE REBUILD - ' . $startTime->format('Y-m-d H:i:s'));
        $this->info('═══════════════════════════════════════════════════════════');
        $this->newLine();

        // Confirm unless forced
        if (!$this->option('force') && !$this->confirm('This will delete all cached price and total data. Continue?', true)) {
            $this->warn('Operation cancelled.');
            return 0;
        }

        try {
            // Step 1: Rebuild Price Caches (unless totals-only)
            if (!$this->option('totals-only')) {
                $this->rebuildPriceCaches();
            }

            // Step 2: Rebuild Total Caches (unless prices-only)
            if (!$this->option('prices-only')) {
                $this->rebuildTotalCaches();
            }

            $endTime = now();
            $duration = $startTime->diffForHumans($endTime, true);

            $this->newLine();
            $this->info('═══════════════════════════════════════════════════════════');
            $this->info('✓ CACHE REBUILD COMPLETE');
            $this->info('  Duration: ' . $duration);
            $this->info('  Finished: ' . $endTime->format('Y-m-d H:i:s'));
            $this->info('═══════════════════════════════════════════════════════════');

            Log::info('Cache rebuild completed successfully', [
                'started_at' => $startTime,
                'completed_at' => $endTime,
                'duration_seconds' => $startTime->diffInSeconds($endTime),
            ]);

            return 0;

        } catch (\Exception $e) {
            $this->error('✗ Cache rebuild failed: ' . $e->getMessage());
            
            Log::error('Cache rebuild failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return 1;
        }
    }

    /**
     * Rebuild all price caches.
     */
    private function rebuildPriceCaches()
    {
        $this->info('┌─ STEP 1: Rebuilding Price Caches');
        $this->newLine();

        // Delete all existing price resolutions
        $this->task('  Deleting all price resolutions', function () {
            $count = DB::table('price_resolutions')->count();
            DB::table('price_resolutions')->delete();
            return $count;
        });

        // Clear Laravel cache tags
        $this->task('  Clearing price cache tags', function () {
            \Cache::tags(['prices'])->flush();
            return true;
        });

        // Get active colourways (those with recent orders or shipments)
        $this->info('  Finding active colourways...');
        
        $activeColourways = Colourways::whereHas('customer_order_lines', function($q) {
            $q->whereHas('customer_orders', function($q2) {
                // Orders from last 2 years
                $q2->where('created_at', '>', now()->subYears(2));
            });
        })
        ->orWhereHas('customer_order_lines.shipment_lines', function($q) {
            // Or with shipments from last year
            $q->where('created_at', '>', now()->subYear());
        })
        ->pluck('id');

        $this->info('  Found ' . $activeColourways->count() . ' active colourways');
        $this->newLine();

        // Warmup price resolutions for active colourways
        $priceService = app(PriceResolutionService::class);
        
        $this->withProgressBar($activeColourways, function ($colourwayId) use ($priceService) {
            try {
                $priceService->warmupColourway($colourwayId);
            } catch (\Exception $e) {
                Log::warning('Failed to warmup colourway during rebuild', [
                    'colourway_id' => $colourwayId,
                    'error' => $e->getMessage()
                ]);
            }
        });

        $this->newLine(2);
        $this->info('└─ ✓ Price caches rebuilt');
        $this->newLine();
    }

    /**
     * Rebuild all total caches.
     */
    private function rebuildTotalCaches()
    {
        $this->info('┌─ STEP 2: Rebuilding Total Caches');
        $this->newLine();

        // Delete all existing total cache entries
        $this->task('  Deleting all total cache entries', function () {
            $count = DB::table('total_cache')->count();
            DB::table('total_cache')->delete();
            return $count;
        });

        $totalService = app(TotalCacheService::class);

        // Warmup customer orders (recent ones)
        $this->info('  Finding recent customer orders...');
        $recentOrders = CustomerOrders::where('created_at', '>', now()->subYear())
            ->whereHas('customer_order_lines')
            ->pluck('id');
        
        $this->info('  Found ' . $recentOrders->count() . ' orders');
        
        if ($recentOrders->count() > 0) {
            $this->task('  Warming up customer order totals', function () use ($totalService, $recentOrders) {
                $totalService->invalidateAndDispatchWarmupBatch('customer_order', $recentOrders->toArray());
                return true;
            });
        }

        // Warmup customer order lines (recent ones)
        $this->info('  Finding recent customer order lines...');
        $recentOrderLines = CustomerOrderLines::whereHas('customer_orders', function($q) {
            $q->where('created_at', '>', now()->subYear());
        })->pluck('id');
        
        $this->info('  Found ' . $recentOrderLines->count() . ' order lines');
        
        if ($recentOrderLines->count() > 0) {
            $this->task('  Warming up order line totals', function () use ($totalService, $recentOrderLines) {
                $totalService->invalidateAndDispatchWarmupBatch('customer_order_line', $recentOrderLines->toArray());
                return true;
            });
        }

        // Warmup shipment lines (recent ones)
        $this->info('  Finding recent shipment lines...');
        $recentShipments = ShipmentLine::where('created_at', '>', now()->subYear())
            ->pluck('id');
        
        $this->info('  Found ' . $recentShipments->count() . ' shipment lines');
        
        if ($recentShipments->count() > 0) {
            $this->task('  Warming up shipment line totals', function () use ($totalService, $recentShipments) {
                $totalService->invalidateAndDispatchWarmupBatch('shipment_line', $recentShipments->toArray());
                return true;
            });
        }

        $this->newLine();
        $this->info('└─ ✓ Total caches rebuilt');
        $this->newLine();
    }
}



















































