<?php

namespace App\Console\Commands;

use App\Models\Colourways;
use Illuminate\Console\Command;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;

class ProcessColourwayImages extends Command
{
    protected $signature = 'images:process-colourways';
    protected $description = 'Process Colourway images using rembg and generate transparent 200x200 thumbnails';

    public function handle()
    {
        $colourways = Colourways::whereNotNull('image')->get();

        foreach ($colourways as $cw) {
            $this->info("Processing Colourway ID: {$cw->id}");

            try {
                $fullPath = storage_path("app/public/{$cw->image}");
                if (!file_exists($fullPath)) {
                    $this->warn("Image not found: {$fullPath}");
                    continue;
                }

                // Temp files
                $tempInput = tempnam(sys_get_temp_dir(), 'input_') . '.jpg';
                $tempOutput = tempnam(sys_get_temp_dir(), 'output_') . '.png';
                copy($fullPath, $tempInput);

                // Run rembg
                $cmd =  "OMP_NUM_THREADS=4 OPENBLAS_NUM_THREADS=4 MKL_NUM_THREADS=4 NUMEXPR_NUM_THREADS=4 " .
                        "NUMBA_CACHE_DIR=/tmp HOME=/tmp rembg i -m u2net --alpha-matting --alpha-matting-foreground-threshold 240 --alpha-matting-background-threshold 10 --alpha-matting-erode-size 10 '$tempInput' '$tempOutput' 2>&1";
                exec($cmd, $output, $result);

                if ($result !== 0 || !file_exists($tempOutput) || filesize($tempOutput) < 1000) {
                    $this->error("rembg failed for Colourway ID {$cw->id}");
                    @unlink($tempInput);
                    @unlink($tempOutput);
                    continue;
                }

                // Resize the output to 200x200
                $thumb = Image::make($tempOutput)
                    ->resize(200, 200, function ($constraint) {
                        $constraint->aspectRatio();
                        $constraint->upsize();
                    })
                    ->encode('png');

                // Save to disk
                $hash = md5_file($fullPath);
                $paththumb = "images/{$hash}-thumb.png";
                Storage::disk('public')->put($paththumb, $thumb);

                // Update model
                $cw->update(['img_thumb' => $paththumb]);

                // Clean up temp files
                @unlink($tempInput);
                @unlink($tempOutput);

                $this->info("Updated thumbnail for Colourway ID {$cw->id}");
            } catch (\Throwable $e) {
                $this->error("Error on Colourway ID {$cw->id}: " . $e->getMessage());
            }
        }

        $this->info('All done.');
    }
}
