<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\PriceResolutionService;
use App\Models\Colourways;

class PopulatePriceResolutions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'price-resolutions:populate 
                            {--colourway= : Specific colourway ID to populate}
                            {--all : Populate all colourways}
                            {--limit=100 : Limit number of colourways to process}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Populate price resolutions cache for colourways';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $service = app(PriceResolutionService::class);
        
        if ($this->option('colourway')) {
            // Populate specific colourway
            $colourwayId = $this->option('colourway');
            $this->info("Populating price resolutions for colourway {$colourwayId}...");
            
            $created = $service->warmupColourway($colourwayId);
            $this->info("Created {$created} price resolutions for colourway {$colourwayId}");
            
        } elseif ($this->option('all')) {
            // Populate all colourways
            $limit = $this->option('limit');
            $this->info("Populating price resolutions for all colourways (limit: {$limit})...");
            
            $colourways = Colourways::limit($limit)->get();
            $totalCreated = 0;
            $bar = $this->output->createProgressBar($colourways->count());
            
            foreach ($colourways as $colourway) {
                $created = $service->warmupColourway($colourway->id);
                $totalCreated += $created;
                $bar->advance();
            }
            
            $bar->finish();
            $this->newLine();
            $this->info("Created {$totalCreated} price resolutions for {$colourways->count()} colourways");
            
        } else {
            // Show help
            $this->info('Usage:');
            $this->line('  php artisan price-resolutions:populate --colourway=123');
            $this->line('  php artisan price-resolutions:populate --all');
            $this->line('  php artisan price-resolutions:populate --all --limit=50');
        }
        
        // Show cache stats
        $stats = $service->getCacheStats();
        $this->newLine();
        $this->info('Cache Statistics:');
        $this->line("  Total resolutions: {$stats['total']}");
        $this->line("  Fresh resolutions: {$stats['fresh']}");
        $this->line("  Stale resolutions: {$stats['stale']}");
    }
}