<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class MigrateCommissionOrders extends Command
{
    protected $signature = 'commission:migrate 
                            {--dry-run : Show what would be migrated without making changes}
                            {--batch-size=50 : Number of orders to process per batch}
                            {--customer-id= : Migrate only orders for a specific customer}
                            {--limit= : Limit total orders to migrate}';

    protected $description = 'Migrate historical commission orders from old tables to new commission tables';

    private $dryRun = false;
    private $stats = [
        'orders' => 0,
        'lines' => 0,
        'quantities' => 0,
        'drops' => 0,
        'drop_sizes' => 0,
        'skipped' => 0,
        'errors' => 0,
    ];

    // Maps old IDs to new IDs
    private $orderIdMap = [];
    private $lineIdMap = [];

    public function handle()
    {
        $this->dryRun = $this->option('dry-run');
        $batchSize = (int) $this->option('batch-size');
        $customerId = $this->option('customer-id');
        $limit = $this->option('limit');

        if ($this->dryRun) {
            $this->warn('DRY RUN MODE - No changes will be made');
        }

        $this->info('Starting commission orders migration...');
        $this->newLine();

        // Get count of orders to migrate
        $query = DB::table('customer_orders')
            ->where('order_type', 'commission')
            ->whereNull('deleted_at');

        if ($customerId) {
            $query->where('customers_id', $customerId);
        }

        $totalOrders = $query->count();
        $this->info("Found {$totalOrders} commission orders to migrate");

        if ($limit) {
            $totalOrders = min($totalOrders, (int) $limit);
            $this->info("Limiting to {$totalOrders} orders");
        }

        if ($totalOrders === 0) {
            $this->info('No orders to migrate.');
            return 0;
        }

        // Check for already migrated orders
        $alreadyMigrated = DB::table('commission_orders')
            ->whereNotNull('legacy_order_id')
            ->count();
        
        if ($alreadyMigrated > 0) {
            $this->warn("{$alreadyMigrated} orders already migrated (have legacy_order_id set)");
        }

        $bar = $this->output->createProgressBar($totalOrders);
        $bar->start();

        // Process in batches
        $offset = 0;
        while ($offset < $totalOrders) {
            $batchQuery = DB::table('customer_orders')
                ->where('order_type', 'commission')
                ->whereNull('deleted_at');

            if ($customerId) {
                $batchQuery->where('customers_id', $customerId);
            }

            $orders = $batchQuery
                ->orderBy('id')
                ->skip($offset)
                ->take($batchSize)
                ->get();

            if ($orders->isEmpty()) {
                break;
            }

            foreach ($orders as $oldOrder) {
                try {
                    $this->migrateOrder($oldOrder);
                    $bar->advance();
                } catch (\Exception $e) {
                    $this->stats['errors']++;
                    Log::error("Error migrating order {$oldOrder->id}: " . $e->getMessage());
                    $this->newLine();
                    $this->error("Error migrating order {$oldOrder->id}: " . $e->getMessage());
                }

                if ($limit && ($this->stats['orders'] + $this->stats['skipped']) >= (int) $limit) {
                    break 2;
                }
            }

            $offset += $batchSize;
        }

        $bar->finish();
        $this->newLine(2);

        // Show summary
        $this->info('Migration Summary:');
        $this->table(
            ['Metric', 'Count'],
            [
                ['Orders Migrated', $this->stats['orders']],
                ['Lines Created', $this->stats['lines']],
                ['Quantities Created', $this->stats['quantities']],
                ['Drops Created', $this->stats['drops']],
                ['Drop Sizes Created', $this->stats['drop_sizes']],
                ['Skipped (already migrated)', $this->stats['skipped']],
                ['Errors', $this->stats['errors']],
            ]
        );

        if ($this->dryRun) {
            $this->warn('This was a DRY RUN - no changes were made');
        }

        return 0;
    }

    private function migrateOrder($oldOrder)
    {
        // Check if already migrated (by checking if legacy_order_id exists)
        $existing = DB::table('commission_orders')
            ->where('legacy_order_id', $oldOrder->id)
            ->first();

        if ($existing) {
            $this->stats['skipped']++;
            return;
        }

        if ($this->dryRun) {
            $this->stats['orders']++;
            $this->migrateLinesForOrder($oldOrder, null, true);
            return;
        }

        DB::beginTransaction();
        try {
            // Create new commission order
            $newOrderId = DB::table('commission_orders')->insertGetId([
                'legacy_order_id' => $oldOrder->id,
                'customers_id' => $oldOrder->customers_id,
                'seasons_id' => $oldOrder->seasons_id,
                'departments_id' => $oldOrder->departments_id,
                'customer_po' => $oldOrder->customer_po,
                'order_date' => $oldOrder->order_date,
                'cancelled' => $oldOrder->cancelled ?? false,
                'status' => 'confirmed', // Historical orders are confirmed
                'confirmed_at' => $oldOrder->created_at,
                'created_at' => $oldOrder->created_at,
                'updated_at' => $oldOrder->updated_at,
            ]);

            $this->orderIdMap[$oldOrder->id] = $newOrderId;
            $this->stats['orders']++;

            // Migrate lines for this order
            $this->migrateLinesForOrder($oldOrder, $newOrderId, false);

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    private function migrateLinesForOrder($oldOrder, $newOrderId, $dryRun)
    {
        $oldLines = DB::table('customer_order_lines')
            ->where('customer_orders_id', $oldOrder->id)
            ->whereNull('deleted_at')
            ->get();

        foreach ($oldLines as $oldLine) {
            // Get commission/discount from first quantity record
            $firstQty = DB::table('customer_order_line_quantities')
                ->where('customer_order_lines_id', $oldLine->id)
                ->whereNull('deleted_at')
                ->first();

            $commissionPercent = $firstQty?->commission ?? null;
            $discountPercent = $firstQty?->discount ?? null;

            if ($dryRun) {
                $this->stats['lines']++;
                $this->migrateQuantitiesForLine($oldLine, null, true);
                $this->migrateDropsForLine($oldLine, null, true);
                continue;
            }

            // Create new commission order line
            $newLineId = DB::table('commission_order_lines')->insertGetId([
                'commission_orders_id' => $newOrderId,
                'colourways_id' => $oldLine->colourways_id,
                'cancelled' => $oldLine->cancelled ?? false,
                'match_status' => 'confirmed',
                'commission_percent' => $commissionPercent,
                'discount_percent' => $discountPercent,
                'created_at' => $oldLine->created_at,
                'updated_at' => $oldLine->updated_at,
            ]);

            $this->lineIdMap[$oldLine->id] = $newLineId;
            $this->stats['lines']++;

            // Migrate quantities and drops
            $this->migrateQuantitiesForLine($oldLine, $newLineId, false);
            $this->migrateDropsForLine($oldLine, $newLineId, false);
        }
    }

    private function migrateQuantitiesForLine($oldLine, $newLineId, $dryRun)
    {
        // Group quantities by size to handle duplicates (sum qty, use max price)
        $quantities = DB::table('customer_order_line_quantities')
            ->select(
                'sizes_id',
                DB::raw('SUM(qty) as total_qty'),
                DB::raw('MAX(price) as price'),
                DB::raw('MIN(created_at) as created_at'),
                DB::raw('MAX(updated_at) as updated_at')
            )
            ->where('customer_order_lines_id', $oldLine->id)
            ->whereNull('deleted_at')
            ->whereNotNull('sizes_id')
            ->groupBy('sizes_id')
            ->get();

        foreach ($quantities as $qty) {
            if ($dryRun) {
                $this->stats['quantities']++;
                continue;
            }

            DB::table('commission_order_line_quantities')->insert([
                'commission_order_lines_id' => $newLineId,
                'sizes_id' => $qty->sizes_id,
                'qty' => $qty->total_qty ?? 0,
                'price' => $qty->price ?? null,
                'created_at' => $qty->created_at,
                'updated_at' => $qty->updated_at,
            ]);

            $this->stats['quantities']++;
        }
    }

    private function migrateDropsForLine($oldLine, $newLineId, $dryRun)
    {
        // Get shipment lines for this order line
        $shipmentLines = DB::table('shipment_lines')
            ->where('customer_order_lines_id', $oldLine->id)
            ->whereNull('deleted_at')
            ->get();

        foreach ($shipmentLines as $sl) {
            // Determine if shipped based on shipped_qty or delivery_date
            $isShipped = !empty($sl->shipped_qty) || !empty($sl->delivery_date);
            
            if ($dryRun) {
                $this->stats['drops']++;
                $this->migrateDropSizesForShipmentLine($sl, null, true);
                continue;
            }

            $newDropId = DB::table('commission_drops')->insertGetId([
                'commission_order_lines_id' => $newLineId,
                'commission_invoice_groups_id' => null, // Will be linked later if needed
                'exfty' => $sl->exfty,
                'shipped_date' => $sl->delivery_date,
                'shipped' => $isShipped,
                'invoiced' => $sl->invoiced ?? false,
                'notes' => $sl->notes,
                'fn_notes' => $sl->fn_notes,
                'factory_invoice' => $sl->factory_invoice,
                'rt_invoice' => $sl->rt_invoice,
                'created_at' => $sl->created_at,
                'updated_at' => $sl->updated_at,
            ]);

            $this->stats['drops']++;
            $this->migrateDropSizesForShipmentLine($sl, $newDropId, false);
        }
    }

    private function migrateDropSizesForShipmentLine($shipmentLine, $newDropId, $dryRun)
    {
        // Group by size to handle any duplicates (sum quantities)
        $sizes = DB::table('shipment_line_sizes')
            ->select(
                'sizes_id',
                DB::raw('SUM(qty) as total_qty'),
                DB::raw('SUM(shipped_qty) as total_shipped_qty'),
                DB::raw('MIN(created_at) as created_at'),
                DB::raw('MAX(updated_at) as updated_at')
            )
            ->where('shipment_line_id', $shipmentLine->id)
            ->whereNull('deleted_at')
            ->whereNotNull('sizes_id')
            ->groupBy('sizes_id')
            ->get();

        foreach ($sizes as $size) {
            if ($dryRun) {
                $this->stats['drop_sizes']++;
                continue;
            }

            DB::table('commission_drop_sizes')->insert([
                'commission_drops_id' => $newDropId,
                'sizes_id' => $size->sizes_id,
                'qty' => $size->total_qty ?? 0,
                'shipped_qty' => $size->total_shipped_qty,
                'created_at' => $size->created_at,
                'updated_at' => $size->updated_at,
            ]);

            $this->stats['drop_sizes']++;
        }
    }
}

